*************************************************************
**This code create the dataset and remove respondents whose** 
***survey is incomplete or who failed the attention checks***
*************************************************************
clear all
/* Set root directory */
*global dir " " //set origin path
global survey_data "$dir/replication_package/data"


******************************************************************
******************************************************************
**#**********  IMPORT DATASET AND FIRST CLEANING  ****************
******************************************************************
******************************************************************
import excel "$survey_data/BPEA_survey_A.xlsx",  firstrow clear

*dropping second line of labels and trial answers
drop in 1/2

*dropping incompletes (all respondents who fail the attention checks or do not complete the survey)
keep if screen=="1" 

**********************************
****** Time spent on survey ******
**********************************

destring Durationinseconds, replace force
gen Durationinminutes=Durationinseconds/60 
label var Durationinminutes "Minutes spent on survey"

************************************
********** Recoding timers *********
************************************ 
foreach var of varlist T* {
	destring `var', replace force
	}


*creating date variables
gen date_start = clock(StartDate, "MDY hm")
format date_start %tc
gen dailydate = dofc(date_start)
format dailydate %td



******************************************************************
******************************************************************
**#***************  INATTENTIVES CLEANING  ***********************
******************************************************************
******************************************************************
**This part of the code removes inattentive respondents**


*generate binary variables to identify the collection wave
gen collection_1 = (dailydate < date("20dec2023", "DMY")) 
gen collection_2 = (dailydate < date("7jan2024", "DMY") & dailydate > date("19dec2023", "DMY"))
gen collection_3 = ( dailydate > date("6jan2024", "DMY"))


*******************************************************
************Find and remove respondents****************
*****inattentive to open-ended questions (bots)********
*******************************************************

***NOTE: the open-ended questions are: ***
* infl_unemp_why_pos infl_unemp_why_neg  react_which_ess_np feedback

***moreover these are other questions with text entry: ***
* inf_info_pos_*  inf_info_neg_*  inf_info_same_*
* inf_exp_pos_*  inf_exp_neg_* inf_exp_same_*


gen inatt_open=0

*STEP 1: to keep date and open ended questions to check answers (THESE MUST BE COMMENTED OUT IN LAST STEP) 

*keep if dailydate == date("19oct2023", "DMY")
*keep infl_unemp_why_pos infl_unemp_why_neg react_which_ess_np feedback


*STEP 2: to identify ID codes of inattentives  
/*

*1. pilot 3 1/12/2023
foreach j in  78 94 113 116 133 193 203 328 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"


*2. pilot 3 4/12/2023
foreach j in 394 404 411 412 420 458 459 467 481 498 528 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"


3* collection 1 6/01/2024
foreach j in 503 752 512 516 517 570 598 624 657 691 699 700 753 764 527 540 727 735 749 477 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"

*4 collection 1 7/01/2024
foreach j in 894 922 939 885 1000 1038 1066 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"

*5 collection 1 10/01/2024
foreach j in 1132 1191 1218 1130 1235 1254 1368 1323  1309 1269 1148 1119 1094 1020 1004 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"

*5 collection 1 10/01/2024
foreach j in 1269 1148 1119 1094 1020 1004 1485 1478 1464 {
local rid_`j' = ResponseId[`j']
local list "`list' `rid_`j''"
}
di "`list'"
*/

*STEP 3: then take list from previous command and copy it below in the place of the one below, to create global_list_remove

global list_remove_1 ///
" R_27kuykUTEwGJ2NW R_8xdqPPur8fS6WUV R_cHLNc1TOUHGXYSl R_3qPgTjOuVVYUQY5 R_erZhD0bJZ0QAZb3 R_2zioTHAJZSSqndF R_OIK7AIyHASA OCit R_2Qfb82lwckS9H1s "

global list_remove_2 ///
" R_1OisbMdXwjsVPj8 R_2ashao0jdBVDKkH R_d6fxULGyonRPwtz R_ZIVYiITyRDzEtUJ R_2Y8QKCqxgF6j5X0 R_us43NluZQBVg7Hb R_1nVqcfVBrIlofsO R_1RETmRFJOqdIzEl R_3huN7HyxOxrdpZQ R_21gJJ9pg8yYmsVB R_xrRIe3jQ9GSdaP7"

global list_remove_3 " R_3KvE1hTGwflCCtA R_78QpMzMQXA8qTbb R_1FFXAFlEQbfzzmK R_8mWodumbjlZ07aH R_3oQSEuICZaNvYSU R_1ffb3AV1cvEMglJ R_6qOv2u45E0HAyXw R_7zwlSdf1ptetBNS R_39gNrOtVBTDeaYv R_1ASVW5TR4yfjXBR R_739uzY3ySrusy1b R_5tM8tOa85dmIi3S R_1545Qn384898DDz R_7IZyRJ8Kz06LZA0 R_5MkUAEqgJXyS94l R_8m7TGwfmIWEa9GZ R_74ITyFkQX9qqy7d R_5pA8IHF68lfHwIb R_3JvZQDAaenA02rf R_2PoCLkozp0GoyDK "

global list_remove_4 ///
" R_1rB679mSXLQrZx1 R_3eeqUXmK31dVarD R_3rqpzJO6D15ij5E R_6ktVIfuV9iAgyA1 R_7cZClaQGHAprk04 R_7beMyTeZehDQhDr R_3iEEtSc1jMeCQSd "

global list_remove_5 ///
"R_5TWRMTS6FAwIo0N R_1jkasCaAvU2JPOx R_52ttU0N4Y9dXBn4 R_5ghCVUvX97SjlSr R_7jE2k3TxovirB4d R_5riAwBpyDe9iI6w R_5eWqqZkQZ48cZxn R_1TDZepBy9WW7uQ9 R_1QA7EGq1Y0bV645 "

 global list_remove_6 ///
 " R_10wwhjimSykh8o9 R_5 gtjDVBratwTbvf R_6BgSr7FIDNNEnM1 R_7cq5ILk0jqiKrPk R_5Ck6yPFekE9eGGT R_1Cs vff93D2uf7q8 R_6qfEdaL4G01oWbj R_3JxUzC5BE01qVbP R_3CJJ57kIhIlGLEE "


*STEP 4: assign dummy equal to 1 if inattentive to the above selected respondents

foreach name in $list_remove_1 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

foreach name in $list_remove_2 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

foreach name in $list_remove_3 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

foreach name in $list_remove_4 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

foreach name in $list_remove_5 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

foreach name in $list_remove_6 {
	replace inatt_open = 1 if ResponseId == "`name'"
}

*******************************************************************
************************* Inaccuracy test *************************
*******************************************************************

*test to check whether the respondents report a different gender in the pre-screening questions asked by Lucid and our survey question

drop if age_lucid=="[%AGE%]"
destring gender_lucid gender, replace 
gen test_gender = gender - gender_lucid

gen inacc_gender = 0 //INACC. divergence gender btw Lucid and Qualtrics
label var inacc_gender "Divergenge gender btw Lucid and Qualtrics"
replace inacc_gender = 1 if abs(test_gender) == 1


***********************************************************
**************Drop inattentive respondents*****************
***********************************************************
drop if inatt_open == 1
drop if (inacc_gender == 1 & gender != 3)

***drop respondents rushing through the survey_data
sum Durationinminutes, detail 
drop if Durationinminutes<`r(p5)'




******************************************************************
******************************************************************
**#*****************  ENCODING VARIABLES  ************************
******************************************************************
******************************************************************
**This part of the code encodes and labels all the variables from the survey**


****COMMON LABELS
label define yes_no 1 "Yes" 2 "No" 

*****************Background questions - QUOTAS ********************
**gender
label define gender 1 "Male" 2 "Female" 3 "Other"
gen gender_en = gender
label var 	gender_en "Gender"
label values gender_en gender

**age
gen age = age_1
destring age, replace

**age_group: Qualtrics and respondents
rename age_group age_group_q
label var age_group_q "Age group qualtrics"

gen 			agegroup="." 
replace 		agegroup="18-29" if(age>17 & age<30)
replace 		agegroup="30-39" if(age>29 & age<40)
replace 		agegroup="40-49" if(age>39 & age<50)
replace 		agegroup="50-59" if(age>49 & age<60)
replace 		agegroup="60-69" if(age>59 & age<70)
label define 	agegroup 1 "18-29" 2 "30-39" 3 "40-49" 4 "50-59" 5 "60-69"
encode			agegroup, gen(age_group) label(agegroup)
drop 			agegroup

**live in the US
destring live_in_us, replace
label values live_in_us yes_no
label var live_in_us "Live in the US"

**area of living
label define area_us 1 "Northeast" 2 "South" 3 "Midwest" 4 "West"
destring us_area, replace
label values us_area area_us
label var us_area "Living area US"

**race
label define race 1 "White" 2 "African American/Black" 3 "Hispanic/Latino" 4 "Asian/Asian American" 5 "Mixed race" 6 "Other"
destring race, replace
label values race race
label var race "Race"

**income
rename income_group income_group_q
label var income_group_q "Qualtrics income group"

gen incomegroup="."
replace incomegroup="$0-$19999" if(income_bracket=="1"| income_bracket== "2"|income_bracket=="3")
replace incomegroup="$20000-$39999"	if(income_bracket=="4"  | income_bracket== "5")
replace incomegroup="$40000-$69999" 	if(income_bracket=="6"  | income_bracket== "7")
replace incomegroup="$70000-$99999" 	if(income_bracket=="8"  | income_bracket== "9")
replace incomegroup = "$100000-$124999" if (income_bracket=="10"  | income_bracket== "11")
replace incomegroup="$125000+" 		if(income_bracket=="12"| income_bracket== "13"|income_bracket=="14")

label define incomegroup 1 "$0-$19999" 2 "$20000-$39999" 3 "$40000-$69999" 4 "$70000-$99999" 5 "$100000-$124999" 6 "$125000+"
encode incomegroup, gen(income_group) label(incomegroup)
drop incomegroup
label var income_group "Income group"

*income - less coarse partition 
label define income_bracket 1 "$0-$9999" 2 "$10000-$14999" 3 "$15000-$19999" 4 "$20000-$29999" 5 "30000-39999" 6 "$40000-$49999" 7 "$50000-$69999" 8 "$70000-$89999" 9 "$90000-$99999" 10 "$100000-$109999" 11 "$110000-$124999" 12 "$125000-$149999" 13 "$150000-$199999" 14 "$200000+"
destring income_bracket, replace
label values income_bracket income_bracket

******************************************************
******************Demographics************************
******************************************************
*born_us
destring born_us, replace
label values born_us yes_no
label var born_us "Born in the US"


*zip_code
*already encoded in 1.0creating_dataset.do


* Number of children
label define nb_child 1 "I do not have children" 2 "1" 3 "2" 4 "3" 5 "4" 6 "5 or more"
destring children, replace
label values children nb_child
label var children "N. of children"

* Highest level of education
label define education_level 1 "Primary edu or less" 2 "Some High School" 3 "High School degree/ GED" 4 "Some College" 5 "2 year College Degree" 6 "4 year College Degree" 7 "Masters Degree" 8 "Doctoral Degree" 9 "Professional Degree (JD, MD, MBA)"
destring educ, gen(education_level)
label values education_level education_level
label var education_level "Education level"


* Field of study
label define field_of_study 1 "Accounting/bookkeeping" 2 "Administrative science/public administration" 3 "Advertising" 4 "Agriculture/horticulture" 5 "Allied health" 6 "Anthropology"  7 "Architecture" 8 "Art" 9 "Aviation/aeronatics" 10 "Biology" 11 "Business administration" 12 "Chemistry" 13 "Child/human/family development" 14 "Comm. disorders" 15 "Communications/speech" 16 "Computer science" 17 "Counseling" 18 "Criminology/criminal justice" 19 "Dance" 20 "Dentistry" 21 "Economics" 22 "Education" 23 "Educational administration" 24 "Electronics" 25 "Engineering" 26 "English" 27 "Environmental science/ecology" 28 "Ethnic studies" 29 "Fashion" 30 "Finance" 31 "Fine arts" 32 "Food science/nutrition/culinary arts" 33 "Foreign language" 34 "Forestry" 35 "General sciences" 36 "General studies" 37 "Geography" 38 "Geology" 39 "Gerontology" 40 "Health" 41 "History" 42 "Home economics" 43 "Human services/human resources" 44 "Humanities" 45 "Industrial relations" 46 "Industry and technology" 47 "Information technology" 48 "Journalism" 49 "Law" 50 "Law enforcement" 51 "Liberal arts" 52 "Library science" 53 "Marketing" 54 "Mathematics" 55 "Mechanics/machine trade" 56 "Medicine" 57 "Music" 58 "Nursing" 59 "Other vocational" 60 "Parks and recreation" 61 "Pharmacy" 62 "Philosophy" 63 "Physical education" 64 "Physics" 65 "Political science/international relations" 66 "Psychology" 67 "Public relations" 68 "Social sciences" 69 "Social work: Sociology" 70 "Special education" 71 "Statistics/biostatistics" 72 "Television/film" 73 "Textiles/cloth" 74 "Theater arts" 75 "Theology" 76 "Urban and regional planning" 77 "Veterinary medicine" 78 "Visual arts/graphic design/ design and drafting" 79 "Other"
destring field_of_study, replace
label val field_of_study field_of_study
label var field_of_study "Field of study"

* Current employment status
label define emp_status 1 "Full-time employee" 2 "Part-time employee" 3 "Self-employed or business owner" 4 "Unemployed and looking for work" 5 "Student" 6 "Not currently working and not looking for a job" 7 "Retiree"
destring emp_status, replace
label values emp_status emp_status
label var emp_status "Employment status"
	
* Main occupation (if currently employed)
destring emp_occupation, replace
label define emp_occupation 1 "Management, business and financial occupations" 2 "Professional and related occupations" 3 "Service occupations" 4 "Sales and related occupations" 5 "Office and administrative support occupations" 6 "Farming, fishing and forestry occupations" 7 "Construction and extraction occupations" 8 "Installation, maintenance and repair occupations" 9 "Production occupations" 10 "Transportation and material moving occupations" 11 "Armed forces" 12 "Other"
label values emp_occupation emp_occupation
label var emp_occupation "Main occupation"

* Main occupation (if currently unemployed)
destring unemp_occupation, replace
label define unemp_occupation 1 "Management, business and financial occupations" 2 "Professional and related occupations" 3 "Service occupations" 4 "Sales and related occupations" 5 "Office and administrative support occupations" 6 "Farming, fishing and forestry occupations" 7 "Construction and extraction occupations" 8 "Installation, maintenance and repair occupations" 9 "Production occupations" 10 "Transportation and material moving occupations" 11 "Armed forces" 12 "Other"
label values unemp_occupation unemp_occupation
label var unemp_occupation "Latest main occupation"

* Employment sector (if currently employed)
destring emp_sector, replace	
label define emp_sector 1 "Agriculture, plantations, other rural sectors" 2 "Basic metal production" 3 "Chemical industries" 4 "Commerce" 5 "Construction" 6 "Education" 7 "Financial services, professional services" 8 "Food, drink, tobacco" 9 "Forestry, wood" 10 "Health services" 11 "Hotels, tourism, catering" 12 "Mining" 13 "Mechanical and electrical engineering" 14 "Media, culture, graphical" 15 "Oil and gas production, oil refining" 16 "Postal and telecommunications services" 17 "Public service" 18 "Shipping, ports, fisheries, inland waterways" 19 "Textiles, clothing, leather, footwear" 20 "Transport (including civil aviation, railways, road transport)" 21 "Transport equipment manufacturing" 22 "Utilities (water, gas, electricity)" 23 "Other"
label values emp_sector emp_sector
label var emp_sector "Employment sector"

* Employment sector (if currently unemployed)
destring unemp_sector, replace	
label define unemp_sector 1 "Agriculture, plantations, other rural sectors" 2 "Basic metal production" 3 "Chemical industries" 4 "Commerce" 5 "Construction" 6 "Education" 7 "Financial services, professional services" 8 "Food, drink, tobacco" 9 "Forestry, wood" 10 "Health services" 11 "Hotels, tourism, catering" 12 "Mining" 13 "Mechanical and electrical engineering" 14 "Media, culture, graphical" 15 "Oil and gas production, oil refining" 16 "Postal and telecommunications services" 17 "Public service" 18 "Shipping, ports, fisheries, inland waterways" 19 "Textiles, clothing, leather, footwear" 20 "Transport (including civil aviation, railways, road transport)" 21 "Transport equipment manufacturing" 22 "Utilities (water, gas, electricity)" 23 "Other"
label values unemp_sector unemp_sector
label var unemp_sector "Latest employment sector"

* Gig economy
destring gig, replace
label values gig yes_no
label var gig "Gig economy"

* Marital status
label define marital_status 1 "Single" 2 "Married" 3 "Legally separated or divorced" 4 "Widowed"
destring marital_status, replace
label values marital_status marital_status
label var marital_status "Marital status"

* Spouse's employment status
destring spouse_emp_status, replace
label values spouse_emp_status emp_status
label var spouse_emp_status "Spouse's employment status"

* Economic assistance 2022
destring cash_assist, replace
label values cash_assist yes_no
label var cash_assist "Cash assistance"

* Medical assistance
destring medicaid, replace
label values medicaid yes_no
label var medicaid "Medical assistance"

* Food assistance
destring food_stamps, replace
label values food_stamps yes_no
label var food_stamps "Food assistance"

* Income uncertainty
label define certainty_scale 0 "0 (Extremely uncertain)" 1 "1" 2 "2" 3 "3" 4 "4" 5 "5" 6 "6" 7 "7" 8 "8" 9 "9" 10 "10 (Extremely certain)"
destring income_uncertain, gen(income_uncertain_en)
label values income_uncertain_en certainty_scale


* On economic policy matters: liberal/conservative spectrum
label define lib_scale 1 "Very liberal" 2 "Liberal" 3 "Moderate" 4 "Conservative" 5 "Very conservative"
destring lib_scale, replace 
label values lib_scale lib_scale
label var lib_scale "Economic ideology"

* Political affiliation
label define pol_aff 1 "Republican" 2 "Democrat" 3 "Independent" 4 "Other" 5 "Non-affiliated"
destring pol_aff, replace
label values pol_aff pol_aff
label var pol_aff "Political affiliation"

* Vote in 2020 presidential election?
destring vote_2020, replace
label values vote_2020 yes_no
label var vote_2020 "Vote 2020"

* 2020 Preferred candidate (if they voted)
	label define vote_who 1 "Joe Biden" 2 "Donald Trump" 3 "Howie Hawkins" 4 "Jo Jorgensen" 5 "Other"
	destring vote_who, replace
	label values vote_who vote_who
	label var vote_who "Preferred candidate (voting)"

* 2020 Preferred candidate (if they did not vote)
destring novote_who, replace
label values novote_who vote_who
label var novote_who "Preferred candidate (not voting)"



****from this point all the variables are encoded in a new variable with same name and indexed with _en
**********************************************************
*******************Definition of inflation****************
**********************************************************

*inf_understanding_1 inf_understanding_2
foreach var in inf_understanding_1 inf_understanding_2 {
gen `var'_N = regexs(1) if regexm(`var', "([0-9.]+)")
destring `var'_N, gen(`var'_en) 
drop `var'_N
}

*inf_yardstick
label define agreement 1 "Strongly agree" 2 "Somewhat agree" 3 "Neither agree nor disagree" 4 "Somewhat disagree" 5 "Strongly agree"
destring inf_yardstick, gen(inf_yardstick_en)
label val inf_yardstick_en agreement

*****************************************************************
********* Information about past and expected Inflation *********
*****************************************************************

*inf_info_utd _i
label define importance 1 "Extremely important" 2 "Very important" 3 "Somewhat important" 4 "Slightly important" 5 "Not important at all"

foreach var in inf_info_utd_i  {
destring `var', gen(`var'_en) 
label values `var'_en importance	
}

*inf_info_att_i
label define increased 1 "Increased a lot" 2 "Somewhat increased" 3 "Remained the same" 4 "Somewhat decreased" 5 "Decreased a lot"
foreach var in inf_info_att_i {
destring `var', gen(`var'_en) 
label values `var'_en increased	
}

*inf_info_source_i
foreach var in inf_info_source_i {
	foreach x in 1 2 3 4 5 6 7 8 9 10 11{ //note: values follow the order of the answer options in the survey
		destring `var'_`x', gen(`var'_`x'_en) force
		label val `var'_`x'_en yes_no	
		}
}
		
*inf_info_dir_i and inf_exp_dir_i
label define inflation 1 "Inflation" 2 "Deflation" 3 "Roughly no change in prices"
foreach var in inf_info_dir_i inf_exp_dir_i {
destring `var', gen(`var'_en) 
label values `var'_en inflation	
}

*GUESSES on past and expected inflation
foreach var in inf_info_pos_i inf_info_neg_i inf_info_same_i inf_exp_pos_i inf_exp_neg_i inf_exp_same_i {
gen `var'_N = regexs(1) if regexm(`var', "([0-9.]+)")
destring `var'_N, gen(`var'_en) 
drop `var'_N
}

*creating common variables for plots
gen inf_info_num_i = inf_info_pos_i_en
replace inf_info_num_i = - inf_info_neg_i_en if missing(inf_info_num_i)
replace inf_info_num_i = inf_info_same_i_en if missing(inf_info_num_i)

gen inf_exp_num_i = inf_exp_pos_i_en
replace inf_exp_num_i = - inf_exp_neg_i_en if missing(inf_exp_num_i)
replace inf_exp_num_i = inf_exp_same_i_en if missing(inf_exp_num_i)

*inf_categories_i 
label define categories 1 "Food" 2 "Gas" 3 "Rent" 4 "Utilities"
foreach var in inf_categories_i {
destring `var', gen(`var'_en) 
label values `var'_en categories	
}

*inf_exp_reason_i
label define inf_exp_reason 1 "News reports" 2 "Official statistics" 3 "Recent price changes of my purchases" 4 "Advice from friends and family"
foreach var in inf_exp_reason_i {
destring `var', gen(`var'_en) 
label values `var'_en inf_exp_reason	
}




*******************************************************************
************************PERSONAL IMPACTS***************************
*******************************************************************

**************************************
*** Personal impacts: DM questions ***
**************************************
*shopping
label define frequency_general 1 "Always" 2 "Most of the time" 3 "Sometimes" 4 "Rarely" 5 "Never"
destring shopping, gen(shopping_en) 
label values shopping_en frequency_general

***fin_decisions
label define fin_decis 1 "Someone else in my household makes all financial decisions" 2 "Someone else in my household makes most financial decisions" 3 "I share financial decisions equally with someone else in my household" 4 "I make most financial decisions myself" 5 "I make all financial decisions myself"
foreach var in fin_decisions {
destring `var', gen(`var'_en) 
label values `var'_en fin_decis
}

*****************************************
**** Personal impacts: as a consumer ****
*****************************************
*pers_imp_cons_ppower and pers_imp_cons_qual
label define decreased 5 "Increased a lot" 4 "Increased somewhat" 3 "Remained the same" 2 "Decreased somewhat" 1 "Decreased a lot"
foreach var in pers_imp_cons_ppower pers_imp_cons_qual {
destring `var', gen(`var'_en) 
label values `var'_en decreased
}

*infl_ppower_know
label define increases 1 "Increases by a lot" 2 "Somewhat increases" 3 "Remains the same" 4 "Somewhat decreases" 5 "Decreases by a lot"
foreach var in infl_ppower_know {
destring `var', gen(`var'_en) 
label values `var'_en increases
}

*pers_imp_cons_compar
label define easier 1 "Much easier" 2 "Somewhat easier" 3 "Neither harder nor easier" 4 "Somewhat harder" 5 "Much harder"
foreach var in pers_imp_cons_compar {
destring `var', gen(`var'_en) 
label values `var'_en easier
}

*pers_imp_cons_quant
label define shrinkflation 1 "Less widespread than before" 2 "As widespread as before" 3 "More widespread than before"
destring pers_imp_cons_quant, gen(pers_imp_cons_quant_en)
label values pers_imp_cons_quant_en shrinkflation

*****************************************
***** Personal impacts: as a worker *****
*****************************************
*job_change job_change_wage jobch_yes_wage_onjob
foreach var in job_change job_change_wage jobch_yes_wage_onjob {
destring `var', gen(`var'_en) 
label values `var'_en yes_no
}

*job_wage_why
label define wage_why 1 "Solely due to my performance and career progression" 2 "Solely to offset recent price increases" 3 "A combination of both factors"
destring job_wage_why, gen(job_wage_why_en)
label values job_wage_why_en wage_why

*wage_indexed
label define yes_no_notknow 1 "Yes" 2 "No" 3 "Do not know"
destring wage_indexed, gen(wage_indexed_en)
label values wage_indexed_en yes_no_notknow

*concern_future_work
label define concerned 1 "Not concerned" 2 "Slightly concerned" 3 "Moderately concerned" 4 "Very concerned" 5 "Extremely concerned"
destring concern_future_work, gen(concern_future_work_en)
label values concern_future_work_en concerned

*pers_imp_work_2_me
label define pers_imp_work_2_me 1 "Up to one month" 2 "Two to six months" 3 "Between seven months and one year" 4 "Two to three years" 5 "More than three years" 6 "I do not know"
destring pers_imp_work_2_me, gen(pers_imp_work_2_me_en)
label values pers_imp_work_2_me_en pers_imp_work_2_me

*pers_imp_work_hyp_sh
label define pers_imp_hyp_sh 1 "My income would be lower" 2 "My income would be the same" 3 "My income would be higher" 4 "Do not know"
destring pers_imp_work_hyp_sh, gen(pers_imp_work_hyp_sh_en)
label values pers_imp_work_hyp_sh_en pers_imp_hyp_sh

*pers_imp_work_satisf
label define pers_imp_work_satisf 1 "It increases" 2 "It remains unchanged" 3 "It decreases"
destring pers_imp_work_satisf, gen(pers_imp_work_satisf_en)
label values pers_imp_work_satisf_en pers_imp_work_satisf

*pers_imp_work_rel pers_imp_work_rel_hi
label define pers_imp_work_rel 1 "Much more quickly" 2 "Slightly more quickly" 3 "At the same pace" 4 "Slightly less quickly" 5 "Much less quickly"
foreach var in pers_imp_work_rel pers_imp_work_rel_hi {
destring `var', gen(`var'_en) 
label values `var'_en pers_imp_work_rel
}

*inf_infl_wage_growth
label define inf_infl_wage_growth 1 "Prices increase much faster" 2 "Prices increase somewhat faster" 3 "Both increase at the same rate" 4 "Wages increase somewhat faster" 5 "Wages increase much faster"
destring inf_infl_wage_growth, gen(inf_infl_wage_growth_en)
label values inf_infl_wage_growth_en inf_infl_wage_growth

*wage_theories
label define wage_theories 1 "Inflation will increase my employer's profits as they can sell their products or services for more, but this won't affect my salary. My employer won't feel the need to increase my pay." 2 "Due to inflation, companies compete more for workers, which could lead to my employer raising my salary to match better offers I might get from other companies." 3 "A sense of fairness and proper behavior will cause my employer to raise my pay."
destring inf_wage_theories, gen(inf_wage_theories_en)
label values inf_wage_theories_en wage_theories

*inf_adj_how_many_sf inf_adj_how_many_bf
label define how_many 1 "Almost none" 2 "A few" 3 "Many" 4 "Almost all"
foreach var in inf_adj_how_many_sf inf_adj_how_many_bf {
		destring `var', gen(`var'_en) force
		label val `var'_en how_many	
		}
		
*inf_adjust_why_sf inf_adjust_why_bf
label define adjust_why 1 "To ensure fairness" 2 "To preserve employees' purchasing power" 3 "To attract and retain talented workers" 4 "To make employees work harder" 5 "To maintain employee morale"	
foreach var in inf_adjust_why_sf inf_adjust_why_bf {
		destring `var', gen(`var'_en) force
		label val `var'_en adjust_why	
		}
		
*inf_notadjust_why_sf inf_notadjust_why_bf
label define not_adjust_why 1 "To deal with uncertainty about the future" 2 "To control their costs and increase their profits" 3 "To push workers to work harder" 4 "Because they know that their employees have very few other options"	
foreach var in inf_notadjust_why_sf inf_notadjust_why_bf {
		destring `var', gen(`var'_en) force
		label val `var'_en not_adjust_why	
		}		

*************************************************************
***** Personal impacts of Inflation: as an asset holder *****
*************************************************************
*pers_imp_ass_fin_ass
label define pers_imp_ass_fin_ass 1 "Increased" 2 "Stayed the same" 3 "Decreased"
destring pers_imp_ass_fin_ass, gen(pers_imp_ass_fin_ass_en)
label values pers_imp_ass_fin_ass_en pers_imp_ass_fin_ass

*pers_imp_ass_savings
label define pers_imp_ass_savings 1 "Increased" 2 "Stayed the same" 3 "Decreased" 4 "Do not have any savings"
destring pers_imp_ass_savings, gen(pers_imp_ass_savings_en)
label values pers_imp_ass_savings_en pers_imp_ass_savings

*pers_imp_ass_debt_va
label define pers_imp_ass_debt_va 1 "It has reduced it" 2 "It has had no effect on  it" 3 "It has increased it" 4 "Do not have any debt"
destring pers_imp_ass_debt_va, gen(pers_imp_ass_debt_va_en)
label values pers_imp_ass_debt_va_en pers_imp_ass_debt_va

*pers_imp_ass_debt_re
label define pers_imp_ass_debt_re 1 "Repayment easier than before" 2 "Repayment is unaffected by inflation" 3 "Repayment is harder than before" 4 "Do not have any loans/debts"
destring pers_imp_ass_debt_re, gen(pers_imp_ass_debt_re_en)
label values pers_imp_ass_debt_re_en pers_imp_ass_debt_re

**********************************************************************
** Personal impacts of Inflation: decision making and psychological **
**********************************************************************
*pers_imp_econoutlook
label define pers_imp_econoutlook 1 "Very negatively" 2 "Somewhat negatively" 3 "Not affected" 4 "Somewhat positively" 5 "Very positively"
destring pers_imp_econoutlook, gen(pers_imp_econoutlook_en)
label values pers_imp_econoutlook_en pers_imp_econoutlook

*pers_imp_psy_stress
label define pers_imp_psy_stress 1 "Less stressed" 2 "Equally stressed" 3 "More stressed"
destring pers_imp_psy_stress, gen(pers_imp_psy_stress_en)
label values pers_imp_psy_stress_en pers_imp_psy_stress

*pers_imp_psy_causes (this questions asked for 6 (7 if children) possible causes whether they were a cause of stress for the respondent. Hence it was made of 6 (7 if chldren) separate questions that are therefore reported separately --> name of single question/option: "pers_imp_psy_causes_#" (each of them has value 1 if the respondents said that was a cause of stress)
foreach var in pers_imp_psy_causes {
	foreach x in 1 2 3 4 5 6 7 { //note: values follow the order of the answer options in the survey
		destring `var'_`x', gen(`var'_`x'_en) force
		label val `var'_`x'_en yes_no	
		}
}

*pers_imp_psy_mostimp (this question showed as answer options the answers selected by the respondent in the previous question, pers_imp_psy_causes)
label define most_imp_cause 1 "Having difficulties paying bill/credit card balance" 2 "Unable to afford essentials" 3 "Worrying about paying my rent" 4 "Worrying about paying my mortgage" 5 "Worrying about investment losses" 6 "Having to cut down on holidays/going out" 7 "Worrying about my children edu tuition" 
destring pers_imp_psy_mostimp, gen (pers_imp_psy_mostimp_en)
label values pers_imp_psy_mostimp_en most_imp_cause

***********************************************************
******************PERSONAL REACTIONS***********************
***********************************************************

*****************************************
***** Actual reactions to Inflation ***** 
*****************************************

*NOTE: there were two branches with same questions with a different formulation in the survey: one for married people (questions indexed with _p) and one for not married people (questions indexed with _np)

*react_quant_p react_quant_np
label define reduction 1 "Reduced a lot" 2 "Reduced somewhat" 3 "Neither reduced nor increased" 4 "Increased somewhat" 5 "Increased a lot"
foreach var in react_quant_p react_quant_np {
destring `var', gen(`var'_en) 
label values `var'_en reduction
}

*react_qual_p react_qual_np
label define red_quality 1 "None of it" 2 "A small portion" 3 "About half" 4 "Most of it" 5 "All of it"
foreach var in react_qual_p react_qual_np {
destring `var', gen(`var'_en) 
label values `var'_en red_quality
}

*react_delay_noess_np react_delay_ess_np react_delay_ess_p react_delay_noess_p
label define acceleration 1 "Accelerated all purchases" 2 "Accelerated some purchases" 3 "Did not change the timing of purchases" 4 "Delayed some purchases" 5 "Delayed all purchases"
foreach var in react_delay_noess_np react_delay_ess_np react_delay_ess_p react_delay_noess_p {
destring `var', gen(`var'_en) 
label values `var'_en acceleration
}

*react_wage_ask_p react_wage_ask_np react_wage_askrec_p react_wage_askrec_np react_add_job_p react_add_job_np react_find_addjob_p react_find_addjob_np react_switch_job_p react_switch_job_np 
foreach var in react_wage_ask_p react_wage_ask_np react_wage_askrec_p react_wage_askrec_np react_add_job_p react_add_job_np react_find_addjob_p react_find_addjob_np react_switch_job_p react_switch_job_np {
destring `var', gen(`var'_en) 
label values `var'_en yes_no
}

*react_incr_hours_p react_incr_hours_np 
foreach var in react_incr_hours_p react_incr_hours_np {
destring `var', gen(`var'_en) 
label values `var'_en increased
}

*react_savings_p react_savings_np
label define react_savings 1 "A lot more" 2 "Somewhat more"3 "The same" 4 "Somewhat less" 5 "A lot less"
foreach var in react_savings_p react_savings_np {
destring `var', gen(`var'_en) 
label values `var'_en react_savings
}

*react_saving_cash_p react_saving_cash_np
foreach var in react_saving_cash_p react_saving_cash_np {
destring `var', gen(`var'_en) 
label values `var'_en increased
}

*react_sell_finass_p react_sell_finass_np
label define react_sell_finass 1 "Bought assets" 2 "Neither bought nor sold assets"3 "Sold assets"
foreach var in react_sell_finass_p react_sell_finass_np {
destring `var', gen(`var'_en) 
label values `var'_en react_sell_finass
}
 
*react_borrowings_np react_borrowings_p
label define react_borrowings 1 "Borrowed more" 2 "Borrowed the same" 3 "Borrowed less" 4 "I do not have any borrowing"
foreach var in react_borrowings_np react_borrowings_p {
destring `var', gen(`var'_en) 
label values `var'_en react_borrowings
}  

*react_repay_loan_np react_repay_loan_p
label define react_repay_loan 1 "Repaid faster than before" 2 "Repaid at the same rate as before" 3 "Repaid slower than before" 4 "I do not have any loan"
foreach var in react_repay_loan_np react_repay_loan_p {
destring `var', gen(`var'_en) 
label values `var'_en react_repay_loan
}   

*react_switch_mort_np react_switch_mort_p
label define react_switch_mort 1 "Yes, from variable-rate to fixed-rate" 2 "Yes, from fixed-rate to variable-rate" 3 "No, did not switch to a different mortgage type" 4 "I do not have any mortgage"
foreach var in react_switch_mort_np react_switch_mort_p {
destring `var', gen(`var'_en) 
label values `var'_en react_switch_mort
}    

*react_repay_bills_np react_repay_bills_p
label define react_repay_bills 1 "Much more difficult" 2 "Somewhat more difficult" 3 "About the same" 4 "Somewhat easier" 5 "Much easier"
foreach var in react_repay_bills_np react_repay_bills_p {
destring `var', gen(`var'_en) 
label values `var'_en react_repay_bills
}    
 
********************************************
** Reactions to higher expected inflation **
********************************************

*inf_exp_react_timing
label define inf_exp_react_timing 1 "Yes, right away" 2 "Yes, close to the time when prices increase" 3 "No"
destring inf_exp_react_timing, gen(inf_exp_react_timing_en)
label values inf_exp_react_timing_en inf_exp_react_timing

*inf_exp_react_now inf_exp_react_later
label define increase_noind 1 "Increase a lot" 2 "Somewhat increase" 3 "Somewhat decrease" 4 "Decrease a lot"
foreach var in inf_exp_react_now inf_exp_react_later {
destring `var', gen(`var'_en) 
label values `var'_en increase_noind
}  

*********************************************************
*********************POLICY VIEWS************************
*********************************************************

*********************************************
**** Policy views: Priority of inflation ****
*********************************************

*inf_priority
label define priority 1 "Not important at all" 2 "Somewhat important" 3 "Very important" 
destring inf_priority, gen(inf_priority_en)
label val inf_priority_en priority

*inf_priority_if_noef
destring inf_priority_if_noef, gen (inf_priority_if_noef_en)
label values inf_priority_if_noef_en yes_no

*encode ranking questions: inf_priority_rank_ec inf_priority_rank_sc (the questions asked to rank 5 and 6 items. Each of them is econded as a separate variable indicating the position where the item was ranked by the respondent)
foreach var in inf_priority_rank_ec {
	foreach x in 1 2 3 4 5  { //note: values follow the order of the answer options in the survey
		destring `var'_`x', gen(`var'_`x'_en) force	
		}
}

foreach var in inf_priority_rank_sc {
	foreach x in 1 2 3 4 5 6 { //note: values follow the order of the answer options in the survey
		destring `var'_`x', gen(`var'_`x'_en) force	
		}
}


******************************************
** Policy views: Inflation and politics **
******************************************

*inf_pol_cohesion
label define cohesion 1 "Increases cohesion a lot" 2 "Somewhat increases cohesion" 3 "Has no impact on cohesion" 4 "Somewhat decreases cohesion" 5 "Decreases cohesion a lot"
destring inf_pol_cohesion, gen(inf_pol_cohesion_en)
label values inf_pol_cohesion_en cohesion

*inf_pol_int_rep
label define reputation 1 "Improves reputation a lot" 2 "Somewhat improves reputation" 3 "Has no impact on reputation" 4 "Somewhat hurts reputation " 3 "Hurts reputation a lot"
destring inf_pol_int_rep, gen(inf_pol_int_rep_en)
label values inf_pol_int_rep_en reputation

*inf_pol_cause_pol
label define pol_stab 1 "Increases political stability a lot" 2 "Somewhat increases political stability" 3 "Has no impact on political stability" 4 "Somewhat decreases political stability" 5 "Decreases political stability a lot"
destring inf_pol_cause_pol, gen(inf_pol_cause_pol_en)
label values inf_pol_cause_pol_en pol_stab

**********************************************
** Policy views: Inflation and unemployment **
**********************************************
*infl_unemp_know
label define unemp_infl 1 "Strongly related" 2 "Weakly related" 3 "Unrelated"
destring infl_unemp_know, gen (infl_unemp_know_en)
label values infl_unemp_know_en unemp_infl

*infl_unemp_dir
label define inf_unemp_rel 1 "When inflation is higher, unemployment is also higher"  2 "When inflation is higher, unemployment is lower"
destring infl_unemp_dir, gen (infl_unemp_dir_en)
label values infl_unemp_dir_en inf_unemp_rel

*infl_unemp_pref
label define infl_unemp_pref 1 "Maintain low unemployment at all costs" 2 "Give priority to unemployment but be mindful about inflation" 3 "Give equal priority to inflation and unemployment" 4 "Give priority to inflation but be mindful of unemployment" 5 "Maintain low inflation at all costs"
destring infl_unemp_pref, gen(infl_unemp_pref_en)
label values infl_unemp_pref_en infl_unemp_pref

**********************************************************
** Policy views: Inflation and other economic variables **
**********************************************************

*infl_as_bad_signal
label define infl_bad_signal 1 "Always" 2 "Often" 3 "Sometimes" 4 "Rarely" 5 "Never"
destring infl_as_bad_signal, gen(infl_as_bad_signal_en)
label values infl_as_bad_signal_en infl_as_bad_signal

*infl_exp_know
label define exports 1 "Exports increase" 2 "Exports are unaffected" 3 "Exports decrease"
destring infl_exp_know, gen(infl_exp_know_en)
label values infl_exp_know_en exports


*******************************************************************
*************ECONOMIC INFORMATION ABOUT THE HOUSEHOLD**************
*******************************************************************

**********************************************
** Economic Information about the household **
**********************************************

*real_estate mortgage loans short_save cod long_save credit_card credit_card_debt
foreach var in real_estate mortgage loans short_save cod long_save credit_card credit_card_debt {
destring `var', gen(`var'_en) 
label values `var'_en yes_no	
}

	*generate dummies for previous variables
label define Yes_No 0 "No" 1 "Yes"
foreach var in real_estate mortgage loans short_save cod long_save credit_card credit_card_debt {
destring `var', gen(`var'_temp)
gen dummy`var'_en = 0 if `var'_temp == 2
replace dummy`var'_en = 1 if `var'_temp == 1
label values dummy`var'_en Yes_No	
}

*real_estate_value 
label define real_estate_value 1 "$0-$49,999" 2 "$50,000-$99,999" 3 "$100,000-$149,999" 4 "$150,000-$199,999" 5 "$200,000-299,999" 6 "$300,000-$499,999" 7 "$500,000-$749,999" 8 "$750,000-999,999" 9 "$1,000,000-$1,499,999" 10 "$1,500,000-$1,999,999" 11 "$2,000,000-2,999,999" 12 "$3,000,000+"
foreach var in real_estate_value {
destring `var', gen(`var'_en) 
label values `var'_en real_estate_value	
}

*mortgage_rate loans_rate
label define rate 1 "All fixed-rate" 2 "All capped-variable-rate" 3 "All variable-rate" 4 "A mix of the previous three"
foreach var in mortgage_rate loans_rate {
destring `var', gen(`var'_en) 
label values `var'_en rate	
}

*mortgage_value loans_value
label define mort_loan_value 1 "$0-$49,999" 2 "$50,000-$99,999" 3 "$100,000-$149,999" 4 "$150,000-$199,999" 5 "$200,000-299,999" 6 "$300,000+"
foreach var in mortgage_value loans_value {
destring `var', gen(`var'_en) 
label values `var'_en mort_loan_value	
}

*short_save_value credit_card_value
label define short_save_credit_card_value 1 "$0-$999" 2 "$1,000-$2,999" 3 "$3,000-$4,999" 4 "$5,000-$9,999" 5 "$10,000-$19,999" 6 "$20,000-$29,999" 7 "$30,000-$49,999" 8 "$50,000-99,999" 9 "$100,000-$149,999" 10 "$150,000-$199,999" 11 "$200,000-299,999" 12 "$300,000+"
foreach var in short_save_value credit_card_value {
destring `var', gen(`var'_en) 
label values `var'_en short_save_credit_card_value	
}

*cod_value long_save_value
label define cod_long_save_value 1 "$0-$9,999" 2 "$10,000-$19,999" 3 "$20,000-$29,999" 4 "$30,000-$49,999" 5 "$50,000-99,999" 6 "$100,000-$149,999" 7 "$150,000-$199,999" 8 "$200,000-299,999" 9 "$300,000+"
foreach var in cod_value long_save_value {
destring `var', gen(`var'_en) 
label values `var'_en cod_long_save_value	
}





******************************************************************
******************************************************************
**#*****************  GENERATE VARIABLES  ************************
******************************************************************
******************************************************************
***This part of the code generates variables used in the analysis such as indicators or coarser partitions of categorical variables**


***Generate branch indicators
*indicator for whether the respondent is married and asnwered to the "partner formulated" reaction questions
gen partner=1 if married =="1"
replace partner=0 if married =="1"
label var partner "1 if partner formulation, 0 if no partner formulation"

*indicator for whether the respondent saw first the policy or the impact-reaction questions
gen policy_first=0
replace policy_first=1 if order=="policy_first"

**************************************************************
**************************************************************
***************************
** Demographic Variables **
***************************

{
****female indicator****
gen women=.
replace women=0 
replace women=1 if gender==2

*******************
********Age********
*******************
*Age group - coarser partition aggregating groups
gen 			agegroup_agg=.
replace 		agegroup_agg=1 if age_group==1 
replace 		agegroup_agg=2 if age_group==2 | age_group==3
replace 		agegroup_agg=3 if age_group==4 | age_group==5
label define 	agegroup_agg 1 "18-29" 2 "30-49" 3 "50-69"
label val 		agegroup_agg agegroup_agg

gen age_class_agg= 0 if !mi(age)
replace age_class_agg=1 if age >= 18 & age <= 29
replace age_class_agg=2 if age >= 30 & age <= 49
replace age_class_agg=3 if age >= 50 & age <= 69
label define age_class_agg 1 "18-29" 2 "30-49" 3 "50-69"
label val age_class_agg age_class_agg

tab(age_class_agg), gen(age_class_agg)

*Age groups - finer partition
gen 	age_1829=.
replace age_1829=0  
replace age_1829=1 if age_group == 1
  
gen 	age_3039=.  
replace age_3039=0  
replace age_3039=1 if age_group == 2
  
gen 	age_4049=.  
replace age_4049=0  
replace age_4049=1 if age_group == 3
  
gen 	age_5059=.  
replace age_5059=0  
replace age_5059=1 if age_group == 4
  
gen 	age_6069=.  
replace age_6069=0  
replace age_6069=1 if age_group == 5

*Young/old indicators
gen 	young =.
replace young = 0  
replace young = 1 if age_group == 1 

gen old=.
replace old=0  
replace old=1 if age_group==4 | age_group==5

********************
*******Income*******
********************

*Income group (coarser partition aggregating groups)
gen 			incomegroup_agg =.
replace 		incomegroup_agg = 1 if income_group == 1 | income_group == 2
replace 		incomegroup_agg = 2 if income_group == 3 | income_group == 4 | income_group == 5 
replace 		incomegroup_agg = 3 if  income_group == 6
label define 	incomegroup_agg 1 "0k-39k" 2 "40k-124k" 3 "125k+"
label val 		incomegroup_agg incomegroup_agg

gen 	rich = 0
replace rich = 1 if incomegroup_agg >= 3

tab incomegroup_agg, gen(incomegroup_agg_)

*income group (finer partition)
tab income_group, gen(income_group)


***********************
********Race***********
***********************
gen 	race_white=.
replace race_white=0  
replace race_white=1 if race==1

gen 	race_black=.
replace race_black=0  
replace race_black=1 if race==2

gen 	race_hisp=.
replace race_hisp=0  
replace race_hisp=1 if race==3

gen 	race_asian=.
replace race_asian=0  
replace race_asian=1 if race==4

gen 	race_mixed=.
replace race_mixed=0  
replace race_mixed=1 if race==5

gen 	race_other=.
replace race_other=0  
replace race_other=1 if race==6

*race groups - coarser partition
gen 			race_group = "White" 	 
replace 		race_group = "Black" 	if(race==2)
replace 		race_group = "Hispanic" if(race==3)
replace 		race_group = "Other" 	if(race>3)
label define 	race_agg 0 "White" 1 "Black" 2 "Hispanic" 3 "Other"
encode 			race_group, gen(race_agg) label(race_agg)

tab race_agg, gen(dummy_race_agg)


**US_zone
tab(us_area), gen(dummy_us_area)


**indicator for having children
gen dummy_children=.
replace dummy_children=0 if children==1
replace dummy_children=1 if (children==2|children==3|children==4|children==5|children==6)

*********************
******Education******
*********************
**Education group (coarser partition aggregating groups)
gen 	education_agg=.
replace education_agg=1 if education_level==1 | education_level==2 
replace education_agg=2 if education_level==3 | education_level==4 | education_level==5
replace education_agg=3 if education_level==6 | education_level==7 
replace education_agg=4 if education_level==8 | education_level==9 
label define education_agg 1 "Less than HS" 2 "Less than 4-year college" 3 "4-year college/Master's" 4 "Professional degree"
label val education_agg education_agg

label variable education_agg "Education level (agg)"

*generate indicators
gen 	educ_low=0  
replace educ_low=1 if education_level==1 | education_level==2 

gen 	educ_mid=0  
replace educ_mid=1 if education_level==3 | education_level==4 | education_level==5

gen		educ_high=0  
replace educ_high=1 if education_level==6 | education_level==7 

gen 	educ_veryhigh = 0  
replace educ_veryhigh = 1 if education_level==8 | education_level==9

* college indicator defined as having a 4-year college degree. 
gen 			college=.
replace 		college=0  
replace 		college=1 if education_level>5 
label define 	college_degree 0 "No College" 1 "4-year College Degree"
label val 		college college_degree
label var 		college "College Degree"

gen no_college = 1 if college==0
	replace no_college = 0 if college==1
label var no_college "No College Degree"

*study_area
gen study_area=.
replace study_area=1 if field_of_study==4 | field_of_study==5 | field_of_study==9 | field_of_study==10 | field_of_study==12 | field_of_study==16 | field_of_study==20 | field_of_study==25 | field_of_study==31 | field_of_study==32 | field_of_study==34 | field_of_study==35 | field_of_study==42 | field_of_study==43 | field_of_study==50 | field_of_study==51 | field_of_study==52 | field_of_study==58 | field_of_study==61 | field_of_study==69 | field_of_study==74 | field_of_study==75
replace study_area=2 if field_of_study==2 | field_of_study==3 | field_of_study==13 | field_of_study==14 | field_of_study==15 | field_of_study==17 | field_of_study==18 | field_of_study==22 | field_of_study==23 | field_of_study==29 | field_of_study==33 | field_of_study==36 | field_of_study==38 | field_of_study==39 | field_of_study==41 | field_of_study==44 | field_of_study==45 | field_of_study==46 | field_of_study==48 | field_of_study==49 | field_of_study==54 | field_of_study==55 | field_of_study==56 | field_of_study==57 | field_of_study==60 | field_of_study==62 | field_of_study==63 | field_of_study==64 | field_of_study==65 | field_of_study==66 | field_of_study==67 | field_of_study==68 | field_of_study==71
replace study_area=3 if field_of_study==6 | field_of_study==7 | field_of_study==8 | field_of_study==6 | field_of_study==11 | field_of_study==13 | field_of_study==15 | field_of_study==19 | field_of_study==24 | field_of_study==26 | field_of_study==28 | field_of_study==30 | field_of_study==37 | field_of_study==40 | field_of_study==47 | field_of_study==53 | field_of_study==59 | field_of_study==70 | field_of_study==72 | field_of_study==73 | field_of_study==76
replace study_area=4 if field_of_study==1 | field_of_study==11 | field_of_study==21 | field_of_study==27
label define study_area 1 "Sciences" 2 "Professional" 3 "Arts" 4 "Econ"
label val study_area study_area

*Indicator for economic studies
gen econ=0
replace econ=1 if study_area==4 & college ==1 
label var econ "Economics related major"

*****************************
*******Employment status*****
*****************************
gen employment_agg = 1  
	replace employment_agg = 2 if(emp_status<4)
	replace employment_agg = 3 if(emp_status==4 |emp_status==6)
	replace employment_agg = 4 if(emp_status==7)
label define employment_agg 1 "Student" 2 "Working" 3 "Not working" 4 "Retiree"
label val employment_agg employment_agg

tab employment_agg, gen(dummy_employment_agg)

**spouse employment status
gen spouse_emp_agg = 1  
	replace spouse_emp_agg = 2 if(spouse_emp_status<4)
	replace spouse_emp_agg = 3 if(spouse_emp_status==4 |spouse_emp_status==6)
	replace spouse_emp_agg = 4 if(spouse_emp_status==7)
label define spouse_emp_agg 1 "Student" 2 "Working" 3 "Not working" 4 "Retiree"
label val spouse_emp_agg spouse_emp_agg

tab spouse_emp_agg, gen(dummy_spouse_emp_agg)

**********************************
*******Political affiliation******
**********************************
*Republican-Democrat indicators
gen republican=.
replace republican=0  
replace republican=1 if pol_aff==1

gen democrat=.
replace democrat=0  
replace democrat=1 if pol_aff==2

*Political groups
gen political_agg = 1 if(pol_aff==2)
	replace political_agg = 2 if(pol_aff==1)
	replace political_agg = 3 if(pol_aff>2)
label define political_agg 1 "Democrat" 2 "Republican" 3 "Independent and others"
label val political_agg political_agg

label variable political_agg "Political affiliation (aggregate)"

tab political_agg, gen(political_agg)


****marital status****
gen dummy_married = 0 if !missing(marital_status)
replace dummy_married=1 if marital_status==2

****medicaid indicator****
gen dummy_medicaid = 0 if !missing(medicaid)
replace dummy_medicaid=1 if medicaid==1

****food_stamps indicator****
gen dummy_food_stamps = 0 if !missing(food_stamps)
replace dummy_food_stamps=1 if food_stamps==1

**aggregate vote and novote preferences
gen vote_who_agg = vote_who
replace vote_who_agg = novote_who if missing(vote_who)
replace vote_who_agg=3 if vote_who_agg==3|vote_who_agg==4|vote_who_agg==5

label define vote_agg 1 "Biden" 2 "Trump" 3 "Others"
label values vote_who_agg vote_agg

***********************************************
************label main indicators**************
***********************************************
label var age_class_agg1 "Age 18-29"
label var age_class_agg2 "Age 30-49"
label var age_class_agg3 "Age 50-69"
label var political_agg2 "Republican"
label var incomegroup_agg_1 "Low-income"
label var incomegroup_agg_2 "Middle-income"
label var incomegroup_agg_3 "High-income"
label var women "Female"
label var democrat "Democrat"
label var republican "Republican"
label var political_agg3 "Independent and Others"
label var dummy_race_agg1 "White"
label var dummy_race_agg2 "Black"
label var dummy_race_agg3 "Hispanic"
label var dummy_race_agg4 "Other"
label var dummy_employment_agg1 "Student"
label var dummy_employment_agg2 "Working"
label var dummy_employment_agg3 "Not working"
label var dummy_employment_agg4 "Retiree"
label var dummy_children "Has children"
label var dummy_us_area1 "Northeast"
label var dummy_us_area2 "South"
label var dummy_us_area3 "Midwest"
label var dummy_us_area4 "West"
label var policy_first "Policy questions first"
label var college "4-year college"
label var dummy_married "Married"


}



***************************************
** Economic information on household **
***********(extensive margin)**********

{
foreach var in real_estate mortgage loans short_save cod long_save credit_card credit_card_debt  {		
	gen dummy_`var' = 0 if !missing(`var'_en)
	replace dummy_`var' = 1 if `var'_en == 1
	label var dummy_`var' "`var'"
}

}



****************************************
** Info on inflation and expectations **
****************************************
{
*inflation as a yardstick
label define inf_yard_agree 0 "Indifferent or disagree" 1 "At least somewhat agree" 
gen inf_yard_agree=0
replace inf_yard_agree=1 if inf_yardstick_en==1|inf_yardstick_en==2
label values inf_yard_agree

*importance of being updated about inflation
label define dummy_inf_info_utd 0 "No more than somewhat important" 1 "At least very important"
gen dummy_inf_info_utd=0 if !missing(inf_info_utd_i_en)
replace dummy_inf_info_utd=1 if inf_info_utd_i_en==1|inf_info_utd_i_en==2
label var dummy_inf_info_utd "Importance of being updated"

*attention towards inflation
label define dummy_inf_info_att 0 "Remained the same or decreased" 1 "At least increased somewhat"
gen dummy_inf_info_att=0 if !missing(inf_info_att_i_en)
replace dummy_inf_info_att=1 if inf_info_att_i_en==1|inf_info_att_i_en==2
label var dummy_inf_info_utd "Indicator equal to 1 if recent increased attention towards inflation"


***correct understanding infl questions
gen correct_under_1=0 if !missing(inf_understanding_1_en)
replace correct_under_1=1 if inf_understanding_1_en==110
label var correct_under_1 "Indicator equal 1 if correct future price given inflation"

gen correct_under_2=0 if !missing(inf_understanding_2_en)
replace correct_under_2=1 if inf_understanding_2_en==1
label var correct_under_2 "Indicator equal 1 if correct inflation rate given future price"



***which sources  of news about inflation you see
*(note that for any item the question asked whether that was or not a main source of news)
foreach var in inf_info_source_i {
	foreach x in 1 2 3 4 5 6 7 8 9 10 11 { //note: values follow the order of the answer options in the survey
		gen dummy_`var'_`x' = 0 if !missing(`var'_`x'_en)
		replace dummy_`var'_`x' = 1 if `var'_`x'_en== 1 
		label var dummy_`var'_`x' "Indicator equal to 1 if it is a main source of info"
}
}

*sources - coarser partition 
gen TV_info=0 
replace TV_info=1 if inf_info_source_i_5_en==1|inf_info_source_i_6_en==1|inf_info_source_i_7_en==1
label var TV_info "Indicator equal 1 if TV is one of main source of info"

gen social_media_info =0
replace social_media_info=1 if inf_info_source_i_1_en==1
label var social_media_info "Indicator equal 1 if social media is one of main source of info"

gen radio_info =0
replace radio_info=1 if inf_info_source_i_8_en==1
label var radio_info "Indicator equal 1 if radio is one of main source of info"

gen newspapers_info=0 
replace newspapers_info=1 if inf_info_source_i_2_en==1|inf_info_source_i_3_en==1|inf_info_source_i_4_en==1
label var newspapers_info "Indicator equal 1 if newspapers is one of main source of info"



*direction of guesses about past and future inflation
foreach var in inf_info_dir_i inf_exp_dir_i {
	gen dummy_`var' = 0 if !missing(`var'_en)
	replace dummy_`var' = 1 if `var'_en == 1
	label var dummy_`var' "`var' 1 if increased"
}


}



******************************
****** Personal impacts ******
******************************

******************
** impacts as DM**
******************
{
	
	*who takes financial decisions
	foreach var in fin_decisions {
	gen dummy_`var' = 0 if !missing(`var'_en)
	replace dummy_`var' = 1 if `var'_en == 4| `var'_en == 5 
	label var dummy_`var' "Make most financial decisions herself "
		}

		
*shopping frequency
label define dummy_shopping 0 "No more than sometimes" 1 "At least most of the time"
gen dummy_shopping = 0 if !missing(shopping_en)
replace dummy_shopping=1 if shopping_en==1|shopping_en==2
label values dummy_shopping dummy_shopping
label var dummy_shopping "Doing household's shopping"


*fin_decisions - coarser partition
label define fin_decisions_agg 1"Someone else makes at least most" 2 "Equally shared" 3 "At least most myself"
gen fin_decisions_agg= 1
replace fin_decisions_agg=2 if fin_decisions_en==3
replace fin_decisions_agg=3 if fin_decisions_en==4|fin_decisions_en==5
label values fin_decisions_agg fin_decisions_agg

}	
	
************************
** impacts as consumer**
************************
{
gen dummy_ppower = 0 if !missing(pers_imp_cons_ppower_en)
replace dummy_ppower=1 if pers_imp_cons_ppower_en==1|pers_imp_cons_ppower_en==2
label var dummy_ppower "Purchasing power decreased"

gen dummy_ppower_know = 0 if !missing(infl_ppower_know_en)
replace dummy_ppower_know=1 if infl_ppower_know_en==4|infl_ppower_know_en==5
label var dummy_ppower_know "Purchasing power decreases even if salaries increases"

gen dummy_compar_shopping = 0 if !missing(pers_imp_cons_compar_en)
replace dummy_compar_shopping=1 if pers_imp_cons_compar_en==4|pers_imp_cons_compar_en==5
label var dummy_compar_shopping "Comparison shopping harder"

gen dummy_cons_quality = 0 if !missing(pers_imp_cons_qual_en)
replace dummy_cons_quality=1 if pers_imp_cons_qual_en==1|pers_imp_cons_qual_en==2
label var dummy_cons_quality "Quality of goods purchased decreased"

gen dummy_shrinkflation = 0 if !missing(pers_imp_cons_quant_en)
replace dummy_shrinkflation=1 if pers_imp_cons_quant_en==3
label var dummy_shrinkflation "More widespread shrinkflation"

}

***********************
** impacts as worker **
***********************
{
*indicator takes value 1 if the answer is "Yes"
foreach var in job_change_en job_change_wage_en wage_indexed_en {
	gen dummy_`var' = 0 if !missing(`var')
	replace dummy_`var' = 1 if  `var'== 1 
}

label var dummy_job_change_en "Changed job in the last two years"
label var dummy_job_change_wage_en "Received wage increase in the last two years"
label var dummy_wage_indexed_en "Salary indexed to inflation"

replace dummy_wage_indexed_en=. if wage_indexed_en==3 // for those who answered "I do not know"


gen dummy_jobch_yes_wage_onjob_en = 0 
	replace dummy_jobch_yes_wage_onjob_en = 1 if  jobch_yes_wage_onjob_en== 1
	label var dummy_jobch_yes_wage_onjob_en "Change in salary due to job change"

	{
*Indicators to check there are no cases of people who said they received a change in wage due to job change, but before either said they did not change job or they did not received any wage increase

gen dummy_jobch_yes_wage_onjob_en_ow = 0 if job_change_wage_en==1
	replace dummy_jobch_yes_wage_onjob_en_ow = 1 if  jobch_yes_wage_onjob_en== 1

gen dummy_jobch_yes_wage_onjob_en_oj = 0 if job_change_en==1
	replace dummy_jobch_yes_wage_onjob_en_oj = 1 if  jobch_yes_wage_onjob_en== 1
	}


gen dummy_concern_future_work=0 if !missing(concern_future_work_en)
replace dummy_concern_future_work=1 if concern_future_work_en==4|concern_future_work_en==5
label var dummy_concern_future_work "Very concerned about future job situation"
	
gen dummy_wage_rel = 0 if !missing(pers_imp_work_rel_en)
replace dummy_wage_rel=1 if pers_imp_work_rel_en==1|pers_imp_work_rel_en==2
label var dummy_wage_rel "Others' wages increase more quickly"


gen dummy_wage_high_rel = 0 if !missing(pers_imp_work_rel_hi_en)
replace dummy_wage_high_rel=1 if pers_imp_work_rel_hi_en==1|pers_imp_work_rel_hi_en==2
label var dummy_wage_high_rel "Higher income people's wages increase more quickly"

gen dummy_wage_prices = 0 if !missing(inf_infl_wage_growth_en)
replace dummy_wage_prices=1 if inf_infl_wage_growth_en==1|inf_infl_wage_growth_en==2
label var dummy_wage_prices "Prices at least somewhat faster than wages"

gen dummy_work_satisf = 0 if !missing(pers_imp_work_satisf_en)
replace dummy_work_satisf=1 if pers_imp_work_satisf_en==3
label var dummy_work_satisf "Job satisfation decreases when income and prices go up by as much"

gen dummy_adjust_sf=0 if !missing(inf_adj_how_many_sf_en)
replace dummy_adjust_sf=1 if inf_adj_how_many_sf_en==3|inf_adj_how_many_sf_en==4
label var dummy_adjust_sf "Many small businesses adjust wages"

gen dummy_adjust_bf=0 if !missing(inf_adj_how_many_bf_en)
replace dummy_adjust_bf=1 if inf_adj_how_many_bf_en==3|inf_adj_how_many_bf_en==4
label var dummy_adjust_bf "Many big companies adjust wages"

gen dummy_work_2_me=0 if !missing(pers_imp_work_2_me_en)
replace dummy_work_2_me=. if pers_imp_work_2_me_en==6
replace dummy_work_2_me=1 if pers_imp_work_2_me_en==4|pers_imp_work_2_me_en==5
label var dummy_work_2_me "Adjust wage in more than one year" 

gen dummy_hyp_sh=0 if !missing(pers_imp_work_2_me_en)
replace dummy_hyp_sh=. if pers_imp_work_hyp_sh_en==4
replace dummy_hyp_sh=1 if pers_imp_work_2_me_en==3
label var dummy_hyp_sh "Income higher if no inflation"

gen dummy_career_progression=0 if !missing(job_wage_why_en)
replace dummy_career_progression=1 if job_wage_why_en==1
label var dummy_career_progression `" "Wage increase primarily due to" "job performance/career progression" "' 

gen dummy_infl_adjustment=0 if !missing(job_wage_why_en)
replace dummy_infl_adjustment=1 if job_wage_why_en==2
label var dummy_infl_adjustment "To offset price increases"

gen dummy_both_factors=0 if !missing(job_wage_why_en)
replace dummy_both_factors=1 if job_wage_why_en==3
label var dummy_both_factors "Both career and inflation reasons"
}


*****************************
** impacts as asset holder **
*****************************
{
	
gen dummy_imp_finass=0 if !missing(pers_imp_ass_fin_ass_en)
replace dummy_imp_finass=1 if pers_imp_ass_fin_ass_en==3
label var dummy_imp_finass "Value of financial assets decreased"

gen dummy_imp_savings=0 if !missing(pers_imp_ass_savings_en)
replace dummy_imp_savings=. if pers_imp_ass_savings_en==4
replace dummy_imp_savings=1 if pers_imp_ass_savings_en==3
label var dummy_imp_savings "Savings decreased"
	
gen dummy_imp_debt_value=0 if !missing(pers_imp_ass_debt_va_en)
replace dummy_imp_debt_value=. if pers_imp_ass_debt_va_en==4
replace dummy_imp_debt_value=1 if pers_imp_ass_debt_va_en==1
label var dummy_imp_debt_value "Reduced real value of debt"

gen dummy_imp_debt_repay=0 if !missing(pers_imp_ass_debt_re_en)
replace dummy_imp_debt_repay=. if pers_imp_ass_debt_re_en==4
replace dummy_imp_debt_repay=1 if pers_imp_ass_debt_re_en==3
label var dummy_imp_debt_repay "Repayment of debt harder"
}


***************************
** psychological impacts **
***************************
{
gen dummy_neg_outlook = 0 if !missing(pers_imp_econoutlook_en)
replace dummy_neg_outlook=1 if pers_imp_econoutlook_en==1|pers_imp_econoutlook_en==2
label var dummy_neg_outlook "Negative effect of inflation on future econ well-being"

gen dummy_stress = 0 if !missing(pers_imp_psy_stress_en)
replace dummy_stress=1 if pers_imp_psy_stress_en==1
label var dummy_stress "More stressed because of inflation"


foreach x in 1 2 3 4 5 6 7 { // note: values follow the order of the answer options in the survey
	gen dummy_pers_imp_psy_causes_`x' = 0 if !missing(pers_imp_psy_causes_`x'_en)
	replace dummy_pers_imp_psy_causes_`x' = 1 if pers_imp_psy_causes_`x'_en == 1 
	label var dummy_pers_imp_psy_causes_`x' " dummy_pers_imp_psy_causes_`x' 1 if it is a cause for stress "
}


}



***************************************
********** Personal reactions *********
***************************************

***********************************
** Actual reactions to inflation **
***********************************

*NOTE: Aggregating partner and no partner formulations
foreach var in react_quant react_qual react_delay_noess react_delay_ess react_wage_ask react_wage_askrec react_add_job react_find_addjob react_incr_hours react_switch_job react_savings react_saving_cash react_sell_finass  react_borrowings react_repay_loan react_switch_mort react_repay_bills  {
	gen `var'_agg_en = `var'_p_en
	replace `var'_agg_en = `var'_np_en if missing(`var'_p_en)
}

*label all the newly created aggregated variables
{

foreach var in react_quant_agg {
label values `var'_en reduction
}

foreach var in react_qual_agg {
label values `var'_en red_quality
}

foreach var in react_delay_ess_agg react_delay_noess_agg { 
label values `var'_en acceleration
}

foreach var in react_wage_ask_agg react_wage_askrec_agg react_add_job_agg   react_find_addjob_agg react_switch_job_agg {
label values `var'_en yes_no
}


foreach var in react_incr_hours_agg react_saving_cash_agg {
label values `var'_en increased
}


foreach var in react_sell_finass_agg  {
label values `var'_en react_sell_finass
}



foreach var in react_savings_agg  {
label values `var'_en react_savings
}

  

foreach var in  react_borrowings_agg {
label values `var'_en react_borrowings
}  


foreach var in react_repay_loan_agg {
label values `var'_en react_repay_loan
}   


foreach var in react_switch_mort_agg {
label values `var'_en react_switch_mort
}    


foreach var in react_repay_bills_agg {
label values `var'_en react_repay_bills
}    	
	
	
	
}
	



{
	
***Reactions as a consumer
gen dummy_react_quant = 0 if !missing(react_quant_agg_en)
replace dummy_react_quant=1 if react_quant_agg_en==1|react_quant_agg_en==2
label var dummy_react_quant "Reduced quantity of goods purchased"

gen dummy_react_qual = 0 if !missing(react_qual_agg_en)
replace dummy_react_qual=1 if react_qual_agg_en==4|react_qual_agg_en==5
label var dummy_react_qual "Most of the shopping has shifted towards lower-priced lower-quality goods"

gen dummy_delay_ess = 0 if !missing(react_delay_ess_agg_en)
replace dummy_delay_ess=1 if react_delay_ess_agg_en==4|react_delay_ess_agg_en==5
label var dummy_delay_ess "Delayed at least some essential goods"

gen dummy_delay_noess = 0 if !missing(react_delay_noess_agg_en)
replace dummy_delay_noess=1 if react_delay_noess_agg_en==4|react_delay_noess_agg_en==5
label var dummy_delay_noess "Delayed at least some non essential goods"

***Reactions as a worker
gen dummy_react_incr_hours = 0 if !missing(react_incr_hours_agg_en)
replace dummy_react_incr_hours=1 if react_incr_hours_agg_en==1|react_incr_hours_agg_en==2
label var dummy_react_incr_hours "Increased hours worked"

gen dummy_react_wage_ask=0 if !missing(react_wage_ask_agg_en)
replace dummy_react_wage_ask=1 if react_wage_ask_agg_en==1
label var dummy_react_wage_ask "Asked for wage increase"

gen dummy_react_wage_askrec=0 if !missing(react_wage_askrec_agg_en)
replace dummy_react_wage_askrec=1 if react_wage_askrec_agg_en==1
label var dummy_react_wage_askrec "Received wage increase asked for"

gen dummy_react_add_job=0 if !missing(react_add_job_agg_en)
replace dummy_react_add_job=1 if react_add_job_agg_en==1
label var dummy_react_add_job "Looked for additional job"

gen dummy_react_find_addjob=0 if !missing(react_find_addjob_agg_en)
replace dummy_react_find_addjob=1 if react_find_addjob_agg_en==1
label var dummy_react_find_addjob "Found additional job looked for"

gen dummy_react_switch_job=0 
replace dummy_react_switch_job=1 if react_switch_job_agg_en==1
label var dummy_react_switch_job "Switched to higher paying job"

***Reactions as asset holder
gen dummy_savings = 0 if !missing(react_savings_agg_en)
replace dummy_savings=1 if react_savings_agg_en==4|react_savings_agg_en==5
label var dummy_savings "Less savings"

gen dummy_cash_savings = 0 if !missing(react_saving_cash_agg_en)
replace dummy_cash_savings=1 if react_saving_cash_agg_en==4|react_saving_cash_agg_en==5
label var dummy_cash_savings "Less cash savings"

gen dummy_sel_finass = 0 if !missing(react_sell_finass_agg_en)
replace dummy_sel_finass=1 if react_sell_finass_agg_en==3
label var dummy_sel_finass "Sold financial assets"

gen dummy_borrowings = 0 if !missing(react_borrowings_agg_en)
replace dummy_borrowings=. if react_borrowings_agg_en==4
replace dummy_borrowings=1 if react_borrowings_agg_en==1
label var dummy_borrowings "Borrowed more"

gen dummy_repay_loan = 0 if !missing(react_repay_loan_agg_en)
replace dummy_repay_loan=. if react_repay_loan_agg_en==4
replace dummy_repay_loan=1 if react_repay_loan_agg_en==3
label var dummy_repay_loan "Repaid loans slower"


gen dummy_diff_bills = 0 if !missing(react_repay_bills_agg_en)
replace dummy_diff_bills=1 if react_repay_bills_agg_en==1|react_repay_bills_agg_en==2
label var dummy_diff_bills "More difficult to pay bills"
	

}

********************************************
** Reactions to higher expected inflation **
********************************************
{
gen dummy_exp_react_now_agg=0 if!missing(inf_exp_react_now_en)	
replace dummy_exp_react_now_agg=1 if inf_exp_react_now_en==3|inf_exp_react_now_en==4
label var dummy_exp_react_now_agg "Decrease spending right away if expected i rises"

gen dummy_exp_react_later_agg=0 if!missing(inf_exp_react_later_en)	
replace dummy_exp_react_later_agg=1 if inf_exp_react_later_en==3|inf_exp_react_later_en==4
label var dummy_exp_react_later_agg "Decrease spending later if expected i rises"

gen dummy_react_timing_now=0 if !missing(inf_exp_react_timing_en)
replace dummy_react_timing_now=1 if inf_exp_react_timing_en==1
label var dummy_react_timing_now "Adjust right away if if expected i rises"

gen dummy_react_timing_later=0 if !missing(inf_exp_react_timing_en)
replace dummy_react_timing_later=1 if inf_exp_react_timing_en==2
label var dummy_react_timing_now "Adjust later if if expected i rises"
}


*****************************************
************* Policy views **************
*****************************************

*INFLATION AND POLITICS
{
gen dummy_inf_pol_stability = 0 if !missing(inf_pol_cause_pol_en)
replace dummy_inf_pol_stability=1 if inf_pol_cause_pol_en==4|inf_pol_cause_pol_en==5
label var dummy_inf_pol_stability "Inflation decreases political stability"

gen dummy_inf_pol_cohesion = 0 if !missing(inf_pol_cohesion_en)
replace dummy_inf_pol_cohesion=1 if inf_pol_cohesion_en==4|inf_pol_cohesion_en==5
label var dummy_inf_pol_cohesion "Inflation decreases social cohesion"

gen dummy_inf_pol_reputation = 0 if !missing(inf_pol_int_rep_en)
replace dummy_inf_pol_reputation=1 if inf_pol_int_rep_en==4|inf_pol_int_rep_en==5
label var dummy_inf_pol_reputation "Inflation decreases international reputation"
}

*********RANKING QUESTIONS
{
*ECONOMIC ISSUES
gen dummy_inf_first_ec = 0 if !missing(inf_priority_rank_ec_1_en)
replace dummy_inf_first_ec=1 if inf_priority_rank_ec_1_en==1
label var dummy_inf_first_ec "Inflation first among economic issues"

gen dummy_lowu_first_ec = 0 if !missing(inf_priority_rank_ec_2_en)
replace dummy_lowu_first_ec=1 if inf_priority_rank_ec_2_en==1
label var dummy_lowu_first_ec "Low unemployment first among economic issues"

gen dummy_growth_first_ec = 0 if !missing(inf_priority_rank_ec_3_en)
replace dummy_growth_first_ec=1 if inf_priority_rank_ec_3_en==1
label var dummy_growth_first_ec "Growth first among economic issues"

gen dummy_defense_first_ec = 0 if !missing(inf_priority_rank_ec_4_en)
replace dummy_defense_first_ec=1 if inf_priority_rank_ec_4_en==1
label var dummy_defense_first_ec "National defense first among economic issues"

gen dummy_finstab_first_ec = 0 if !missing(inf_priority_rank_ec_5_en)
replace dummy_finstab_first_ec=1 if inf_priority_rank_ec_5_en==1
label var dummy_finstab_first_ec "Financial stability first among economic issues"

*SOCIAL ISSUES
gen dummy_inf_first_sc = 0 if !missing(inf_priority_rank_sc_1_en)
replace dummy_inf_first_sc=1 if inf_priority_rank_sc_1_en==1
label var dummy_inf_first_sc "Inflation first among social issues"

gen dummy_gun_first_sc = 0 if !missing(inf_priority_rank_sc_2_en)
replace dummy_gun_first_sc=1 if inf_priority_rank_sc_2_en==1
label var dummy_gun_first_sc "Gun rights first among economic issues"

gen dummy_civil_first_sc = 0 if !missing(inf_priority_rank_sc_3_en)
replace dummy_civil_first_sc=1 if inf_priority_rank_sc_3_en==1
label var dummy_civil_first_sc "Civil rights among economic issues"

gen dummy_abortion_first_sc = 0 if !missing(inf_priority_rank_sc_4_en)
replace dummy_abortion_first_sc=1 if inf_priority_rank_sc_4_en==1
label var dummy_abortion_first_sc "Abortion first among economic issues"

gen dummy_educ_first_sc = 0 if !missing(inf_priority_rank_sc_5_en)
replace dummy_educ_first_sc=1 if inf_priority_rank_sc_5_en==1
label var dummy_educ_first_sc "Education first among economic issues"

gen dummy_health_first_sc = 0 if !missing(inf_priority_rank_sc_6_en)
replace dummy_health_first_sc=1 if inf_priority_rank_sc_6_en==1
label var dummy_health_first_sc "Health first among economic issues"

}

*****Other questions in policy views survey block 
{
gen dummy_inf_priority = 0 if !missing(inf_priority_en)
replace dummy_inf_priority=1 if inf_priority_en==3
label var dummy_inf_priority "Inflation is a very important priority"

gen dummy_infl_unemp_know = 0 if !missing(infl_unemp_know_en)
replace dummy_infl_unemp_know=1 if infl_unemp_know_en==1|infl_unemp_know_en==2
label var dummy_infl_unemp_know "u and pi related"

gen dummy_infl_unemp_dir=0 if !missing(infl_unemp_dir_en)
replace dummy_infl_unemp_dir=1 if infl_unemp_dir_en==2
label var dummy_infl_unemp_dir "u and pi negatively related"

gen dummy_infl_unemp_dir_pos = 0 if !missing(infl_unemp_dir_en)
replace dummy_infl_unemp_dir_pos=1 if infl_unemp_dir_en==1
label var dummy_infl_unemp_dir_pos "u and pi positively related"

gen dummy_infl_unemp_pref = 0 if !missing(infl_unemp_pref_en)
replace dummy_infl_unemp_pref=1 if infl_unemp_pref_en==4 | infl_unemp_pref_en==5
label var dummy_infl_unemp_pref "pi more important than i"

gen dummy_infl_bad_signal = 0 if !missing(infl_as_bad_signal_en)
replace dummy_infl_bad_signal=1 if infl_as_bad_signal_en==1 | infl_as_bad_signal_en==2
label var dummy_infl_bad_signal "pi is often or always a bad signal for the economy"

gen dummy_infl_exports = 0 if !missing(infl_exp_know_en)
replace dummy_infl_exports=1 if infl_exp_know_en==3 
label var dummy_infl_exports "pi decreases exports"

}