// =============================================================================
// 
//	Event Study of Sustained Debt Reductions
//
// =============================================================================


/*-------------------------------------------
	Globals: Variables etc
---------------------------------------------*/

* variables in levels vs differences relative to time 0
global lev_vars  "r_real real_gdp_gr r_minus_g  prim_surplus_gdp surplusx_gdp unemp_weo ngap_gdp " 
		// r_real r_minus_g cons_gdp ex_gdp im_gdp  inv_gdp cons_gdp polar prim_surplus_gdp"
global diff_vars "debt_gdp rev_gdp exp_gdp non_interest_expx_gdp"

* window around event	
global W = 5
* window requiring non-missing observations
global Wnm = $W 



/*-------------------------------------------
	Prepare Analysis Dataset
---------------------------------------------*/

use "$cleaned/working.dta", clear

ren surplus_gdp          surplusx_gdp
ren non_interest_exp_gdp non_interest_expx_gdp

* level variables: leads and lag
foreach x of varlist $lev_vars {	
	foreach t of numlist 0(1)$W{ 
		cap drop m_Ld`t'_`x' m_Fd`t'_`x' 
		// lagged level
		gen m_Ld`t'_`x' = L`t'.`x'
		// forward level
		gen m_Fd`t'_`x' = F`t'.`x'
		winsor2 m_Ld`t'_`x' m_Fd`t'_`x', cuts(1 99) replace // winsor. Equit Guinea is a large outlier in terms of primary surplus				

	}
}

* differences variables: leads and lag
foreach x of varlist $diff_vars {	
	foreach t of numlist 0(1)$W{ 
		cap drop m_Ld`t'_`x' m_Fd`t'_`x' 
		// lagged difference
		gen m_Ld`t'_`x' = -(`x' - L`t'.`x')				// minus because we want to show cumulative change relative to 0
		// forward difference
		gen m_Fd`t'_`x' = F`t'.`x' - `x'
		winsor2 m_Ld`t'_`x' m_Fd`t'_`x', cuts(1 99) replace
	}
}

tempfile event_data 
save    `event_data'


/*-------------------------------------------
	Define sample of events
---------------------------------------------*/

*** (1) All large debt reductions without a default/restructuring
use `event_data', clear
keep if large_debt_red==1 
drop if restru_flag==1
drop if year>=2016

* require balance panel for key variables
foreach x of varlist real_gdp_gr debt_gdp prim_surplus_gdp surplusx_gdp {
	foreach t of numlist 0(1)$Wnm { 
		// lagged difference
		drop if m_Ld`t'_`x'== .
		// forward difference
		drop if m_Fd`t'_`x'== .
	}
}

gen event1 = 1
keep country country_name year event1 

keep country year event1 
	drop if country=="GNQ" // this is a large primary surplus outlier, but winsorizing deals just as well with it
count 
local N1 = `r(N)'
tempfile events1
save    `events1'


*** (2) Emerging markets/DC (https://www.imf.org/en/Publications/WEO/weo-database/2023/April/groups-and-aggregates)
use `events1', clear
drop if country=="AND" | country=="AUS" | country=="AUT" | country=="BEL" | country=="CAN" | country=="HRV" | country=="CYP" | country=="CZE" |  country=="DNK" | country=="EST" | country=="FIN" | country=="FRA" | country=="DEU" | country=="GRC" | country=="HKG" | country=="ISL" | country=="IRL" | country=="ISR" |  country=="ITA" | country=="JPN" | country=="KOR" | country=="LVA" | country=="LTU" |  country=="LUX" | country=="MAC" | country=="MLT" | country=="NLD" | country=="NZL" | country=="NOR" | country=="PRT" | country=="PRI" | country=="SMR" | country=="SGP" | country=="SVK" | country=="SVN" | country=="ESP" | country=="SWE" | country=="CHE" | country=="TWN" |  country=="GBR" | country=="USA"

gen event2 = 1
count 
local N2 = `r(N)'
keep country year event2
tempfile events2
save    `events2'

*** (3) Advanced economy events based on IMF classification 
use `events1', clear
keep if country=="AND" | country=="AUS" | country=="AUT" | country=="BEL" | country=="CAN" | country=="HRV" | country=="CYP" | country=="CZE" |  country=="DNK" | country=="EST" | country=="FIN" | country=="FRA" | country=="DEU" | country=="GRC" | country=="HKG" | country=="ISL" | country=="IRL" | country=="ISR" |  country=="ITA" | country=="JPN" | country=="KOR" | country=="LVA" | country=="LTU" |  country=="LUX" | country=="MAC" | country=="MLT" | country=="NLD" | country=="NZL" | country=="NOR" | country=="PRT" | country=="PRI" | country=="SMR" | country=="SGP" | country=="SVK" | country=="SVN" | country=="ESP" | country=="SWE" | country=="CHE" | country=="TWN" |  country=="GBR" | country=="USA"
gen event3 = 1
keep country year event3
count 
local N3 = `r(N)'
tempfile events3
save    `events3'

*** (4) Jamaica 2013
use `event_data', clear
keep if country=="JAM" & year==2013
gen event4 = 1
keep country year event4
tempfile events4
save    `events4'




*** Merge events back onto dataset
use `event_data', clear
forv e=1(1)4 {
	merge 1:1 country year using `events`e'', nogen
}

tempfile event_data_final
save    `event_data_final'


use `event_data_final', clear

/*-------------------------------------------
	Event study (sample mean)
---------------------------------------------*/

foreach e in 1 2 3 4 {
	
	use `event_data_final', clear
	keep if event`e'==1
	count
	
	// get the mean for difference of variable
	collapse (mean) m_Ld* m_Fd*
	
	// placeholder variable: no meaning
	gen tmp = 1
	
	// reshape to long dataset, each row is a time period
	greshape long ///
	 @_real_gdp_gr @_r_real @_r_minus_g @_prim_surplus_gdp @_surplusx_gdp ///
	 @_debt_gdp ///
	 @_rev_gdp @_non_interest_expx_gdp @_unemp_weo @_ngap_gdp, i(tmp) j(string_var) string
			
	ren _* *
	// separate the statistics in the string variable name, "m" represents mean
	gen stat = substr(string_var, 1, strpos(string_var, "_")-1)
	gen num_str = substr(string_var, strpos(string_var, "_")+1, .)
	drop if num_str =="Fd0" // Fd0 and Ld0 is the same
	
	// separate the period in the string variable name
	gen num = .
	replace num = real(substr(num_str, strpos(num_str, "d")+1, .))
	replace num = -num if strpos(num_str, "Ld")>0

	drop num_str string_var

	replace stat = "_" + stat
		
	drop tmp
	sort num
	
	gen event = `e'
	order event num 
	tempfile means_`e'
	save    `means_`e''
}


* combine event studies 
use `means_1', clear
append using `means_2'
append using `means_3'
append using `means_4'


/*-------------------------------------------
	Event study plot
---------------------------------------------*/

* graph globals
glo labs1 10pt
glo xs1   10pt 
glo ts1   16pt 
glo legs1 10pt

* debt_to_gdp
tw 	(conn debt_gdp num if event==1, $m1 $l1 lwidth(thick) lp(solid)) ///
	(conn debt_gdp num if event==3, $m4 $l4 lwidth(thick) lp(dash)) ///
	(conn debt_gdp num if event==2, $m3 $l3 lwidth(thick) lp(dash)) ///
	(conn debt_gdp num if event==4, $mJ $lJ lwidth(thick) lp(solid)) ///
	,xlabel(-${W}(1)${W}, labsize($labs1 )) ylabel(,labsize($labs1 )) ///
	xtitle("Year Relative to Large Debt Reduction", margin(top) size($xs1 )) ///
	ytitle("Public debt-to-GDP change relative to t=0", size($xs1 ) ) ///
	legend(order(1 "Large debt reductions (N=`N1')" 2 "Advanced (N=`N3')"  3 "Emerging and developing (N=`N2')"  4 "Jamaica (time 0=2013)" ) ///
		rows(3) region(lstyle(none)) fcolor(white%0) size($legs1 ) ) ///
	/*title("Government Debt-to-GDP (normalized)", size($ts1 ))*/  /*$fig_size*/ name(real_gdp,replace)
graph export "$pres/Fig_event_debt_gdp.pdf", as(pdf) replace


*** 6 panel graph

foreach v of var real_gdp_gr r_real r_minus_g prim_surplus_gdp rev_gdp non_interest_expx_gdp{
	if "`v'" == "real_gdp_gr"{
		loc tag = "Real GDP Growth, g (%)"
		}
	if "`v'" == "r_real"{
		loc tag = "Real Interest Rate, r (%)"
		}
	if "`v'" == "r_minus_g"{
		loc tag = "r-g (%)"
		}
	if "`v'" == "prim_surplus_gdp"{
		loc tag = "Primary-surplus/GDP (%)"
		}
	if "`v'" == "rev_gdp"{
		loc tag = "Revenue/GDP (ppt)"
		}
	if "`v'" == "non_interest_expx_gdp"{
		loc tag = "Non-int. spending/GDP (ppt)"
		}
		
	tw 	(conn `v' num if event==1, $m1 $l1 lwidth(thick) lp(solid)) ///
		(conn `v' num if event==3, $m4 $l4 lwidth(thick) lp(dash)) ///
		(conn `v' num if event==2, $m3 $l3 lwidth(thick) lp(dash)) ///
		(conn `v' num if event==4, $mJ $lJ lwidth(thick) lp(solid)) ///
		,xlabel(-${W}(1)${W}, labsize($labs1 )) ylabel(,labsize($labs1 )) ///
		xtitle("") ///
		ytitle("", size($xs1 ) ) ///
	legend(order(1 "Large debt reductions (N=`N1')" 2 "Advanced (N=`N3')"  3 "Emerging and developing (N=`N2')"  4 "Jamaica (time 0=2013)" ) ///
			rows(3) region(lstyle(none)) fcolor(white%0) size($legs1 ) ) ///
		title("`tag'", size($ts1 ))  $fig_size name(`v',replace)	
}

* combine and output
grc1leg real_gdp_gr r_real r_minus_g prim_surplus_gdp  rev_gdp non_interest_expx_gdp
graph export "$pres/Fig_event.pdf", as(pdf) replace


* unemployment
tw 	(conn unemp_weo num if event==1, $m1 $l1 lwidth(thick) lp(solid)) ///
	(conn unemp_weo num if event==3, $m4 $l4 lwidth(thick) lp(solid)) ///
	(conn unemp_weo num if event==2, $m3 $l3 lwidth(thick) lp(dash)) ///
	(conn unemp_weo num if event==4, $mJ $lJ lwidth(thick) lp(solid)) ///
	,xlabel(-${W}(1)${W}, labsize(13pt )) ylabel(,labsize(13pt )) ///
	xtitle("") ///
	ytitle("", size($xs1 ) ) ///
	legend(order(1 "Large debt reductions (N=`N1')" 2 "Advanced (N=`N3')"  3 "Emerging and developing (N=`N2')"  4 "Jamaica (time 0=2013)" ) ///
		rows(3) region(lstyle(none)) fcolor(white%0) size(16pt ) ) ///
	title("Unemployment rate (%)", size(20pt ))  $fig_size name(unemp,replace)
graph export "$pres/Fig_event2.pdf", as(pdf) replace
