// =============================================================================
//
// Load in Data
//
// =============================================================================


/*-------------------------------------------
	world economic outlook
---------------------------------------------*/

* load in
import delim using "$input/WEOOct2023all.csv", clear

* rename
foreach v of var v*{
	local x: variable label `v' 
	rename `v' v`x'
}

ren (weosubjectcode  subjectdescriptor) (sub_code sub_des)
keep iso v* sub_code sub_des units

* keep relevant variables
keep if inlist(sub_code, "GGR_NGDP", "GGX_NGDP", "GGXCNL_NGDP", "GGSB_NPGDP", ///
						 "GGXONLB_NGDP", "GGXWDN_NGDP", "GGXWDG_NGDP", "BCA_NGDPD","NGDP") | inlist(sub_code,"NGDPD","NGDP_D","LUR","NGAP_NPGDP")

* conver tstirng to real
greshape long v@, i(iso sub_code sub_des) j(year)
gen value = real(v)

drop v units sub_des

replace sub_code = lower(sub_code)
drop if value==.
greshape wide value, i(iso year) j(sub_code) string


* rename
ren value* *
ren ngdp   nominal_gdp_lcu
ren ngdpd  nominal_gdp_usd 
ren ngdp_d gdp_def_weo 
ren lur    unemp_weo

ren (*_npgdp iso) (*_ngdp country)


* label variables
label var	ggr_ngdp		"General government revenue to nominal GDP"
label var	ggx_ngdp		"General government total expenditure to nominal GDP"
label var	ggxcnl_ngdp		"General government net lending/borrowing to nominal GDP"
label var	ggsb_ngdp		"General government structural balance to nominal GDP"
label var	ggxonlb_ngdp	"General government primary net lending/borrowing to nominal GDP"
label var	ggxwdn_ngdp		"General government net debt to nominal GDP"
label var	ggxwdg_ngdp		"General government gross debt to nominal GDP"
label var	bca_ngdp		"Current account balance to nominal GDP"
label var	nominal_gdp_lcu		"Nominal GDP, LCU"
label var	nominal_gdp_usd    	"Nominal GDP, USD"
label var	gdp_def_weo    		"GDP Deflator"
label var   unemp_weo           "Unemployment rate"
label var   ngap_ngdp     "Output gap"

* unit convert to percent
foreach v of var *_ngdp{
	replace `v' = `v'/100
}
ren *_ngdp *_gdp

* merge on country name
preserve
	import delim using "$input/WEOOct2023all.csv", clear
	keep iso country
	ren country country_name
	ren iso country 
	duplicates drop
	drop if country==""
	isid country
	tempfile country_names
	save    `country_names'
restore
merge m:1 country using `country_names', keep(1 3) nogen

* save
save "$cleaned/weo.dta", replace


/*-------------------------------------------
	world bank GDP and GDP per capita data
---------------------------------------------*/

* load in
wbopendata, long year(1940:2023) indicator(NY.GDP.PCAP.PP.KD;FP.CPI.TOTL;NE.EXP.GNFS.ZS;NE.IMP.GNFS.ZS;GC.NFN.TOTL.GD.ZS;NE.CON.TOTL.ZS;NY.GDP.DEFL.ZS;NY.GDP.DEFL.ZS.AD;NY.GDP.MKTP.KN;NY.GDP.MKTP.CN;PX.REX.REER;PA.NUS.FCRF;SI.POV.NAHC) clear

* rename
ren ny_gdp_mktp_kn          gdp_const_lcu
ren ny_gdp_mktp_cn          gdp_curr_lcu
ren ny_gdp_defl_zs          gdp_def
ren ny_gdp_defl_zs_ad       gdp_def_linked
ren ny_gdp_pcap_pp_kd 		gdppc_wb
ren fp_cpi_totl 			cpi_wb 
ren ne_exp_gnfs_zs 			ex_gdp   // export to GDP
ren ne_imp_gnfs_zs 			im_gdp   // import to GDP
ren gc_nfn_totl_gd_zs 		inv_gdp  // net investment in nonfin asset
ren ne_con_totl_zs 			cons_gdp // final consumption expenditure
ren px_rex_reer				reer
ren pa_nus_fcrf				er_official
ren si_pov_nahc				poverty_rate
ren countrycode 			country 


* keep the regional level and merge back on country
preserve
	keep if regionname=="Aggregates"
	keep country year gdppc_wb cpi_wb
	ren country adminregion
	ren *_wb *_wb_region
	tempfile region_agg
	save `region_agg'
restore


merge m:1 adminregion year using `region_agg', keep(1 3)
keep country adminregion adminregionname year gdppc_* cpi_* *gdp* er_official reer poverty_rate

* save
save "$cleaned/wb.dta", replace




/*-------------------------------------------
	sovereign default: trebesch haircut
---------------------------------------------*/

// link: https://sites.google.com/site/christophtrebesch/data 
// section: Haircut dataset (creditor losses in sovereign debt restructurings) by Cruces/Trebesch


* load in
import excel using "$input/Cruces-Trebesch-Haircuts-2014-update.xlsx", ///
	sheet("2014 Version (new)") cellrange(B12:U200) clear firstrow
drop if Case==.
drop NewCase CountryCase

* convert from string to value
foreach v of var *{
	if "`v'"!="Code" & "`v'"!="Date" & "`v'"!="Case"{
		gen real_`v' = real(`v')
		drop `v'
		ren real_`v' `v'

	} 
}

* rename
ren DebtRestructu~S debt_restru
ren PreferredHair~Z pref_haircut
ren MarketHaircutHM mkt_haircut
ren ReductinFaceV~e face_val_reduc
ren FaceValueRedu~n face_val_reduc_pct

ren *, lower
ren code country
ren date restru_date

sort country restru_date
keep country debt_restru pref_haircut mkt_haircut face_val_reduc* case *_date

* save
save "$cleaned/trebesch_haircut.dta", replace



/*-------------------------------------------
	sovereign default: trebesch default type
---------------------------------------------*/
// link: https://sites.google.com/site/christophtrebesch/data 
// section: Monthly sovereign default dataset by Asonuma/Trebesch 

* load in
import excel using "$input/Asonuma_Trebesch_DEFAULT_DATABASE.xlsx", ///
	sheet("DATASET Defaults & Restruct.") cellrange(B6:O203) clear firstrow
	
* rename
ren *, lower

ren casenr case_new
ren casenrincruce~b	case
ren wdicode country
ren startofdefaul~n	start_restru_date
ren endofrestruct~n end_restru_date

ren strictlypreem~e strict_preempt
ren weaklypreempt~e weak_premept
ren postdefault     post_default
ren defaultdate     default_date
ren announcement    announce_date

* save
keep case*  *_preempt post_default *_date country *_date
save "$cleaned/trebesch_default.dta", replace

/*-------------------------------------------
sovereign default: merge and update
---------------------------------------------*/

* case that also in 'trebech_haircut'
preserve	
	use "$cleaned/trebesch_default.dta", clear
	keep if case!=.
	tempfile non_missing
	save `non_missing'
restore

* new case relative to trebech_haircut (update from trebech)
preserve
	use "$cleaned/trebesch_default.dta", clear
	keep if case==.
	tempfile missing
	save `missing'
restore

* merge the cases both in default data and haircut data
use "$cleaned/trebesch_haircut.dta",clear
merge 1:1 case using `non_missing', nogen

* update the cases in default data
append using `missing' 

drop case*

foreach v of var *_date{
	gen `v'_year = year(`v')
}

ren *_date_year *_year
drop *_date

* fill in the debt restructure dates
replace end_restru_year = restru_year if end_restru_year==.
drop if start_restru_year==.
drop restru_year

* save
save "$cleaned/trebesch_all.dta", replace


/*-------------------------------------------
 Jamaica FX debt share
---------------------------------------------*/

import excel "$input/JAM_FX.xlsx", sheet("FX_WEO_data") firstrow clear
gen LCshare = 100*LC/(LC + FC)
ren B year
keep year LCshare
gen country="JAM"
save "$cleaned/LCdebt_share_JAM.dta", replace

