program define blmsim
	version 17.0
	syntax, SIMNum(integer) SIMDesc(string) URsim(string) Beveridge(string) Expectations(string) [Expshock(varlist)] [Zshock(varlist)] [Gamma(real 0.945)]
	
	/***************************************************************************
	
	*** Description of blmsim ***
	
	simnum: Number to identify model
	simdesc: Text label to identift model
	ursim: Unemployment rate scenario
		Options:
			SEP: FOMC Sep. 2022 SEP from Ball, Leigh, Mishra
			IMF: IMF July 2022 WEO from Ball Leigh, Mishra
			HigherU:  Higher u given pandemic BC from Ball, Leigh, Mishra
			Target, URATE: Unemployment rate, URATE, in 2023-2024 
	beveridge: Beveridge curve scenario
		Options: 
			remain: BC stays at pandemic BC
			revert: BC gradually returns to pre-pandemic BC
			partial: BC gradually returns 2/3 of the way back to its pre-pandemic BC
	expectations: Inflation expectations scenario
		Options
			revert: Inflation expectations are fully anchored
			drift: Inflation expectations drift based on actual inflation and deanchoring gamma term0
	expshock [Optional]: Variable that specifies user's exogenous shock to inflation expectations
	zshock [Optional]: Variable that specifies user's headline inflation shock
	gamma [Optional]: Gamma value for inflation expectations anchoring 
	
	***************************************************************************/
	
	/***************************************************************************
	* SIMULATION
	***************************************************************************/
	* If a shock (either headline inflation or to expectations) is provided, save that and merge it in
	if "`zshock'" != "" | "`expshock'" !=""{
		keep t `zshock' `expshock'
		tempfile tmp
		save `tmp', replace
	}
	
	use workdata/simdata.dta, clear
	if "`zshock'" != "" | "`expshock'" !=""{
		merge 1:1 t using `tmp', nogen
	}
	local sim = `simnum'
	local name = "`simdesc'"
	
	
	local latest = tm(2022m9)	
	local latestpe = tm(2022m9)
	local latestbc = tm(2022m8)

	***************************************************************
	***** Assumption #1. Unemployment rate path *****
	***************************************************************
	if "`ursim'" == "SEP"{
		* Projections similar to FOMC September 2022 SEP
		replace u = . if t>`latest'
		replace u = 3.8 if t ==tm(2022m11)
		replace u = 4.4 if t ==tm(2023m11)
		replace u = 4.4 if t ==tm(2024m11)
		replace u = 4.3 if t ==tm(2025m11)
	}
	
	else if "`ursim'" == "IMF"{
		* IMF July 2022 WEO from Ball Leigh, Mishra
		replace u = UWEO if t>`latest'
	}
	
	else if "`ursim'" == "HigherU"{
		* Higher u given pandemic BC from Ball, Leigh, Mishra
		replace u = . if t>`latest'
		replace u = 7.5 if t==tm(2023m1)
		replace u = 7.5 if t==tm(2024m12)
		replace u = UWEO if t>=tm(2025m12)
	}

	else if regexm("`ursim'", "Target"){
		* Unemployment Rate for 2023-2024
		parse "`ursim'", parse(",")
		replace u = . if t>`latest'
		replace u = `3' if inrange(t, tm(2023m1), tm(2024m12))
		replace u = 4.5 if t>=tm(2025m12)
	}


	* Interpolate ("int") between missing datapoints
	cap drop uint
	ipolate u t if t>=`latest', generate(uint)
	replace u = uint if t>=`latest' & u==.
	
	***************************************************************
	***** Assumption #2. Headline inflation shocks *****
	***************************************************************
	* Set "z" to zero:
	replace z	= 0	if t>`latest'
	
	* Incorporate user-defined headline shock, if provided
	if "`zshock'" != ""{
		replace z = `zshock' if `zshock' < .
	}
	
	* 12-month average
	replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.


	***************************************************************
	***** Assumption #3. Beveridge Curve (BC) *****
	***************************************************************
	* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC (fully or partially)
	if "`beveridge'" == "remain" {
		local rho		= 1
	}
	else if "`beveridge'" == "revert" | "`beveridge'" == "partial" {
		local rho		= 0.75	
	}
	

	***************************************************************
	***** Assumption #4. Expectations anchoring *****
	***************************************************************
	* Assumed feedback from inflation to change in expected inflation
	* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
	* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
	local gammaQ	= `gamma'
	local gammaM	= ((`gammaQ')^(1/3))

	* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
	* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
	* Or: No return (return = 0) with option to de-anchor (see above).

	if "`expectations'" == "revert" {
		local feedback	= 0
		local return	= .25
	}
	else if "`expectations'" == "drift" {
		local feedback	= 1 - `gammaM'
		local return	= 0
	}
	
	local preCOVID	= 2.2
	
	* Add exogenous expectations change, if provided
	tempvar exogenous_dspe
	gen double `exogenous_dspe' = 0
	if "`expshock'" != ""{
		replace `exogenous_dspe' = `expshock'
	}

	***************************************************************
	* Variables
	***************************************************************

	* Vacancy rate
	* V/U and 12-month average
	g lu	= ln(u)
	g lv	= ln(v)
	label variable lv "Log vacancy rate"
	label variable lu "Log unemployment rate"

	* Extend lv based on BC relation and assumed u path
	* Relation between lv and lu
	reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
	* Extend lv
	predict lvhat
	replace lv	= lvhat if lv==.
	g vhat		= exp(lvhat)

	* Derive v and vu
	replace v = vhat if v==.
	replace vu = v/u if vu==.

	* BC relation between lv and lu pre-COVID
	reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
	* Fitted line
	predict lvhatpre
	g vhatpre		= exp(lvhatpre)
	* Compute path of v based on move from COVID BC to pre-COVID BC
	* Convergence parameter 
	if "`beveridge'" == "remain" | "`beveridge'" == "revert"{
		g vhatgap		= vhat - vhatpre
		g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1	
	}
	
	* If only returning part of way back to pre-COVID BC
	else if "`beveridge'" == "partial"{
		g vhatgap		= (vhat - vhatpre) * 2/3  
		g _vhat	= vhatpre + (vhat - vhatpre) * 1/3 + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1 
	}
	
	replace v		= _vhat if t>=`latest'+1 
	* Recompute V/U
	replace vu = v/u if t>`latest'+1

	* 12-month average of V/U
	cap drop vu_12ma
	g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

	***************************************************************
	* Simulate variables
	***************************************************************

	* Headline inflation
	g headline = HEADLINE if t<=`latest'

	* Phillips curve estimation
	* Define PC variables:
	cap drop y x*
	g y		= CORE-pe
	* Measure of slack
	foreach var in vu	 {
		* Define measure of slack ("x")
		cap drop x*
		g x		= `var'_12ma
		g x2	= (x)^2
		g x3	= (x)^3
		* Compute nonlinear "z" terms
		g z_12ma2 = z_12ma^2
		g z_12ma3 = z_12ma^3	
		* Estimation
		reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
	}

	* Predict inflation gap
	predict yhat
	* Insample predict inflation rate
	g phat = yhat + pe if e(sample)

	* Forecast of future headline inflation
	g headlineforecast	= phat if t<=`latest'

	* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
	g sp	= CORE			if t<=`latest' 
	g spe	= pe			if t<=`latestpe'
	g dspe	= pe - pe[_n-1]	if t<=`latestpe'
	* This requires loop over future dates
	* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
	replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
	forval h = 1/50 {
		replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
		replace sp	= yhat + spe				if t==`latest'+`h'
		replace headline = sp + z				if t==`latest'+`h'
		replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
		replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1]) + `exogenous_dspe'	if t==`latest'+`h'+1
	}	

	* Compute 12-month rates ("a" for annual)
	foreach var in sp headline Median {
		gen index_`var'		= 100 if t==tm(1989m12)
		replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
		gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
	}

	***************************************************************
	* Name and save simulation results
	***************************************************************

	preserve
	rename u su
	rename v sv
	rename vu svu
	rename z sz
	foreach var in sp spa spe headline headlinea su sv svu sz {
		rename `var' `var'`sim'
		label variable `var'`sim' "`simdesc'"
	}
	keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
	save output/sim`sim', replace
	sort t
	restore
	
	end