% PlotCountries.m
%
%   Plot GDP loss and Covid deaths for countries
%   Also Google Activity data versus Covid deaths
%
%   8/29/20: Put cumulative numbers in units of "Percent Years" for
%            easy interpretation

clear all; close all;
diarychad('PlotCountries');

fname='../Data/Data_GDP2020.xlsx';

GDPNames=readmatrix(fname,'Sheet','WithNames','Range','B5:B68','OutputType','char');
GDPCodes=readmatrix(fname,'Sheet','WithNames','Range','A5:A68','OutputType','char');
GDPQ1       =readmatrix(fname,'Sheet','WithNames','Range','G5:G68'); 
GDPQ2       =readmatrix(fname,'Sheet','WithNames','Range','H5:H68'); 
GDPQ2(GDPQ2==-999)=NaN;

load CovidData
CovidNames(TWN)={'Taiwan'};

% Gather the latest deaths per million cumulative
% and put GDP in "Covid data" order
GDPQ1_orig=GDPQ1;
GDPQ2_orig=GDPQ2;
Ncountries=length(GDPNames);
GDPQ1=zeros(size(CovidDeaths,2),1)*NaN; GDPQ2=GDPQ1;
for i=1:Ncountries;
  if exist(GDPCodes{i});
    GDPQ1(eval(GDPCodes{i}))=GDPQ1_orig(i);
    GDPQ2(eval(GDPCodes{i}))=GDPQ2_orig(i);
  end;
  
end;

GDPLevelQ1 = 100*(1+GDPQ1/100);
GDPLevelQ2 = GDPLevelQ1.*(1+GDPQ2/100);
GDPLossQ1  = 100 - GDPLevelQ1;
GDPLossQ2  = 100 - GDPLevelQ2;
%GDPLoss  = GDPLossQ1+GDPLossQ2;
GDPLoss  = (GDPLossQ1+GDPLossQ2)/4;  % Divide by 4 to make units = "Percent-Years"

GDPDeaths=CovidDeathsPerMillion(end,:)';
gdpsmpl=~isnan(GDPQ1.*GDPQ2);

fmt='%10.2f %10.2f %10.1f %10.1f %10.1f%% %10.1f%% %12.0f';
tle='LevelQ1 Q2 LossQ1 Q2 TotalQ TotalA Deaths_pm';
blah=[GDPLevelQ1 GDPLevelQ2 GDPLossQ1 GDPLossQ2 GDPLoss*4 GDPLoss GDPDeaths];
cshow(CovidNames(gdpsmpl),blah(gdpsmpl,:),fmt,tle);

disp ' ';
disp 'NOTE: TotalA is annualized, so units are Percent-Years';
disp '      i.e. a 5 percent cumulative number is the equivalent of';
disp '      losing 5 percent a one-years GDP.'
disp ' ';

definecolors
namethese=[];
figure(1); figsetup; 
plotnamesym2(GDPDeaths,GDPLoss,CovidNames,12,[],.8,.01,namethese,0,myblue*1.1);
chadfig2('COVID deaths per million people','GDP Loss (percent years)',1,0)
makefigwide;
wait
%print PlotCountriesGDP.eps

kept= ~isnan(GDPDeaths.*GDPLoss);
cshow(CovidNames(kept),[GDPDeaths(kept) GDPLoss(kept)],'%10.1f %10.2f','Deaths GDPLoss');

%wait

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Google Activity Data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

load GoogleCountryData;

% The ordering is different in the two data sets
%   - Covid:  CovidCodes   and CovidNames
%   - Google: Country3Codes for 3 letter codes that should match
% So we need to merge on the codes

Tcovid=cell2table(CovidCodes);
Tcovid.CovidCodes(PHL)={'PHI'};    % Fixing Philadelphia duplicate with Philippines
Tcovid.Properties.VariableNames{'CovidCodes'}='Codes';

Tgoogle=table(Country3Codes,KeepCtys,GoogleCountryActivity');
Tgoogle.Properties.VariableNames{'Country3Codes'}='Codes';
[T,ileft,iright]=outerjoin(Tcovid,Tgoogle,'Type','left'); % Does 252, googleorder

% Put back in the original Covid order using ileft
T.ileft=ileft;
Told=T;
T=sortrows(Told,'ileft')

GCountryLevel=T.Var3';
GCountryLevel=-GCountryLevel(2:end,:);
%GCountryCumulative=cumsum(GCountryLevel);
GCountryCumulative=cumsum(GCountryLevel)/12;  % "Percent Years" units
GKeepCtys = T.KeepCtys;
GKeepCtys(isnan(GKeepCtys))=0;

% Three groups of countries to plot
Groups.Small = [USA ITA DEU GBR SWE NOR];
Groups.Big   = [JPN KOR IDN IND TWN MEX PRT AUT BEL ESP FRA DNK CHE];		
Groups.Both  = [Groups.Small Groups.Big];
Groups.Max   = GKeepCtys;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Graph GDPLoss versus Google Activity. How strongly are they correlated?
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure(2); figsetup;
addols=1;
plotnamesym2(GCountryCumulative(end,gdpsmpl)',GDPLoss(gdpsmpl),CovidNames(gdpsmpl),12,[],[],[],namethese,addols,myblue*1.1);
chadfig2('Google Cumulative Reduced Activity','GDP Loss (percent years)',1,0)
makefigwide;
wait
print PlotCountriesGDPGoogle.eps  % Fonts screw up with this -- use png instead
%print PlotCountriesGDPGoogle.png

corr1=corrcoef(packr([GCountryCumulative(end,gdpsmpl)' GDPLoss(gdpsmpl)]));
fprintf('   The correlation is %5.2f\n',corr1(1,2));
wait

%abc



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Google Activity Plots
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% MONTHLY DEATHS
% Gather the latest deaths per million cumulative
% Deaths from the 15th of each month
MonthDates=datetime({'2020-Feb-15','2020-Mar-15','2020-Apr-15','2020-May-15','2020-Jun-15','2020-Jul-15','2020-Aug-15','2020-Sep-15'});
for t=1:length(MonthDates);
  MonthDates_indx(t)=find(datenum(CovidDates)==datenum(MonthDates(t)));
end;
T=length(MonthDates);

MonthlyDeaths=CovidDeathsPerMillion(MonthDates_indx,:);
DeathsLatest =CovidDeathsPerMillion(end,:);

% Google Activity Graphs
MakeKeyPlots(GCountryLevel,MonthlyDeaths,GCountryCumulative,DeathsLatest,CovidNames,Groups,...
	       'Reduced Activity','GoogleCountries_');


disp ' ';
disp 'NOTE: Cumulative Google Activity is annualized, so units are Percent-Years';
disp '      i.e. a 12 percent cumulative number is the equivalent of';
disp '      losing 12 percent a one-years activity.'


diary off

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% %               FUNCTIONS
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% % MakeKeyPlots(FlowEconomy,Deaths,CumulativeEconomy,DeathsLatest,Names,SortOrder,AxisTitle,gname);

%% function []=MakeKeyPlots(FlowEconomy,Deaths,CumulativeEconomy,DeathsLatest,Names,Groups,AxisTitle,gname);

%%   definecolors
%%   namethese=[];
%%   LastMonth=size(FlowEconomy,1);

%%   figure(1); figsetup; 
%%   plotnamesym2(DeathsLatest',FlowEconomy(end,:)',Names,12,[],.8,.01,namethese,0,myblue*1.1);
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   makefigwide;
%%   %wait
%%   print('-depsc',[gname 'Basic']);

%%   % Both Groups, Flow
%%   figure(2); figsetup; 
%%   plotnamesym2(DeathsLatest(Groups.Both)',FlowEconomy(end,Groups.Both)',Names(Groups.Both),12,[],.8,.01,namethese,0,myblue*1.1);
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   makefigwide;
%%   %wait
%%   print('-depsc',[gname 'Both']);
%%   pause(.5)


%%   % Both Groups, Cumulative 
%%   figure(3); figsetup; 
%%   plotnamesym2(DeathsLatest(Groups.Both)',CumulativeEconomy(end,Groups.Both)',Names(Groups.Both),12,[],.8,.01,namethese,0,myblue*1.1);
%%   chadfig2('COVID deaths per million people',['Cumulative ' AxisTitle '(percent years)'],1,0)
%%   makefigwide;
%%   %wait
%%   print('-depsc',[gname 'CumulativeBoth']);
%%   pause(.5)


  
%%   %%%%%%%%%%
%%   % DYNAMICS
%%   %%%%%%%%%%
  
%%   Powersof2=2.^(0:11)';  
%%   StringPowersof2=strmat(' 1 # 2 # 4 # 8 # 16# 32# 64#128#256#512#1024#2048','#');
%%   LW=2;
%%   Transparency=1;
%%   mycolors=[myrainbow(1:6,:); 0.9*myrainbow(1:6,:); 0.8*myrainbow(1:6,:); 0.75*myrainbow(1:6,:)];
%%   mycolors=[mycolors ones(size(mycolors,1),1)*Transparency];
 
%%   months=(1:LastMonth); % Feb Mar Apr May June July August
%%   figure(4); figsetup;
%%   ctys=Groups.Both;
%%   for i=1:length(ctys)
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2(log(1+Deaths(end,ctys(i))'),FlowEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   relabelaxis(log(1+Powersof2),num2str(Powersof2),'x');
%%   print('-depsc',[gname 'Dynamics1']);
%%   pause(.5)


%%   % Drop March as it is always at low deaths, low economic activity...
%%   months=(2:LastMonth); % Mar Apr May June July etc


%%   % Small Select Group (USA, DEU, ITA, GBR, etc)
%%   figure(5); figsetup;
%%   ctys=Groups.Small;
%%   for i=1:length(ctys)
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2(log(1+Deaths(end,ctys(i))'),FlowEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   relabelaxis(log(1+Powersof2),num2str(Powersof2),'x');
%%   print('-depsc',[gname 'Dynamics2']);
%%   pause(.5)


%%   % Big group of "Middle" countries (JPN, BEL, MEX, KOR,etc)
%%   figure(6); figsetup;
%%   ctys=Groups.Big;
%%   for i=1:length(ctys)  
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot(log(1+Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2(log(1+Deaths(end,ctys(i))'),FlowEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   relabelaxis(log(1+Powersof2),num2str(Powersof2),'x');
%%   print('-depsc',[gname 'Dynamics2b']);
%%   pause(.5)


%%   % Both Groups
%%   figure(7); figsetup;
%%   ctys=Groups.Both;
%%   for i=1:length(ctys);
%%     plot((Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot((Deaths(months,ctys(i))),FlowEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2((Deaths(end,ctys(i))'),FlowEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',[AxisTitle ' (percent)'],1,0)
%%   print('-depsc',[gname 'Dynamics3']);
%%   pause(.5)


%%   % Small top 5
%%   figure(8); figsetup;
%%   ctys=Groups.Small;
%%   for i=1:length(ctys)
%%     plot(log(1+Deaths(months,ctys(i))),CumulativeEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot(log(1+Deaths(months,ctys(i))),CumulativeEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2(log(1+Deaths(end,ctys(i))'),CumulativeEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',['Cumulative ' AxisTitle '(percent years)'],1,0)
%%   relabelaxis(log(1+Powersof2),num2str(Powersof2),'x');
%%   print('-depsc',[gname 'Dynamics2c5']);
%%   pause(.5)


  
%%   % Big middle set
%%   ctys=Groups.Big;
%%   figure(9); figsetup;
%%   for i=1:length(ctys)
%%     plot(log(1+Deaths(months,ctys(i))),CumulativeEconomy(months,ctys(i)),'-','Color',mycolors(i,:),'LineWidth',LW);
%%     plot(log(1+Deaths(months,ctys(i))),CumulativeEconomy(months,ctys(i)),'o','Color',mycolors(i,:),'LineWidth',LW);
%%     plotnamesym2(log(1+Deaths(end,ctys(i))'),CumulativeEconomy(end,ctys(i)),Names(ctys(i)),12,[],.8,.01,[],0,myblue*1.1);
%%   end;
%%   chadfig2('COVID deaths per million people',['Cumulative ' AxisTitle '(percent years)'],1,0)
%%   relabelaxis(log(1+Powersof2),num2str(Powersof2),'x');
%%   print('-depsc',[gname 'Dynamics2c15']);

  
%% end

