% PlotCities.m
%
%   Cities: Google Activity data versus Covid deaths
%
%   8/29/20: Put cumulative numbers in units of "Percent Years" for
%            easy interpretation

clear all; close all;
diarychad('PlotCities');


load CovidData
CovidNames(TWN)={'Taiwan'};

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Google Activity Data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

load GoogleCityData;

% The ordering is different in the two data sets
%   - Covid:  CovidCodes   and CovidNames
%   - Google: GoogleCityCodes for 3 letter codes that should match
% So we need to merge on the codes

Tcovid=cell2table(CovidCodes);
%Tcovid.CovidCodes(PHL)={'PHI'};    % Fixing Philadelphia duplicate with Philippines
Tcovid.Properties.VariableNames{'CovidCodes'}='Codes';

Tgoogle=table(GoogleCityCodes,GoogleCityActivity');
Tgoogle.Properties.VariableNames{'GoogleCityCodes'}='Codes';
[T,ileft,iright]=outerjoin(Tcovid,Tgoogle,'Type','left'); % Does 252, googleorder

% Put back in the original Covid order using ileft
T.ileft=ileft;
Told=T;
T=sortrows(Told,'ileft')

GCityLevel=T.Var2';
GCityLevel=-GCityLevel(2:end,:);
%GCityCumulative=cumsum(GCityLevel);
GCityCumulative=cumsum(GCityLevel)/12; % "Percent-Years" units for cumulative

% Three groups of countries to plot
Groups.Small = [NYC LOM LON PIF MAD STK TOK SEO];
Groups.Big   = [BOS LAX SF MIA CHI HOU]
Groups.Both  = [Groups.Small Groups.Big];

CovidNames{BOS}='Boston';
CovidNames{HOU}='Houston';
CovidNames{NYC}='New York City';
CovidNames{LON}='London';
CovidNames{STK}='Stockholm';
CovidNames{PIF}='Paris';


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Google Activity Plots
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% MONTHLY DEATHS
% Gather the latest deaths per million cumulative
% Deaths from the 15th of each month
MonthDates=datetime({'2020-Feb-15','2020-Mar-15','2020-Apr-15','2020-May-15','2020-Jun-15','2020-Jul-15','2020-Aug-15','2020-Sep-15'});
for t=1:length(MonthDates);
  MonthDates_indx(t)=find(datenum(CovidDates)==datenum(MonthDates(t)));
end;
T=length(MonthDates);

MonthlyDeaths=CovidDeathsPerMillion(MonthDates_indx,:);
DeathsLatest =CovidDeathsPerMillion(end,:);

% Google Activity Graphs
MakeKeyPlots(GCityLevel,MonthlyDeaths,GCityCumulative,DeathsLatest,CovidNames,Groups,...
	       'Reduced Activity','GoogleCities_');

disp ' ';
disp 'NOTE: Cumulative Google Activity is annualized, so units are Percent-Years';
disp '      i.e. a 12 percent cumulative number is the equivalent of';
disp '      losing 12 percent a one-years activity.'


diary off
