********************************************************************************
********************************************************************************
********************************************************************************
* This is the master Stata file that replicates the empirical results in
* "COVID-19 is Also a Reallocation Shock," by Jose Maria Barrero, Nick Bloom,
* and Steven J. Davis
*
* July 2020
********************************************************************************
********************************************************************************
********************************************************************************


********************************************************************************
********************************************************************************
********************************************************************************
* Preliminaries
********************************************************************************
********************************************************************************
********************************************************************************

version 16
clear all

********************************************************************************
********************************************************************************
********************************************************************************
* Set working directories using global variables. 
********************************************************************************
* THIS IS THE ONLY SECTION OF CODE WHERE USERS MUST MAKE CHANGES TO THE CODE
********************************************************************************
* We use two directories here, one for the confidential microdata from the
* Survey of Business Uncertianty, and another for the public use microdata.
*
* (Researchers may request access to the confidential data files directly with 
*  the Federal Reserve Bank of Atlanta, subject to to signing an NDA).
********************************************************************************
********************************************************************************
********************************************************************************

global NONCONFIDENTIALDATALOC =  // Directory with the non-confidential data
global CONFIDENTIALDATALOC = // Directory with the confidential data

global WD = 
// Directory where output and intermediate files will be saved

cd $WD

********************************************************************************
********************************************************************************
********************************************************************************
* TABLE 1 -- Gross Staffing Changes in Reaction to the COVID-19 Pandemic
********************************************************************************
********************************************************************************
********************************************************************************

	use "$CONFIDENTIALDATALOC/COVIDstaffingchanges.dta", clear

	local quitsperlayoff = 0.2314
	
	foreach i in 1 2 12 {
		
		gen covid_impact_emp`i'_grossred_perq = covid_impact_emp`i'_grossred_per*(1 + `quitsperlayoff')
		gen quits`i' = covid_impact_emp`i'_grossred_per*(`quitsperlayoff')
		gen covid_impact_emp`i'_overall_perq = covid_impact_emp`i'_overall_per - quits`i'
	}
	
	****************************************************************************
	* Net Staffing Changes, Exclusive of Quits
	mean covid_impact_emp1_overall_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp2_overall_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp12_overall_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	
	
	* Net Staffing Changes, with Imputed Quits
	mean covid_impact_emp1_overall_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp2_overall_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp12_overall_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	
	****************************************************************************	
	* Gross Staffing Reductions, Exclusive of Quits
	mean covid_impact_emp1_grossred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp2_grossred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp12_grossred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	
	* Gross Staffing Reductions, With Imputed Quits
	mean covid_impact_emp1_grossred_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp2_grossred_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp12_grossred_perq if date==ym(2020,4)  [aw=emp_current_march1_w]
	
	
		* Permanent layoffs
		mean covid_impact_emp1_permlayoff_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp2_permlayoff_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp12_permlay_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		
		* Temporary layoffs and furloughs
		mean covid_impact_emp1_templayoff_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp2_templayoff_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp12_templay_per if date==ym(2020,4)  [aw=emp_current_march1_w]

		
		* Cuts in contractors and leased workers
		mean covid_impact_emp1_contcut_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp2_contcut_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp12_contcut_per if date==ym(2020,4)  [aw=emp_current_march1_w]

		* Imputed Quits
		mean quits1 if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean quits2 if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean quits12 if date==ym(2020,4)  [aw=emp_current_march1_w]
		
	****************************************************************************
	* Gross Staffing Increases
	mean covid_impact_emp1_addred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp2_addred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	mean covid_impact_emp12_addred_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		
	
		* Hires of new employees
		mean covid_impact_emp1_empadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp2_empadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp12_empadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
	
		* Additions to contractors and leased workers
		mean covid_impact_emp1_contadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp2_contadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		mean covid_impact_emp12_contadded_per if date==ym(2020,4)  [aw=emp_current_march1_w]
		
		
********************************************************************************
********************************************************************************
********************************************************************************
* Figure 1 -- Weekly Count of High-Propensity Business Applications in 2020 and 
*             Percent Change Relative to the Same Week in 2019
*
* The Figure in the paper is created using an Excel file. The non-confidential
* data package includes a file with the processed data and figure.
* The code below infiles the raw, publicly-available Census data and exports the
* data to an Excel file for plotting.
********************************************************************************
********************************************************************************
********************************************************************************

clear
import delimited using "$NONCONFIDENTIALDATALOC/bfs_us_apps_weekly_nsa_Jul162020update.csv", varnames(1)

gen dateyw = yw(year,week)
format date %tw

tsset dateyw

label variable ba_nsa  "Business Applications, not seasonally adjusted"
label variable hba_nsa "High-Propensity Business Applications, not seasonally adjusted"
label variable wba_nsa "Business Applications with Planned Wages, not seasonally adjusted"
label variable cba_nsa "Business Applications from Corporations, not seasonally adjusted"

label variable yy_ba_nsa  "YOY % Change in Business Applications, not seasonally adjusted"
label variable yy_hba_nsa "YOY % Change in High-Propensity Business Applications, not seasonally adjusted"
label variable yy_wba_nsa "YOY % Change in Business Applications with Planned Wages, not seasonally adjusted"
label variable yy_cba_nsa "YOY % Change in Business Applications from Corporations, not seasonally adjusted"

gen date_daily = dofw(dateyw)
replace date_daily = date_daily + 3
format date_daily %td
label variable date_daily "Week Ending"

keep if date_daily>=mdy(1,11,2020) & date_daily~=.

tostring date_daily, gen(date_daily_string) force  format(%td)
label variable date_daily_string "Week Ending"

export excel hba_nsa yy_hba_nsa date_daily date_daily_string using "$WD/Figure1Data.xlsx", replace firstrow(varlabels)

			

********************************************************************************
********************************************************************************
********************************************************************************
* TABLE 2, and Figure 2 and Figure C.1
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
* Collapse the confidential firm level data into the time series of expected 
* growth rates and expected excess reallocation rates.
*
* We need to create the time series of sales and employment separately because
* we need to use employment/sales-specific activity weights when aggregating the
* firm-level microdata

	****************************************************************************
	* First, infile the microdata file, which contains expected employment and 
	* sales growth rates (arc-percentage changes) as well as job and sales 
	* creation and destruction (i.e. the absolute value of expected job and sales 
	* growth rates).
	
	use "$CONFIDENTIALDATALOC/SBU_forrealloc6.2020.dta", clear
	
	****************************************************************************
	* Create the employment expectations and reallocation series

	count if emp_grmean~=. & emp_weight_forward_alt_w~=. & date>=ym(2020,4) & date<=ym(2020,6)

	
	collapse (mean) emp_grealloc emp_grmean [aw=emp_weight_forward_alt_w], by(date)
	
	
	gen emp_egrealloc = emp_grealloc - abs(emp_grmean) //excess reallocation (across firms) = aggregate gross creation/destruction - net employment growth
	
	* Translate into percentage terms
	foreach var in emp_grmean emp_egrealloc {
		gen `var'_pcent = `var'*100 
	}
	
	
	label variable emp_grealloc "Job Reallocation Rate"
	label variable emp_egrealloc "Excess Job Reallocation Rate"
	label variable emp_egrealloc_pcent "Excess Job Reallocation Rate (%)"
	label variable emp_grmean "Expected Employment Growth Rate"
	label variable emp_grmean_pcent "Expected Employment Growth Rate (%)"
	
	* Save
	save "$WD/EmpReallocforFigures.dta", replace
	
	
	****************************************************************************
	* Repeat to create the sales expectations and reallocation series.
	
	use "$CONFIDENTIALDATALOC/SBU_forrealloc6.2020.dta", clear
	
	count if sr_grealloc~=. & sr_weight_forward_w~=. & date>=ym(2020,4) & date<=ym(2020,6)
	
	collapse (mean) sr_grealloc sr_gr_dhsforecast [aw=sr_weight_forward_w], by(date)
	
	gen sr_egrealloc = sr_grealloc - abs(sr_gr_dhsforecast)
	
	foreach var in sr_gr_dhsforecast sr_egrealloc {
		gen `var'_pcent = `var'*100 
	}
	
	label variable sr_grealloc "Reallocation Rate for Sales Revenue"
	label variable sr_egrealloc "Excess Reallocation Rate for Sales Revenue"
	label variable sr_egrealloc_pcent "Excess Reallocation Rate for Sales Revenue (%)"
	label variable sr_gr_dhsforecast "Expected Sales Revenue Growth Rate"
	label variable sr_gr_dhsforecast_pcent "Expected Sales Revenue Growth Rate (%)"
	
	save "$WD/SalesReallocforFigures.dta", replace
	
	****************************************************************************
	* Merge  the files containing the employment and sales series
	
	use "$WD/EmpReallocforFigures.dta", clear
	
	merge 1:1 date using "$WD/SalesReallocforFigures.dta"
	
	save "$WD/EmpSaleReallocforFigures.dta", replace
	
	****************************************************************************
	* TABLE 2 --Average Values of Expected Growth Rates and Expected Excess 
	*           Reallocation Rates from September 2016 to January 2020
	
	use "$WD/EmpSaleReallocforFigures.dta", clear
	
		* Row 1: September 2016 to January 2020
			* Expected Growth Rates
			mean sr_gr_dhsforecast_pcent emp_grmean_pcent  if date>=ym(2016,9) & date<=ym(2020,1)
	
			* Expected Excess Reallocation Rates
			mean sr_egrealloc_pcent emp_egrealloc_pcent  if date>=ym(2016,9) & date<=ym(2020,1)

		* Row 2: September 2016 to January 2020

			* Expected Growth Rates
			mean sr_gr_dhsforecast_pcent emp_grmean_pcent  if date>=ym(2020,4) & date<=ym(2020,6)
	
			* Expected Excess Reallocation Rates
			mean sr_egrealloc_pcent emp_egrealloc_pcent  if date>=ym(2020,4) & date<=ym(2020,6)
			
			
	****************************************************************************
	* Figure 2
	* Expected Rates of Excess Sales and Job Reallocation at a One-Year Forecast 
	* Horizon (12 Months for Employment, 4 quarters for Sales)
	* September 2016 to June 2020
	
	use "$WD/EmpSaleReallocforFigures.dta", clear
	
	twoway (tsline emp_egrealloc_pcent, lwidth(medthick)  lcolor(navy) ytitle("") ylabel(-1(1)6.8, axis(1) grid gstyle(major)) yaxis(1)) (tsline sr_egrealloc_pcent, yaxis(2) lcolor(maroon) lpattern(dash) lwidth(medthick) ytitle("", axis(2)) ylabel(-1(1)6.55, axis(2))), yline(0, lcolor(gs10)) legend( label(1 "Expected Excess Job Reallocation Rate") label(2 "Expected Excess Reallocation Rate for Sales Revenue") cols(1)  position(6) ring(0) region(lstyle(none) color(none%0) )) tlabel(2016m9(4)2020m5, angle(45) format(%tmNN/yy)) xmtick(680(1)725)  xtitle("") title("Percent of Employment (left) or Sales (right)") plotregion(lstyle(none))		
		
	****************************************************************************
	* Figure C.1
	* Expected Employment and Sales Revenue Growth Rates at One-Year Forecast
    * Horizons (12 Months for Employment, 4 quarters for Sales)
	* September 2016 to June 2020
	
	use "$WD/EmpSaleReallocforFigures.dta", clear

	twoway (tsline emp_grmean_pcent, lwidth(medthick)  lcolor(navy) ytitle("") ylabel(-1(1)3.5, axis(1) grid gstyle(major)) yaxis(1)) (tsline sr_gr_dhsforecast_pcent, yaxis(2) lcolor(maroon) lpattern(dash) lwidth(medthick) ytitle("", axis(2)) ylabel(-4(2)6, axis(2))), yline(0, lcolor(gs10)) legend( label(1 "Expected Employment Growth Rate") label(2 "Expected Sales Revenue Growth Race") cols(1)  position(6) ring(0) region(lstyle(none) color(none%0) )) tlabel(2016m9(4)2020m5, angle(45) format(%tmNN/yy)) xmtick(680(1)725)  xtitle("") title("Percent of Employment (left) or Sales (right)") plotregion(lstyle(none))	
	
	*****************************************************************************
	* Exporting the data for the above Figures to Excel, with one tab for each of 
	* Figures 2 and C.1
	use "$WD/EmpSaleReallocforFigures.dta", clear

	keep date emp_grmean_pcent emp_egrealloc_pcent sr_gr_dhsforecast_pcent sr_egrealloc_pcent
	label variable date Date
	label variable emp_grmean_pcent "Expected Employment Growth Rate"
	label variable emp_egrealloc_pcent "Expected Excess Job Reallocation Rate"
	label variable sr_gr_dhsforecast_pcent "Expected Sales Revenue Growth Rate"
	label variable sr_egrealloc_pcent "Expected Excess Reallocation Rate for Sales Revenue"
	export excel date emp_egrealloc_pcent sr_egrealloc_pcent using  "$WD/Figures2andC1Data.xlsx", sheet("Figure 2") firstrow(varlabels) replace
	export excel date emp_grmean_pcent sr_gr_dhsforecast_pcent using "$WD/Figures2andC1Data.xlsx", sheet("Figure C1", modify) firstrow (varlabels)

********************************************************************************
********************************************************************************
********************************************************************************
* Figure 3 -- Working from Home Accounts for More Than 60 Percent of U.S. Labor 
*             Services Supplied in May 2020
********************************************************************************
********************************************************************************
********************************************************************************

use "$NONCONFIDENTIALDATALOC/WFHquestionprodata.dta", clear

graph hbar (sum) cratio100 icratio100, over(workstatus_current, sort( (sum) cratio) descending) blabel(bar, format(%12.1f)) ytitle("Percent of wages by May 2020 work situation") legend(label(1 Unweighted) label(2 Earnings Weighted) span) bar(1, color(black) fcolor(black)) bar(2, color(red) fcolor(red) fintensity(70)) ylabel(0(10)55)


********************************************************************************
********************************************************************************
********************************************************************************
* TABLE 3 -- Working from Home before and after the COVID-19 Pandemic
********************************************************************************
********************************************************************************
********************************************************************************
use "$CONFIDENTIALDATALOC/WFHprepostCOVID.dta", clear

* Row 1 (Pre-COVID)
mean wfh_precovid_rarelynever   [aw=empwt]
mean wfh_precovid_1dayweek   [aw=empwt]
mean wfh_precovid_2to4dayweek   [aw=empwt]
mean wfh_precovid_5dayweek   [aw=empwt]
mean wfhdays_precovid_frac [aw=empwt]

* Row 2 (Planned Post-COVID)
mean wfh_postcovid_rarelynever   [aw=empwt]
mean wfh_postcovid_1dayweek   [aw=empwt]
mean wfh_postcovid_2to4dayweek   [aw=empwt]
mean wfh_postcovid_5dayweek   [aw=empwt]
mean wfhdays_postcovid_frac [aw=empwt]

* Row 3 (ATUS 2017-2018) - From BLS website (see paper)


********************************************************************************
********************************************************************************
********************************************************************************
* TABLE 4 -- Working from Home before and after the COVID-19 Pandemic by Major 
*            Industry Sector
********************************************************************************
********************************************************************************
********************************************************************************

use "$CONFIDENTIALDATALOC/WFHprepostCOVID.dta", clear

* Column 1 (Pre-COVID)
mean wfhdays_precovid_frac [aw=empwt]
mean  wfhdays_precovid_frac  [aw=empwt], over(naics1)

* Column 2 (Planned Post-COVID)
mean wfhdays_postcovid_frac [aw=empwt]
mean  wfhdays_postcovid_frac [aw=empwt], over(naics1)


********************************************************************************
********************************************************************************
********************************************************************************
* Figure 4
* The Dispersion of Firm-Level Stock Returns, January 1984 to June 2020
********************************************************************************
********************************************************************************
********************************************************************************


use "$NONCONFIDENTIALDATALOC/equity_monthly.dta", clear

xtset sec mdate 

//*******// CREATE A VARIABLE FOR WEIGHTS BASED ON MARKET VALUE
gen emv_w = (emv_last+l.emv_last)/2
replace emv_w = (emv_first+emv_last)/2 if emv_w == .

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)

//*******// COLLAPSE TO MONTHLY TIME SERIES

collapse ///
(mean) market_return er_raw er_raw_w  ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
[aw=emv_w], by(mdate)

save "$WD/er_returns_monthly", replace



// NOW REPEAT THE COLLAPSE FOR THE "FEB-24 - Mar 31" PERIOD

use "$NONCONFIDENTIALDATALOC/equity_daily_Feb24_Jul14.dta", clear

drop if date>=td(31mar2020)

collapse ///
(sum) market_return er_raw ///
(first) emv_first = emv ///
(last) emv_last = emv ///
, by(security_id)

//******* CREATE A VARIABLE FOR WEIGHTS
gen emv_w = (emv_first+emv_last)/2

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)


collapse ///
(mean) er_raw er_raw_w  ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
[aw=emv_w]

foreach var in er_raw er_iqr er_sd {	
	rename `var' `var'_F24M21	
}

gen mdate = ym(2020,3)
format mdate %tm
keep mdate *_F24M21

merge 1:1 mdate using "$WD/er_returns_monthly.dta", nogen
sort mdate

tsset mdate
order mdate
save "$WD/equity_monthly_plusF24Mar21.dta", replace	

******************************************************************************** 
* Figure 4A
// IQR Value-weighted Equity Returns
********************************************************************************

use "$WD/equity_monthly_plusF24Mar21.dta", clear

drop if mdate == ym(2020,7)

twoway (tsline er_iqr, lcolor(navy) ytitle(IQR of Value-weighted Equity Returns, size(medlarge))) (scatter er_iqr_F24M21 mdate if er_iqr_F24M21~=., msize(large) yaxis(1) mfcolor(navy) mlcolor(navy)) if mdate >= ym(1984,1), xtitle(Month) ttick(1984m1(12)2020m4) legend(off) //legend(label(1 IQR of Value-weighted Equity Returns)  label(2 IQR Value-weighted Equity Returns February 24 - March 21 2020) cols(1))

********************************************************************************
* Figure 4B
// SD Value-weighted Equity Returns
********************************************************************************

use "$WD/equity_monthly_plusF24Mar21.dta", clear

drop if mdate == ym(2020,7)

// SD Value-weighted Equity Returns
twoway (tsline er_sd, lcolor(navy) ytitle(SD of Value-weighted Equity Returns, size(medlarge))) (scatter er_sd_F24M21 mdate if er_sd_F24M21~=., msize(large) yaxis(1) mfcolor(navy) mlcolor(navy)) if mdate >= ym(1984,1),  xtitle(Month) ttick(1984m1(12)2020m6) legend(off) //legend(label(1 SD Value-weighted Equity Returns)  label(2 SD of Value-weighted Equity Returns February 24 - Mar 21 2020)  cols(1))



********************************************************************************
********************************************************************************
********************************************************************************
* Appendix A
*
* The Excel File "Inferring Permanent Job Losses from CPS data.xlsx" goes through
* the calculations in Appendix A step-by-step and in detail.
*
********************************************************************************
********************************************************************************
********************************************************************************


********************************************************************************
********************************************************************************
********************************************************************************
* Appendix B
*
* The Excel File "Calculations on JOLTS Data" provided contains our calculations 
* of the permanent layoff share using JOLTS data and reproduces Table B.1
*
* Supporting calculations for the statistics about initial California claims
* can be found in Excel file "California Initial Claims.xlsx"
*
********************************************************************************
********************************************************************************
********************************************************************************


********************************************************************************
********************************************************************************
********************************************************************************
*Appendix C -- Additional Material
*
********************************************************************************
********************************************************************************
********************************************************************************


********************************************************************************
* TABLE C.1 -- Anticipated Coronavirus Impact on 2020 Sales Revenue, Percentage
*            Amounts
********************************************************************************

use "$CONFIDENTIALDATALOC/COVID19salesimpact42020.dta", clear

mean covid_impact_allalt [aw = sr_current_forweights] 
mean covid_impact_allalt [aw = srwt]
disp(`e(N)')


********************************************************************************
* Figure C.2 - Four-month equity returns
********************************************************************************

use "$NONCONFIDENTIALDATALOC/equity_monthly.dta", clear

xtset sec mdate 

gen dateymd = dofm(mdate)
format dateymd %td

gen month = month(dateymd)
gen year  = year(dateymd)

gen season = 1 if month==11|month==12|month==1|month==2
replace season = 2 if month==3|month==4|month==5|month==6
replace season = 3 if month==7|month==8|month==9|month==10

gen seasony = year + (month==11|month==12) + round((season - 1)/3,.01)

gen lemv_last = l.emv_last

// *******// Collapse to quarterly
collapse (sum) market_return er_raw (firstnm) lemv_last emv_first (lastnm) emv_last month year, by(security_id seasony)

//*******// CREATE A VARIABLE FOR WEIGHTS BASED ON MARKET VALUE
gen emv_w = (emv_last+lemv_last)/2
replace emv_w = (emv_first+emv_last)/2 if emv_w == .

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)

//*******// CALCULATE MONTHLY: REALLOCATION, EXCESS REALLOCATION RATES, AND IQR OF EQUITY RETURNS

gen realloc = abs(er_raw)
gen realloc_w = abs(er_raw_w)

collapse ///
(mean) market_return er_raw realloc er_raw_w  realloc_w ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
(lastnm) month year ///
[aw=emv_w], by(seasony)

gen mdate=ym(year,month)
tsset mdate
format mdate %tm

drop if mdate>=ym(2020,7)

gen erealloc = realloc - abs(er_raw)
gen erealloc_w = realloc_w - abs(er_raw_w)

save "$WD/er_returns_fourmonth", replace


// NOW REPEAT THE COLLAPSE FOR THE "FEB-24 - Mar 31" PERIOD
use "$NONCONFIDENTIALDATALOC/equity_daily_Feb24_Jul14.dta", clear

drop if date>=td(25jun2020)

collapse ///
(sum) market_return er_raw ///er_est er_abnormal 
(first) emv_first = emv ///
(last) emv_last = emv ///
, by(security_id)

//******* CREATE A VARIABLE FOR WEIGHTS
gen emv_w = (emv_first+emv_last)/2

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)

gen realloc = abs(er_raw)
gen realloc_w = abs(er_raw_w)

collapse ///
(mean) er_raw realloc er_raw_w realloc_w ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
[aw=emv_w]

gen erealloc = realloc - abs(er_raw)
gen erealloc_w = realloc_w - abs(er_raw_w)


foreach var in er_raw realloc erealloc er_iqr er_sd {	
	rename `var' `var'_F24J24	
}

gen mdate = ym(2020,6)
format mdate %tm
keep mdate *_F24J24

merge 1:1 mdate using "$WD/er_returns_fourmonth", nogen
sort mdate

tsset mdate
order mdate
save "$WD/equity_fourmonth_plusF24Jun24", replace	


* Figure C2.A
// IQR Value-weighted Four-month Equity Returns

use "$WD/equity_fourmonth_plusF24Jun24.dta", clear

twoway (tsline er_iqr, lcolor(navy) ytitle(IQR of Value-weighted Equity Returns, size(medlarge))) (scatter er_iqr_F24J24 mdate if er_iqr_F24J24~=., msize(large) yaxis(1) mfcolor(navy) mlcolor(navy)) if mdate >= ym(1984,1), xtitle(Month) ttick(1984m1(12)2020m6) legend(off) //legend(label(1 IQR of Value-weighted Equity Returns)  label(2 IQR Value-weighted Equity Returns February 24 - March 21 2020) cols(1))


* Figure C.2B
// SD Value-weighted Four-month Equity Returns
use "$WD/equity_fourmonth_plusF24Jun24.dta", clear

// SD Value-weighted Equity Returns
twoway (tsline er_sd, lcolor(navy) ytitle(SD of Value-weighted Equity Returns, size(medlarge))) (scatter er_sd_F24J24 mdate if er_sd_F24J24~=., msize(large) yaxis(1) mfcolor(navy) mlcolor(navy)) if mdate >= ym(1984,1),  xtitle(Month) ttick(1984m1(12)2020m6) legend(off) //legend(label(1 SD Value-weighted Equity Returns)  label(2 SD of Value-weighted Equity Returns February 24 - Mar 21 2020)  cols(1))


			

********************************************************************************
* Table C.2A - Equity Returns by 2-digit SIC
********************************************************************************

use "$NONCONFIDENTIALDATALOC/equity_monthly.dta", clear

xtset sec mdate 

gen dateymd = dofm(mdate)
format dateymd %td

gen month = month(dateymd)
gen year  = year(dateymd)

gen season = 1 if month==11|month==12|month==1|month==2
replace season = 2 if month==3|month==4|month==5|month==6
replace season = 3 if month==7|month==8|month==9|month==10

gen seasony = year + (month==11|month==12) + round((season - 1)/3,.01)

gen lemv_last = l.emv_last

keep if seasony>2020 & seasony<2020.5


// *******// Collapse to quarterly
collapse (sum) market_return er_raw (firstnm) lemv_last emv_first (lastnm) emv_last month year, by(security_id seasony sic naics)

//*******// CREATE A VARIABLE FOR WEIGHTS BASED ON MARKET VALUE
gen emv_w = (emv_last+lemv_last)/2
replace emv_w = (emv_first+emv_last)/2 if emv_w == .

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)

//*******// CALCULATE MONTHLY: REALLOCATION, EXCESS REALLOCATION RATES, AND IQR OF EQUITY RETURNS

gen realloc = abs(er_raw)
gen realloc_w = abs(er_raw_w)


*gen sic2 = floor(sic/100)
tostring sic, replace
gen sic2 = substr(sic,1,2) if strlen(sic)==4
replace sic2 = "0" + substr(sic,1,1) if strlen(sic)==3
gen naics2 = floor(naics/10000)

egen nfirms = total(1), by(sic2)

collapse ///
(mean) market_return er_raw realloc er_raw_w  realloc_w nfirms ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
(lastnm) month year ///
[aw=emv_w], by(sic2)

sort er_raw,

gen sicname = sic
destring sicname, replace

label define sic_lbl 83 "Social Services" 12 "Coal Mining" 99 "Unclassifiable (Conglomerates)" 79 "Amusement and Recreation Services" 46 "Pipelines Except natural Gas" 70 "Hotels, Rooming Houses, Camps, and Other Lodging Places" 13 "Oil and Gas" 65 "Real Estate" 7 "Agricultural Services" 56 "Apparel and Accesory Stores" 57 "Home Furniture, Furnishings, And Equipment Stores" 73 "Business Services" 39 "Miscellaneous Manufacturing Industries" 52 "Building Materials, Hardware, Garden Supply, and Mobile Home Dealers" 78 "Motion Pictures" 36 "Electronic and Other Electrical Equipment and Components, Except Computer Equipment" 42 "Motor Freight Transportation and Warehousing" 2 "Agriculture production livestock and animal specialties" 10 "Metal Mining" 59 "Miscellaneous Retail"

label values sicname sic_lbl


keep if _n<=10 | _n>=(_N-9)

keep sic2 sicname er_raw nfirms

label variable sic2 "2-Digit SIC"
label variable sicname "Sector Name"
label variable er_raw "Mean Value-Weighted Return"
label variable nfirms "No of Firms"

export excel sic2 sicname er_raw nfirms using  "$WD/TableC2.xlsx", sheet("Table C2A") firstrow(varlabels) replace


********************************************************************************
* Table C.2B Equity returns by 3-digit NAICS	
********************************************************************************

use "$NONCONFIDENTIALDATALOC/equity_monthly.dta", clear

xtset sec mdate 

gen dateymd = dofm(mdate)
format dateymd %td

gen month = month(dateymd)
gen year  = year(dateymd)

gen season = 1 if month==11|month==12|month==1|month==2
replace season = 2 if month==3|month==4|month==5|month==6
replace season = 3 if month==7|month==8|month==9|month==10

gen seasony = year + (month==11|month==12) + round((season - 1)/3,.01)

gen lemv_last = l.emv_last

keep if seasony>2020 & seasony<2020.5


// *******// Collapse to quarterly
collapse (sum) market_return er_raw (firstnm) lemv_last emv_first (lastnm) emv_last month year, by(security_id seasony sic naics)

//*******// CREATE A VARIABLE FOR WEIGHTS BASED ON MARKET VALUE
gen emv_w = (emv_last+lemv_last)/2
replace emv_w = (emv_first+emv_last)/2 if emv_w == .

winsor2 er_raw emv_w, cuts(1 99) suffix(_w)

//*******// CALCULATE MONTHLY: REALLOCATION, EXCESS REALLOCATION RATES, AND IQR OF EQUITY RETURNS

gen realloc = abs(er_raw)
gen realloc_w = abs(er_raw_w)

gen sic2 = floor(sic/100)
tostring naics, replace
gen naics3 = substr(naics,1,3)

egen nfirms = total(1), by(naics3)


collapse ///
(mean) market_return er_raw realloc er_raw_w  realloc_w nfirms ///
(iqr) er_iqr = er_raw ///
(sd) er_sd = er_raw ///
(lastnm) month year ///
[aw=emv_w], by(naics3)

drop if naics3=="."

gen naicsname = naics
destring naicsname, replace

label define naics_lbl 525 "Funds, Trusts, and Other Financial Vehicles" 999 "Unclassifiable (Conglomerates)" 512 "Motion Picture and Sound Recording Industries" 323 "Printing and Related Support Activities" 481 "Air Transportation" 711 "Performing Arts, Spectator Sports, and Related Industries" 721 "Accommodation" 622 "Hospitals" 623 "Nursing and Residential Care Facilities" 213 "Support Activities for Mining" 326 "Plastics and Rubber Manufacturing" 334 "Computer and Electronic Product Manufacturing" 451 "Sporting Goods, Hobby, Musical Instrument, and Book Stores" 519 "Other Information Services" 511 "Publishing Industries" 484 "Truck Transportation" 112 "Animal Production and Aquaculture" 454 "Nonstore Retailers" 453 "Miscellaneous Store Retailers" 811 "Repair and Maintenance"

label values naicsname naics_lbl

sort er_raw, 

keep if _n<=10 | _n>=(_N-9)

keep naics3 naicsname er_raw nfirms

label variable naics3 "3-Digit NAICS"
label variable naicsname "Sector Name"
label variable er_raw "Mean Value-Weighted Return"
label variable nfirms "No of Firms"

export excel naics3 naicsname er_raw nfirms using "$WD/TableC2.xlsx", sheet("Table C2B", modify) firstrow (varlabels)

