cap log close
clear
set more off
program drop _all

global lastpayroll_filename "0628"
global lastweeknum = week(date(string(2020)+"${lastpayroll_filename}","YMD")+6)
global projdir <The Replication Packet Directory>
global initial_microdatadir <Where The Initial Micro Data Are Stored. Not Included with Replication Packet>
global figuredir "$projdir/Results/WageChanges"
global graphdatadir "$projdir/GraphData/WageChanges"

log using "$projdir/Logs/wage_change.log", replace

program main
	data_setup
	
	* Figure 6
	bar_charts, beginmonth(3) beginmonthname(March) endmonthnum(6) endmonthname(June) weightvars(naics2 size_SUSB) weightfile(SUSB_weights_size_naics2)
	
	* Figure A5
	histogram_2020, beginmonth(3) beginmonthname(March) endmonthnum(6) endmonthname(June) weightvar(weight_microdata_emp)
	histogram_2019, beginmonth(3) beginmonthname(March) endmonthnum(6) endmonthname(June) weightvar(weight_microdata_emp)
end

program data_setup
	** Calculates modal month of wage adjustment for firms in 2018, 2019, and 2020,
	** as well as the share of employees who receive a wage adjustment in that month.
	
	// Initialize file
	set obs 1
	gen client_code = ""
	save $projdir/Data/Processed/FirmWageAdjustMonths, replace

	** Loop through each month of each year, calculate share of employees who
	** receive month-over-month changes in base wages at each firm.
	foreach year in 2018 2019 2020 {
		foreach month in 01 02 03 04 05 06 07 08 09 10 11 12 {
			if `month' == 12 {
				local yrplus1 = `year' + 1
				local monthplus1 01
				}
			else {
				local yrplus1 = `year'
				local monthplus1 = `month' + 1
				if `monthplus1' < 10 {
					local zero 0
					}
				else {
					local zero
					}
				}
			di "`year' `month'"
			
			cap use emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c yr_month using "$initial_microdatadir/data_`year'`month'.dta", clear
			if _rc == 0 & !(`year' == 2020 & `month' == 04) {
				append using "$initial_microdatadir/data_`yrplus1'`zero'`monthplus1'.dta", keep(emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c yr_month)
				
				* Turn time variable from %td to %tm
				replace yr_month = mofd(yr_month)
				format yr_month %tm
				
				* Generate base wage
				gen std_wage = std_pay_rt if std_rt_type_c == "H"
				replace std_wage = std_pay_rt/40 if std_rt_type_c == "S" & pay_freq_c == "W" & mi(std_wage)
				replace std_wage = std_pay_rt/80 if std_rt_type_c == "S" & inlist(pay_freq_c,"B","S") & mi(std_wage)
				replace std_wage = . if std_wage < 2.13

				* Generate job ID
				egen empid = group(client_code emp_pur_c)
				drop if mi(empid)
				
				* Keep one observation per employee-month. There are very few
				* observations with multiple observations per employee-month, and
				* they generally reflect transient HR miscodings in the data.
				bys empid yr_month (std_wage): keep if _n == 1
				xtset empid yr_month, m
				sort empid yr_month
				
				* Generate wage change indicator
				gen byte has_wage_change = std_wage != l.std_wage if !mi(std_wage) & !mi(l.std_wage)
				keep if !mi(has_wage_change)
				
				* Collapse to share with wage change by firm-month
				collapse has_wage_change (count) N = empid, by(client_code yr_month) fast
				gen year  = year(dofm(yr_month))
				gen month = month(dofm(yr_month))
				
				* Add to existing dataset
				append using $initial_microdatadir/FirmWageAdjustMonths
				compress
				save $initial_microdatadir/FirmWageAdjustMonths, replace
				}
			}
		}

	* Generate indicator for a month being a firm's modal month of adjustment
	bys client_code year (has_wage_change): gen byte max_adjust = _n == _N
	
	* Generate total number of wage adjustments that the firm does in that year
	bys client_code year: egen tot_adjust = total(has_wage_change)
	
	* Generate share of firm's adjustments that happen in a given month.
	gen share_adjust = has_wage_change/tot_adjust
	
	* Save
	drop if mi(client_code)
	compress
	save $initial_microdatadir/FirmWageAdjustMonths, replace
end

program histogram_2020
	syntax, beginmonth(integer) beginmonthname(string asis) endmonthnum(integer) endmonthname(string asis) [weightvar(string asis)]
	
	** Bring in dataset of firms' modal wage adjustment month. Generate a dataset
	** with three variables: a firm identifier, the month in which they adjusted
	** the majority of their workerforce in 2019, and the share of their workforce
	** that they adjusted in that month. Save it as a tempfile
	use $initial_microdatadir/FirmWageAdjustMonths if max_adjust == 1 & year == 2019, clear
	keep client_code month share_adjust
	rename month modal_month
	tempfile modal_months
	save `modal_months', replace
	
	* Bring in weekly 2020 payroll data
	use emp_pur_c client_code empid week_collapse std_wage std_pay_rt_a week size_SUSB naics2 using "$initial_microdatadir/wkly_autopay_2020_cleaned.dta", clear
	gen month = month(dofw(week_collapse))
	replace std_wage = . if mi(std_pay_rt_a)
	drop if mi(std_wage)
	
	* Keep employees who were employed in February
	drop if month < `beginmonth'
	bys empid (month): keep if month[1] == `beginmonth'
	
	* Keep last observation in each month
	bys empid month (week_collapse): keep if _n == _N
	
	* Fix firm size to first observed level
	bys empid (month week_collapse): replace size_SUSB = size_SUSB[1]
	
	* Keep only first and last observations of employees, and make sure they're in the data at least two months
	bys empid (week_collapse): keep if inlist(_n,1,_N)
	
	* Generate wage changes
	bys empid (week_collapse): gen init_wage = std_wage[1]
	bys empid (week_collapse): gen delta_wage = 100*(std_wage - std_wage[_n-1])/std_wage[_n-1]
	gen byte wage_decrease = 100*(delta_wage < -0.1) if !mi(delta_wage)
	gen byte wage_change = 100*(abs(delta_wage) > 0.1) if !mi(delta_wage)
	
	** Merge in weight variables, and convert for use with fweights for histograms
	if "`weightvar'" != "" {
		merge m:1 naics2 size_SUSB using "$initial_microdatadir/weights_microdata", nogen keep(1 3)
		replace `weightvar' = floor(100000000*`weightvar')
		local weight [fw=`weightvar']
	}
	
	* Calculate unconditional share with wage decline so can calculate number of
	* wage cuts in the economy overall
	gen temp = delta_wage < -0.1
	bys empid (week_collapse): egen wage_decline_uncond = max(temp)
	bys empid (week_collapse): replace wage_decline_uncond = . if _n != _N
	
	// % of people with wage cut, unconditional on job staying
	summ wage_decline_uncond `weight'
	
	// % of job-stayers with wage cut
	bys empid (week_collapse): keep if _N >= 2
	summ wage_decline_uncond `weight' if inlist(week,${lastweeknum},${lastweeknum}-1) & !mi(delta_wage)
	
	* Only keep job-stayers throughout the whole sample
	keep if inlist(week,${lastweeknum},${lastweeknum}-1) & !mi(delta_wage)
	
	* Merge in client code's modal adjustment month in 
	merge m:1 client_code using `modal_months', keep(1 3)
	
	** See what share of workers are in firms that do at least 75% of their wage 
	** adjustment in March-June. Add 1 to begin month as these are backward looking
	** changes: a modal month of April implies you change wages between March and April.
	gen byte tokeep = inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75
	tab tokeep
	
	
	
	**** Make histogram of wage changes - FIGURE A5 Panel B
	#delimit ;
	// Only firms that have modal month in `beginmonthname'-`endmonthname' and do at least 75% of their adjustments in that month;
	summ wage_decrease if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_decrease: di %5.1f r(mean);
	summ wage_change if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_change: di %5.1f r(mean);
	local share_changes_decrease: di %5.1f 100*`share_decrease'/`share_change';
	hist delta_wage if inrange(delta_wage,-50,50) & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		percent graphregion(color(white)) 
		ylabel(0(10)60) ytick(0(5)60,grid)
		xtitle("`beginmonthname'-`endmonthname' 2020 Base Wage Change (%)") fc(gs12) lc(black) width(1) start(-50) 
		xline(0) xlabel(-50(10)50) xtick(-50(5)50,grid)
		note("`share_decrease'% had wage decline, `share_changes_decrease'% of all wage changes (`share_change'%)");
	graph export "$figuredir/delta_wage_histogram2020.pdf", replace;
	
	twoway__histogram_gen delta_wage if inrange(delta_wage,-50,50)  
		& inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		width(1) start(-50) gen(histval_2020 xval, replace) percent;
	
	
	**** Make histogram of wage changes, conditional on non-zero - Figure A5 Panel D;
	// Only firms that have modal month in `beginmonthname'-`endmonthname' and did at least 75% of their adjustments in that month;
	summ wage_decrease if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_decrease: di %5.1f r(mean);
	summ wage_change if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_change: di %5.1f r(mean);
	local share_changes_decrease: di %5.1f 100*`share_decrease'/`share_change';
	hist delta_wage if inrange(delta_wage,-50,50) & abs(delta_wage) > 0.1 & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', percent graphregion(color(white))
		xtitle("`beginmonthname'-`endmonthname' 2020 Base Wage Change (%)") fc(gs12) lc(black) width(1) start(-50) 
		xline(0) xlabel(-50(10)50) xtick(-50(5)50,grid)
		ylabel(0(10)40) ytick(0(5)40,grid)
		note("`share_decrease'% had wage decline, `share_changes_decrease'% of all wage changes (`share_change'%)");
	graph export "$figuredir/delta_wage_histogram2020_cond.pdf", replace;
	
	twoway__histogram_gen delta_wage if inrange(delta_wage,-50,50)  
		& abs(delta_wage) > 0.1 & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		width(1) start(-50) gen(histval_2020_cond xval, replace) percent;
	#delimit cr
	
	** Make histogram data for Excel sheet
	keep if !mi(xval)
	keep xval histval_2020*
	rename xval delta_wage
	label var delta_wage "Midpoint of Wage Change (%) bin: March to June"
	label var histval_2020 "Percent of Job-Stayers in bin (2020)"
	label var histval_2020_cond "Percent of Job-Stayers in bin, conditional on non-zero change (2020)"
	order delta_wage
	export excel using "Q:\Sheldon\ReplicationPacket\GraphData\WageChangesdelta_wage_histogram_data_2020.xlsx", replace firstrow(varlabels)
end

program histogram_2019
	syntax, beginmonth(integer) beginmonthname(string asis) endmonthnum(integer) endmonthname(string asis) [weightvar(string asis)]
	
	** Bring in dataset of firms' modal wage adjustment month. Generate a dataset
	** with three variables: a firm identifier, the month in which they adjusted
	** the majority of their workerforce in 2019, and the share of their workforce
	** that they adjusted in that month. Save it as a tempfile
	use "$initial_microdatadir/FirmWageAdjustMonths" if max_adjust == 1 & year == 2019, clear
	keep client_code month share_adjust
	rename month modal_month
	tempfile modal_months
	save `modal_months', replace
	
	** Now bring in 2019 monthly micro data
	use emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c naics2 clientpaysize using "$initial_microdatadir/data_20190`beginmonth'.dta", clear
	append using "$initial_microdatadir/data_20190`endmonthnum'.dta", keep(emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c naics2 clientpaysize) gen(post)
	
	* Generate base wage
	gen std_wage = std_pay_rt if std_rt_type_c == "H"
	replace std_wage = std_pay_rt/40 if std_rt_type_c == "S" & pay_freq_c == "W" & mi(std_wage)
	replace std_wage = std_pay_rt/80 if std_rt_type_c == "S" & inlist(pay_freq_c,"B","S") & mi(std_wage)
	replace std_wage = . if std_wage < 2.13 // Winsorize below federal tipped minimum wage

	* Generate Employee ID
	egen empid = group(client_code emp_pur_c)
	drop if mi(empid)
	bys empid post (std_wage): keep if _n == 1
	
	** Generate Firm Size Bins
	gen size_SUSB = recode(clientpaysize,0,49,499,999,2499,999999)
	label define size_SUSB 19 "1-19 Employees" 49 "< 50 Employees" 499 "50-499 Employees" ///
				999 "500-999 Employees" 2499 "1000-2499 Employees" 999999 "2500+ Employees"
	label val size_SUSB size_SUSB
	
	* Harmonize 2-digit NAICS codes
	replace naics2 = "44-45" if inlist(naics2,"44","45")
	replace naics2 = "48-49" if inlist(naics2,"48","49")
	replace naics2 = "31-33" if inlist(naics2,"31","32","33")
	
	* Fix firm size to first observed level
	bys empid (post): replace size_SUSB = size_SUSB[1]
	
	* Generate wage changes
	bys empid (post): gen init_wage = std_wage[1]
	gen delta_wage = 100*(std_wage - init_wage)/init_wage[_n-1]
	gen byte wage_decrease = 100*(delta_wage < -0.1) if !mi(delta_wage)
	gen byte wage_change = 100*(abs(delta_wage) > 0.1) if !mi(delta_wage)
	
	* Keep last observation of each person
	keep if !mi(delta_wage) & post == 1
	
	* Merge in client code's modal adjustment month in 
	merge m:1 client_code using `modal_months', keep(1 3)
	
	** Keep only workers working for firms who were due to adjust their wages between
	** March and June, and who adjusted at least 75% of their workers wages in that
	** modal month
	gen byte tokeep = inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75
	tab tokeep
	
	* Merge in weights and convert to fweights for use with histograms
	if "`weightvar'" != "" {
		merge m:1 naics2 size_SUSB using "$initial_microdatadir/weights_microdata", nogen keep(1 3)
		replace `weightvar' = floor(100000000*`weightvar')
		local weight [fw=`weightvar']
	}
	
	**** Make histogram of wage changes
	#delimit ;
	// Only firms that have modal month in `beginmonthname'-`endmonthname' ;
	summ wage_decrease if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_decrease: di %5.1f r(mean);
	summ wage_change if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_change: di %5.1f r(mean);
	local share_changes_decrease: di %5.1f 100*`share_decrease'/`share_change';
	hist delta_wage if inrange(delta_wage,-50,50) & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		percent graphregion(color(white))
		xtitle("`beginmonthname'-`endmonthname' 2019 Base Wage Change (%)") fc(gs12) lc(black) width(1) start(-50) 
		xline(0) xlabel(-50(10)50) xtick(-50(5)50,grid)
		ylabel(0(10)60) ytick(0(5)60,grid)
		note("`share_decrease'% had wage decline, `share_changes_decrease'% of all wage changes (`share_change'%)");
	graph export "$figuredir/delta_wage_histogram2019.pdf", replace;
	
	twoway__histogram_gen delta_wage if inrange(delta_wage,-50,50)  
		& inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		width(1) start(-50) gen(histval_2019 xval, replace) percent;
	
	
	
	**** Make histogram of wage changes, conditional on non-zero;	
	// Only firms that have modal month in `beginmonthname'-`endmonthname' and did at least 75% of their adjustments in that month;
	summ wage_decrease if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_decrease: di %5.1f r(mean);
	summ wage_change if inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight';
	local share_change: di %5.1f r(mean);
	local share_changes_decrease: di %5.1f 100*`share_decrease'/`share_change';
	hist delta_wage if inrange(delta_wage,-50,50) & abs(delta_wage) > 0.1 & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		percent graphregion(color(white))
		xtitle("`beginmonthname'-`endmonthname' 2019 Base Wage Change (%)") fc(gs12) lc(black) width(1) start(-50) 
		xline(0) xlabel(-50(10)50) xtick(-50(5)50,grid)
		ylabel(0(10)40) ytick(0(5)40,grid)
		note("`share_decrease'% had wage decline, `share_changes_decrease'% of all wage changes (`share_change'%)");
	graph export "$figuredir/delta_wage_histogram2019_cond.pdf", replace;
	
	twoway__histogram_gen delta_wage if inrange(delta_wage,-50,50)  
		& abs(delta_wage) > 0.1 & inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75 `weight', 
		width(1) start(-50) gen(histval_2019_cond xval, replace) percent;
	#delimit cr
	
	* Make histogram data
	keep if !mi(xval)
	keep xval histval_2019*
	rename xval delta_wage
	label var delta_wage "Midpoint of Wage Change (%) bin: March to June"
	label var histval_2019 "Percent of Job-Stayers in bin (2019)"
	label var histval_2019_cond "Percent of Job-Stayers in bin, conditional on non-zero change (2019)"
	order delta_wage
	export excel using "Q:\Sheldon\ReplicationPacket\GraphData\WageChanges/delta_wage_histogram_data_2019.xlsx", replace firstrow(varlabels)
end

program bar_charts
	syntax, beginmonth(integer) beginmonthname(string asis) endmonthnum(integer) endmonthname(string asis) ///
			[weightvars(string asis) weightfile(string asis) suffix(string asis) full_sample]
			
	** Bring in dataset of firms' modal wage adjustment month. Generate a dataset
	** with three variables: a firm identifier, the month in which they adjusted
	** the majority of their workerforce in 2019, and the share of their workforce
	** that they adjusted in that month. Save it as a tempfile
	use $initial_microdatadir/FirmWageAdjustMonths if max_adjust == 1 & year == 2019, clear
	keep client_code month share_adjust
	rename month modal_month
	tempfile modal_months
	save `modal_months', replace
	
	** Now bring in 2020 weekly micro data
	use emp_pur_c client_code empid week_collapse std_wage std_pay_rt_a week size_SUSB naics2 using "$initial_microdatadir/wkly_autopay_2020_cleaned.dta", clear
	gen month = month(dofw(week_collapse))
	replace std_wage = . if mi(std_pay_rt_a)
	drop if mi(std_wage)
	
	* Keep employees who were employed in February
	drop if month < `beginmonth'
	bys empid (month): keep if month[1] == `beginmonth'
	
	** Count firm size as of the beginning of the sample for weighting
	bys empid (month week_collapse): replace size_SUSB = size_SUSB[1]
	
	* Keep last observation in each month
	bys empid month (week_collapse): keep if _n == _N
	
	* Keep only first and last observations of employees, and make sure they're in the data at least two months
	bys empid (week_collapse): keep if inlist(_n,1,_N) & _N >= 2
	
	* Generate wage changes
	bys empid (week_collapse): gen init_wage = std_wage[1]
	gen delta_wage = 100*(std_wage - init_wage)/init_wage[_n-1]
	gen byte wage_decrease = 100*(delta_wage < -0.1) if !mi(delta_wage)
	gen byte wage_change = 100*(abs(delta_wage) > 0.1) if !mi(delta_wage)
	
	* Generate initial wage quintiles
	gen wage_quintile = 1 if init_wage < 13.5
	replace wage_quintile = 2 if inrange(init_wage,13.5,16.41)
	replace wage_quintile = 3 if inrange(init_wage,16.41,24.53)
	replace wage_quintile = 4 if inrange(init_wage,24.53,32.45)
	replace wage_quintile = 5 if std_wage >= 32.45 & !mi(init_wage)
	
	* Keep only the final weeks of the sample
	keep if inlist(week,${lastweeknum},${lastweeknum}-1) & !mi(delta_wage)
	
	* Merge in client code's modal adjustment month in 
	merge m:1 client_code using `modal_months', keep(1 3)
	
	** Keep only workers working for firms who were due to adjust their wages between
	** March and June, and who adjusted at least 75% of their workers wages in that
	** modal month
	gen byte tokeep = inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75
	tab tokeep
	if "`full_sample'" == "" {
		keep if tokeep == 1
	}
	
	* Winsorize top and bottom 1% of wages for average wage change calculation
	summ delta_wage, d
	replace delta_wage = . if !inrange(delta_wage,r(p1),r(p99))
	
	* Collapse to total economy changes/changes by wage quintile.
	tempfile all
	if "`weightvars'" == "" {
		xcollapse delta_wage wage_decrease wage_change, saving(`all', replace)
	}
	else {
		xcollapse delta_wage wage_decrease wage_change, by(`weightvars') saving(`all', replace)
	}
	tempfile naics2020
	xcollapse delta_wage wage_decrease wage_change, by(naics2) saving(`naics2020', replace)
	collapse delta_wage wage_decrease wage_change if !mi(wage_quintile), by(wage_quintile `weightvars') fast
	append using `all'
	
	* Generate wage freeze variable
	replace wage_quintile = 0 if mi(wage_quintile)
	gen wage_freeze = 100 - wage_change
	foreach var in delta_wage wage_decrease wage_change wage_freeze {
		rename `var' `var'2020
		}
	
	* If weighting, merge in weight files and take weighted average change probabilities
	if "`weightvars'" != "" {
		merge m:1 `weightvars' using "$initial_microdatadir/`weightfile'"
		collapse *2020 [aw=share_emp], by(wage_quintile) fast
	}
	
	tempfile chg2020
	compress
	save `chg2020', replace
	
	
	*********************************
	******* 2019 MONTHLY DATA *******
	*********************************
	
	use emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c naics2 clientpaysize using "$initial_microdatadir/data_20190`beginmonth'.dta", clear
	append using "$initial_microdatadir/data_20190`endmonthnum'.dta", keep(emp_pur_c client_code std_pay_rt std_rt_type_c pay_freq_c naics2 clientpaysize) gen(post)
	
	* Harmonize 2-digit NAICS codes
	replace naics2 = "44-45" if inlist(naics2,"44","45")
	replace naics2 = "48-49" if inlist(naics2,"48","49")
	replace naics2 = "31-33" if inlist(naics2,"31","32","33")
	
	* Generate Base Wage Variable
	gen std_wage = std_pay_rt if std_rt_type_c == "H"
	replace std_wage = std_pay_rt/40 if std_rt_type_c == "S" & pay_freq_c == "W" & mi(std_wage)
	replace std_wage = std_pay_rt/80 if std_rt_type_c == "S" & inlist(pay_freq_c,"B","S") & mi(std_wage)
	replace std_wage = . if std_wage < 2.13

	** Generate job ID
	egen empid = group(client_code emp_pur_c)
	drop if mi(empid)
	bys empid post (std_wage): keep if _n == 1
	
	** Generate Firm Size Bins
	gen size_SUSB = recode(clientpaysize,0,49,499,999,2499,999999)
	label define size_SUSB 19 "1-19 Employees" 49 "< 50 Employees" 499 "50-499 Employees" ///
				999 "500-999 Employees" 2499 "1000-2499 Employees" 999999 "2500+ Employees"
	label val size_SUSB size_SUSB
	
	* Fix firm size to first observed level
	bys empid (post): replace size_SUSB = size_SUSB[1]
	
	* Generate wage changes
	bys empid (post): gen init_wage = std_wage[1]
	gen delta_wage = 100*(std_wage - init_wage)/init_wage[_n-1]
	gen byte wage_decrease = 100*(delta_wage < -0.1) if !mi(delta_wage)
	gen byte wage_change = 100*(abs(delta_wage) > 0.1) if !mi(delta_wage)
	
	* Generate initial wage quintiles
	gen wage_quintile = 1 if init_wage < 13.5
	replace wage_quintile = 2 if inrange(init_wage,13.5,16.41)
	replace wage_quintile = 3 if inrange(init_wage,16.41,24.53)
	replace wage_quintile = 4 if inrange(init_wage,24.53,32.45)
	replace wage_quintile = 5 if std_wage >= 32.45 & !mi(init_wage)
	
	keep if !mi(delta_wage) & post == 1
	
	* Merge in client code's modal adjustment month in 
	merge m:1 client_code using `modal_months', keep(1 3)
	
	** Keep only workers working for firms who were due to adjust their wages between
	** March and June, and who adjusted at least 75% of their workers wages in that
	** modal month
	gen byte tokeep = inrange(modal_month,`beginmonth'+1,`endmonthnum') & share_adjust >= 0.75
	tab tokeep
	keep if tokeep == 1
	
	* Winsorize top and bottom 1% of wages for average wage change calculation
	summ delta_wage, d
	replace delta_wage = . if !inrange(delta_wage,r(p1),r(p99))
	
	* Collapse to total economy changes/changes by wage quintile.
	if "`weightvars'" == "" {
		xcollapse delta_wage wage_decrease wage_change, saving(`all', replace)
	}
	else {
		xcollapse delta_wage wage_decrease wage_change, by(`weightvars') saving(`all', replace)
	}
	tempfile naics2019
	xcollapse delta_wage wage_decrease wage_change, by(naics2) saving(`naics2019', replace)
	collapse delta_wage wage_decrease wage_change if !mi(wage_quintile), by(wage_quintile `weightvars') fast
	append using `all'
	
	* Generate wage freeze
	replace wage_quintile = 0 if mi(wage_quintile)
	gen wage_freeze = 100 - wage_change
	foreach var in delta_wage wage_decrease wage_change wage_freeze {
		rename `var' `var'2019
		}
	
	* If weighting, merge in weight files and take weighted average change probabilities
	if "`weightvars'" != "" {
		merge m:1 `weightvars' using "$initial_microdatadir/`weightfile'"
		collapse *2019 [aw=share_emp], by(wage_quintile) fast
	}
		
	* Merge in 2020 data
	merge 1:1 wage_quintile using `chg2020', nogen
	
	* Label variables and export data to an Excel document
	label define quintile 0 "All" 1 "1st" 2 "2nd" 3 "3rd" 4 "4th" 5 "5th"
	label val wage_quintile quintile
	
	drop delta_wage2019 delta_wage2020
	label var wage_decrease2019 "% With Wage Decrease, 2019"
	label var wage_decrease2020 "% With Wage Decrease, 2020"
	label var wage_change2019 "% With Wage Change, 2019"
	label var wage_change2020 "% With Wage Change, 2020"
	label var wage_freeze2019 "% With Wage Freeze, 2019"
	label var wage_freeze2020 "% With Wage Freeze, 2020"
	
	sort wage_quintile
	order wage_quintile wage_freeze2019 wage_freeze2020 wage_decrease2019 wage_decrease2020 wage_change2019 wage_change2020
	
	export excel using "$graphdatadir/freeze_cut_quintiles.xlsx", replace firstrow(varlabel)
	
	
	
	******************************
	****** MAKE BAR CHARTS *******
	******************************
	
	#delimit ;
	** Figure 6 Panel A;
	graph bar wage_decrease2019 wage_decrease2020, over(wage_quintile)
				graphregion(color(white)) bar(1,fc(edkblue)) bar(2,fc(eltblue))
				ytitle("% With Base Wage Decrease") ytick(0(2)16,grid) ylabel(0(4)16)
				legend(label(1 "2019") label(2 "2020"));
	graph export "$figuredir/wage_cuts_quintile_20192020.pdf", replace;
	
	** Figure 6 Panel B;
	graph bar wage_freeze2019 wage_freeze2020, over(wage_quintile)
				graphregion(color(white)) bar(1,fc(edkblue)) bar(2,fc(eltblue))
				ytitle("% With Base Wage Freeze") ytick(0(5)70,grid) ylabel(0(10)70)
				legend(label(1 "2019") label(2 "2020"));
	graph export "$figuredir/wage_freeze_quintile_20192020.pdf", replace;
	
	#delimit cr
end

main
