cap log close
clear
set more off
program drop _all

global projdir <The Replication Packet Directory>
global initial_microdatadir <Where The Initial Micro Data Are Stored. Not Included with Replication Packet>
global inputdatadir "$projdir/InputData"
global graphdatadir "$projdir/GraphData"
global figuredir "$projdir/Recall"

global refweek = 7 // Week number to calculate wage percentiles in and plot relative plots against
global lastpayroll_filename "0628"
global lastweek = wofd(date(string(2020)+"${lastpayroll_filename}","YMD")+6)
global lastweeknum = week(date(string(2020)+"${lastpayroll_filename}","YMD")+6)
global empcutoff_for_closure = 0.1
set seed 1823718

program main
	collapse_to_firm_data
	generate_exit_vars, file(Firm_Emp)
	generate_exit_vars, file(Firm_Emp_SameEmployees)
	generate_exit_vars, file(Firm_Emp_by_WageQuintile) wage_quintile
	make_plots
end

program collapse_to_firm_data
	** Generate firm exit
	use pay_freq_c client_code emp_pur_c week_collapse std_wage naics2 using "$initial_microdatadir/wkly_autopay_2020_cleaned.dta", clear
	tab week_collapse
	di ${lastweek}
	gen byte weekly = pay_freq_c == "W"
	gen byte biweekly = inlist(pay_freq_c,"BE","BO","B")
	gen N = biweekly + weekly/2

	* Generate two-week variable
	gen twoweek = 2*ceil(week_collapse/2)
	gen twoweek_week = week(dofw(twoweek))
	tab twoweek_week 
	keep if twoweek_week <= ${lastweeknum}

	* Keep only firms around in first 2 weeks of February
	bys client_code (twoweek): keep if twoweek_week[1] == $refweek

	* Generate flag for whether the employee was around during those first two weeks of Feb
	bys client_code emp_pur_c (twoweek): gen byte first_twoweek_emp = twoweek_week[1] == $refweek

	* Generate Wage Quintiles
	gen wage_quintile = 1 if std_wage < 13.5
	replace wage_quintile = 2 if inrange(std_wage,13.5,16.41)
	replace wage_quintile = 3 if inrange(std_wage,16.41,24.53)
	replace wage_quintile = 4 if inrange(std_wage,24.53,32.45)
	replace wage_quintile = 5 if std_wage >= 32.45 & !mi(std_wage)

	* Collapse to firm employment
	replace naics2 = substr(naics2,1,2)
	destring naics2, replace
	#delimit ;
	xcollapse (sum) N (min) naics2, by(client_code twoweek) saving("$initial_microdatadir/Firm_Emp", replace);
	
	xcollapse (sum) N_same_employees = N (min) naics2 if first_twoweek_emp == 1, 
					   by(client_code twoweek) 
					   saving("$initial_microdatadir/Firm_Emp_SameEmployees", replace);
					   
	xcollapse (sum) N (min) naics2, by(client_code wage_quintile twoweek) 
					   saving("$initial_microdatadir/Firm_Emp_by_WageQuintile", replace);
	#delimit cr
end

program generate_exit_vars
	syntax, file(string asis) [wage_quintile]
	use "$initial_microdatadir/`file'", clear
	
	* Harmonize industry codes
	set seed 1823718
	gen rnum = runiform() if naics2 != 99
	bys client_code (rnum): replace naics2 = naics2[1]
	drop rnum
	
	* Calculate number of workers in each wage quintile in each firm, reshaping wide.
	if "`wage_quintile'" != "" {
		replace wage_quintile = 0 if mi(wage_quintile)
		reshape wide N, i(client_code twoweek) j(wage_quintile)
		egen N = rowtotal(N*)
		}
		
	* Create full panel
	fillin client_code twoweek
	drop _fillin
	
	* Generate initial N variable
	foreach var of varlist N* {
		replace `var' = 0 if mi(`var')
		bys client_code (twoweek): gen initial_`var' = `var'[1]
		}

	* Generate exit flag
	gen byte exited = N < ${empcutoff_for_closure}*initial_N

	* Carry forward exit flag
	gen ever_exited = exited
	bys client_code (twoweek): replace ever_exited = max(exit,ever_exited[_n-1]) if !mi(exited[_n-1])

	* Generate re-entry indicator
	gen byte reentry = N >= ${empcutoff_for_closure}*initial_N & ever_exited == 1

	* Label variables
	label var exited "Business Closed (<10% of February Employment)"
	label var ever_exited "Business Closed at some point in past"
	label var reentry "Re-entered (Business Closed in past, but have at least 10% of Feb employment in t"

	* Generate share of employment that has returned
	gen share_return = N/initial_N if reentry
	
	* Generate re-entry employment
	gen reentered_empcount = N if reentry
	
	* Generate total February employment in exited firms
	gen exited_empcount = initial_N if exited
	
	* Do the same as above by wage quintile
	if "`wage_quintile'" != "" {
		forvalues quintile = 1/5 {
			gen share_return_quintile`quintile' = N`quintile'/initial_N`quintile' if reentry
			gen reentered_empcount`quintile' = N`quintile' if reentry
			gen exited_empcount`quintile' = initial_N`quintile' if exited
			}
		}

	* Generate day variable for time series plots
	gen day = dofw(twoweek)+2
	format day %tdMondd

	gen week = week(day)
	format twoweek %tw

	summ day
	local beginday = r(min)
	local endday = r(max)
	
	gen byte lastweek = twoweek == 2*(floor($lastweek/2))

	compress
	save "$initial_microdatadir/`file'", replace
end

program make_plots
	make_plots_overall, file(Firm_Emp) weightvar(weight_microdata_firm) suffix(_weighted_Naics2Size) // Figure 8 and A7
	make_plots_quintile // Figure A6
	share_return_init_emp,  weightvar(weight_microdata_firm) suffix(_weighted_Naics2Size) // Figure 9 and A8
end

program make_plots_overall
	syntax, file(string asis) [suffix(string asis) weightvar(string asis)]
	
	
	use "$initial_microdatadir/Firm_Emp", clear
	
	* Merge in number of employees that were previously employed by firm in February
	merge 1:1 client_code twoweek using "$initial_microdatadir/Firm_Emp_SameEmployees", keepusing(N_same_employees) nogen
	
	replace day = day - 11
	
	** Generate Firm Size Bins
	gen size_SUSB = recode(initial_N,0,49,499,999,2499,999999)
	label define size_SUSB 19 "1-19 Employees" 49 "< 50 Employees" 499 "50-499 Employees" ///
				999 "500-999 Employees" 2499 "1000-2499 Employees" 999999 "2500+ Employees"
	label val size_SUSB size_SUSB
	
	* Merge in weights
	if "`weightvar'" != "" {
		tostring naics2, replace
		replace naics2 = "31-33" if naics2 == "31"
		replace naics2 = "44-45" if naics2 == "44"
		replace naics2 = "48-49" if naics2 == "48"
		merge m:1 naics2 size_SUSB using "$inputdatadir/weights_microdata", nogen keep(1 3)
		replace `weightvar' = floor(100000000*`weightvar')
		local weight [fw=`weightvar']
		gen weight = initial_N*`weightvar'
	}
	
	** Make plots showing CDF of employment on re-entry and share of re-entered
	** employment that is a recall
	#delimit ;
	
	// Figure 8A;
	replace share_return = 1.5 if share_return > 1.5 & !mi(share_return);
	summ share_return `weight', d;
	#delimit ;
	cdfplot share_return if reentry == 1 & lastweek == 1 `weight', opt1(xtitle("Re-Opening Employment Relative to Size Feb 1-14 Employment") lw(thick)
						  graphregion(color(white)) xlabel(0(0.5)1.5) xtick(0(0.1)1.5, grid) xline(1));
	graph export "$figuredir/CDF_share_emp_return`suffix'.pdf", replace;
	cumul share_return if reentry == 1 & lastweek == 1 `weight', gen(reentry_size_cdf);
	
	// Figure A7A;
	cdfplot share_return if reentry == 1 & lastweek == 1 [fw=weight], opt1(xtitle("Re-Opening Employment Relative to Feb 1-14 Employment") lw(thick)
						  graphregion(color(white)) xlabel(0(0.5)1.5) xtick(0(0.1)1.5, grid) xline(1));
	graph export "$figuredir/CDF_share_emp_return_empweight`suffix'.pdf", replace;
	cumul share_return if reentry == 1 & lastweek == 1 [fw=weight], gen(reentry_size_empweight_cdf);
	
	
	**** Make plots of share of returning employees who are recalls;
	gen share_same = N_same_employees/N;
	summ share_same `weight', d;
	#delimit ;
	// Figure 8B;
	cdfplot share_same if lastweek & reentry == 1 `weight', opt1(graphregion(color(white))
				xtick(0(0.05)1,grid) xlabel(0(0.1)1, gstyle(major)) ytick(0(0.1)1,grid) 
				xtitle("Share of Returning Employees Present in February 1-14") lw(thick));
	graph export "$figuredir/CDF_share_returners_same_emp`suffix'.pdf", replace;
	cumul share_same if lastweek & reentry == 1 `weight', gen(recall_cdf);	
	
	// Figure A7B;
	cdfplot share_same if lastweek & reentry == 1 [fw=weight], opt1(graphregion(color(white))
				xtick(0(0.05)1,grid) xlabel(0(0.1)1, gstyle(major)) ytick(0(0.1)1,grid) 
				xtitle("Share of Returning Employees Present in February 1-14") lw(thick));
	graph export "$figuredir/CDF_share_returners_same_emp_empweight`suffix'.pdf", replace;
	cumul share_same if lastweek & reentry == 1 [fw=weight], gen(recall_empweight_cdf);	

	#delimit cr
	
	** Make data for plots in Excel document
	keep share_return reentry_size*cdf share_same recall*cdf
	
	// Figure 8A
	preserve
	keep share_return reentry_size_cdf
	keep if !mi(reentry_size_cdf)
	bys share_return (reentry_size_cdf): keep if _n == _N
	label var share_return "Re-Entering Firm Size Relative to February"
	label var reentry_size_cdf "CDF Value"
	export excel using "$graphdatadir/Figure8A.xlsx", replace firstrow(varlabel)
	restore
	
	// Figure A7A
	preserve
	keep share_return reentry_size_empweight_cdf
	keep if !mi(reentry_size_empweight_cdf)
	bys share_return (reentry_size_empweight_cdf): keep if _n == _N
	label var share_return "Re-Entering Firm Size Relative to February"
	label var reentry_size_empweight_cdf "CDF Value (Employment Weighted)"
	export excel using "$graphdatadir/FigureA7A.xlsx", replace firstrow(varlabel)
	restore
	
	// Figure 8B
	preserve
	keep share_same recall_cdf
	keep if !mi(recall_cdf)
	bys share_same (recall_cdf): keep if _n == _N
	label var share_same "Share of Returning Employees that are Recalled (Employed by Same Firm in February)"
	label var recall_cdf "CDF Value"
	export excel using "$graphdatadir/Figure8B.xlsx", replace firstrow(varlabel)
	restore
	
	// Figure A7B
	preserve
	keep share_same recall_empweight_cdf
	keep if !mi(recall_empweight_cdf)
	bys share_same (recall_empweight_cdf): keep if _n == _N
	label var share_same "Share of Returning Employees that are Recalled (Employed by Same Firm in February)"
	label var recall_empweight_cdf "CDF Value"
	export excel using "$graphdatadir/FigureA7B.xlsx", replace firstrow(varlabel)
	restore
end

program make_plots_quintile
	** Make exit/re-entry plots by wage quintile
	use "$initial_microdatadir/Firm_Emp_by_WageQuintile", clear
	replace day = day - 11
	
	#delimit ;
	collapse (sum) reentered_empcount* exited_empcount* initial_N*, by(day) fast;

	sort day;
	
	* Generate the share of employment by wage quintile that has ever exited or;
	* re-entered in each day;
	forvalues quintile = 1/5 {;
		cap replace reentered_empcount`quintile' = reentered_empcount`quintile'/initial_N`quintile';
		cap replace exited_empcount`quintile' = exited_empcount`quintile'/initial_N`quintile';
		cap gen ever_exited_empcount`quintile' = reentered_empcount`quintile' + exited_empcount`quintile';		
		gen share_emp_exit_reentered`quintile' = reentered_empcount`quintile'/ever_exited_empcount`quintile'[_N];
		};
	
	*** Make plots;
	** Figure A6 Panel A;
	twoway 	(line share_emp_exit_reentered1 day, lc(edkblue) lw(thick) lp(solid))
			(line share_emp_exit_reentered2 day, lc(emidblue) lw(thick) lp(dash))
			(line share_emp_exit_reentered3 day, lc(eltblue) lw(thick) lp(dash_dot))
			(line share_emp_exit_reentered4 day, lc(red) lw(thick) lp(solid))
			(line share_emp_exit_reentered5 day, lc(black) lw(thick) lp(solid)),
			ytitle("Share of Quintile Employment" "in Exited Firms that Have Returned") xtitle("")
			graphregion(color(white))
			legend(label(1 "1st") label(2 "2nd") label(3 "3rd") label(4 "4th") label(5 "5th") row(1)) ;
	graph export "$figuredir/share_exited_reentered_quintile_empweighted.pdf", replace;
	
	** Figure A6 Panel B;
	twoway 	(line exited_empcount1 day, lc(edkblue) lw(thick) lp(solid))
			(line exited_empcount2 day, lc(emidblue) lw(thick) lp(dash))
			(line exited_empcount3 day, lc(eltblue) lw(thick) lp(dash_dot))
			(line exited_empcount4 day, lc(red) lw(thick) lp(solid))
			(line exited_empcount5 day, lc(black) lw(thick) lp(solid)),
			ytitle("Share of Initial Quintile Employment in Exited Firms") xtitle("")
			graphregion(color(white))
			legend(label(1 "1st") label(2 "2nd") label(3 "3rd") label(4 "4th") label(5 "5th") row(1)) ;
	graph export "$figuredir/exited_empcount_quintile_empweighted.pdf", replace;
	
	#delimit cr
	
	export excel using "$figuredir/exit_reentry_quintile.xlsx", firstrow(varlabel) replace
end

program share_return_init_emp
	syntax, [weightvar(string asis) suffix(string asis)]
	
	** Continuing firm size distribution plots (Figure 9 and A8)
	
	** Bring in Firm Employment Levels
	use "$initial_microdatadir/Firm_Emp", clear
	
	** Merge in Firm employment of the employees who were employed with the firm in January
	merge 1:1 client_code twoweek using "$initial_microdatadir/Firm_Emp_SameEmployees", keepusing(N_same_employees) nogen
	
	** Generate Firm Size Bins
	gen size_SUSB = recode(initial_N,0,49,499,999,2499,999999)
	label define size_SUSB 19 "1-19 Employees" 49 "< 50 Employees" 499 "50-499 Employees" ///
				999 "500-999 Employees" 2499 "1000-2499 Employees" 999999 "2500+ Employees"
	label val size_SUSB size_SUSB
	
	* Merge in weights to represent firm size x industry mix
	tostring naics2, replace
	replace naics2 = "31-33" if naics2 == "31"
	replace naics2 = "44-45" if naics2 == "44"
	replace naics2 = "48-49" if naics2 == "48"
	merge m:1 naics2 size_SUSB using "$inputdatadir/weights_microdata", nogen keep(1 3)
	replace `weightvar' = floor(100000000*`weightvar')
	local weight [fw=`weightvar']
	
	* Keep continuing firms
	bys client_code (twoweek): egen ever_exit = max(exited)
	
	* Create CDF of share of employment growth from initial employees
	// Generate week number in which firm had bottomed out employment
	bys client_code (N week): gen minweek = week[1]
	tab minweek
	
	// Calculate smallest employment of each firm
	bys client_code (N week): gen min_N = N[1]
	
	// Calculate smallest employment of February employees of each firm
	bys client_code (N week): gen min_N_same_emp = N_same_employees[1]
	
	// Create beginning and ending employment
	bys client_code (week): gen N_orig = N[1]
	bys client_code (week): gen N_final = N[_N]
	
	// Create employment in final period relative to employment in first observed period
	gen relsize_orig = N_final/N_orig
	gen relsize_min = N_final/min_N
	
	// Calculate employment growth since minimum
	gen grow = N - min_N
	
	// Calculate employment growth from initial employees
	gen grow_same = N_same_employees - min_N_same_emp
	
	// Calculate share of employment growth accounted for by initial employees
	gen share_grow_same = grow_same/grow
	
	// Top and bottom code these shares for figure
	replace share_grow_same = -1 if share_grow_same < -1
	replace share_grow_same = 1 if share_grow_same > 1 & !mi(share_grow_same)
	summ relsize*, d
	replace relsize_orig = min(relsize_orig,1.5)
	replace relsize_min = min(relsize_min,2)
	
	*** MAKE FIGURES for FIGURE 9 and A8
	#delimit ;
	
	// Figure 9A;
	cdfplot relsize_orig if lastweek & ever_exit == 0 & minweek >= 15 `weight', xline(1) ytick(0(0.1)1, grid) 
									  xlabel(0(0.2)1.5) xtick(0(0.1)1.5,grid) 
									  xtitle("Firm Size Relative to Early February") 
									  graphregion(color(white)) opt1(lw(thick));
	graph export "$figuredir/relsize_orig_growing_firms`suffix'.pdf", replace;
	cumul relsize_orig `weight' if lastweek & ever_exit == 0 & minweek >= 15, gen(relsize_orig_cdf);	
	
	// Figure 9B;
	cdfplot share_grow_same if lastweek & ever_exit == 0 & grow >= 10 & minweek >= 15 `weight', xline(0) ytick(0(0.1)1, grid) 
									  xlabel(-1(0.2)1) xtick(-1(0.1)1,grid) 
									  xtitle("Share of Employment Growth From Initial Employees") 
									  graphregion(color(white)) opt1(lw(thick))
									  text(0.9 -0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 -0.5 "Shedding Initial Emp", place(b)) 
									  text(0.9 0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 0.5 "Hiring Initial Emp", place(b));
	graph export "$figuredir/share_growth_returners_continuingFirms_atleast10grow`suffix'.pdf", replace;
	cumul share_grow_same if lastweek & ever_exit == 0 & grow >= 10 & minweek >= 15 `weight', gen(share_grow_same_cdf);	
	
	// Figure A8A;
	cdfplot share_grow_same if lastweek & ever_exit == 0 & minweek >= 15 `weight', xline(0) ytick(0(0.1)1, grid) 
									  xlabel(-1(0.2)1) xtick(-1(0.1)1,grid) 
									  xtitle("Share of Employment Growth From Initial Employees") 
									  graphregion(color(white)) opt1(lw(thick))
									  text(0.9 -0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 -0.5 "Shedding Initial Emp", place(b)) 
									  text(0.9 0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 0.5 "Hiring Initial Emp", place(b));
	graph export "$figuredir/share_growth_returners_continuingFirms`suffix'.pdf", replace;
	cumul share_grow_same if lastweek & ever_exit == 0 & grow >= 5 & minweek >= 15 `weight', gen(share_grow_same_atleast1_cdf);	
	
	// Figure A8B;
	cdfplot share_grow_same if lastweek & ever_exit == 0 & grow >= 5 & minweek >= 15 `weight', xline(0) ytick(0(0.1)1, grid) 
									  xlabel(-1(0.2)1) xtick(-1(0.1)1,grid) 
									  xtitle("Share of Employment Growth From Initial Employees") 
									  graphregion(color(white)) opt1(lw(thick))
									  text(0.9 -0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 -0.5 "Shedding Initial Emp", place(b)) 
									  text(0.9 0.5 "Hiring Outside Emp", place(b))  
									  text(0.85 0.5 "Hiring Initial Emp", place(b));
	graph export "$figuredir/share_growth_returners_continuingFirms_atleast5grow`suffix'.pdf", replace;
	cumul share_grow_same if lastweek & ever_exit == 0 & grow >= 5 & minweek >= 15 `weight', gen(share_grow_same_atleast5_cdf);
	
	** Make graph data for export;
	keep relsize_orig relsize_orig_cdf share_grow_same share_grow_same*cdf;
	drop if mi(relsize_orig_cdf) & mi(share_grow_same_cdf) & mi(share_grow_same_atleast1_cdf) & mi(share_grow_same_atleast5_cdf);
	
	* Keep only relative size for Figure 9A;
	preserve;
	keep relsize_orig relsize_orig_cdf;
	keep if !mi(relsize_orig_cdf);
	bys relsize_orig (relsize_orig_cdf): keep if _n == _N;
	label var relsize_orig "Firm Size Relative to February";
	label var relsize_orig_cdf "CDF Value";
	export excel using "$graphdatadir/Figure9A.xlsx", replace firstrow(varlabel); 
	restore;
	
	* Now do the same for Figure 9B;
	preserve;
	keep share_grow_same share_grow_same_cdf;
	keep if !mi(share_grow_same_cdf);
	bys share_grow_same (share_grow_same_cdf): keep if _n == _N;
	label var share_grow_same "Share of Employment Growth that is Recall (i.e. Employed by Same Firm in February)";
	label var share_grow_same_cdf "CDF Value";
	export excel using "$graphdatadir/Figure9B.xlsx", replace firstrow(varlabel); 
	restore;
	
	* Now do the same for Figure A8A;
	preserve;
	keep share_grow_same share_grow_same_atleast1_cdf;
	keep if !mi(share_grow_same_atleast1_cdf);
	bys share_grow_same (share_grow_same_atleast1_cdf): keep if _n == _N;
	label var share_grow_same "Share of Employment Growth that is Recall (i.e. Employed by Same Firm in February)";
	label var share_grow_same_atleast1_cdf "CDF Value";
	export excel using "$graphdatadir/FigureA8A.xlsx", replace firstrow(varlabel); 
	restore;
	
	* Now do the same for Figure A8A;
	preserve;
	keep share_grow_same share_grow_same_atleast5_cdf;
	keep if !mi(share_grow_same_atleast5_cdf);
	bys share_grow_same (share_grow_same_atleast5_cdf): keep if _n == _N;
	label var share_grow_same "Share of Employment Growth that is Recall (i.e. Employed by Same Firm in February)";
	label var share_grow_same_atleast5_cdf "CDF Value";
	export excel using "$graphdatadir/FigureA8B.xlsx", replace firstrow(varlabel); 
	restore;
	
	#delimit cr
end

main

