/********************************************************************************

The following .do file identifies the workers who were laid off and rehired, and
estimates how worker characterictics predict being laid off or rehired.

First, we pull in and clean survey information. 

Next, we open the cleaned Homebase file. We clean it further and identify
whether worker is laidoff at height of crisis, and whether they have been 
rehired. We also identify an array of other characteristics observable in 
the Homebase data.

Finally, we run a series of logits estimating likelhood of layoff and rehiring
by these characteristics.

*******************************************************************************/


cap log close workerlevel_prob_log
local date: display %tdCCYY-NN-DD =daily("`c(current_date)'", "DMY")
log using "/accounts/projects/jrothst/homebase/data/bpea_replication_archive/logs/03b_workerlevel_`date'.log", ///
	replace name(workerlevel_prob_log)


set more off
set varabbrev off

loc dir 	"/accounts/projects/jrothst/homebase/data/bpea_replication_archive/"
loc code	"`dir'/code/"
loc results	"`dir'/results/"
loc sourcedata	"`dir'/data_raw/"
loc workingdata	"`dir'/data_clean/"



********************************************************************************
* Pull in, clean and save tempfile with PPP and UI information
********************************************************************************

use `workingdata'/ppp/ppp_stc_sel, clear
 keep st stfips loanc uic
 ren st state
tempfile pppuic
save `pppuic'


********************************************************************************
* Pull in and clean survey information
********************************************************************************

use `workingdata'/survey_worker/homebase_worker_survey_raw_20200707, clear
 ren userid person
 
 gsort person -date_end
 by person: gen count=_n
 keep if count==1
 isid person
 drop count
 
 forvalues q=1/4 {
  replace fQ15_`q'=fQ16_`q' if fQ15_`q'==. // job satisfaction
 }
 
 gen fQ26=.
 forvalues q=1/11 {
  replace fQ26=`q' if fQ26_`q'==1
 }

 ren user_gender	gender
 ren user_age		age
 ren user_income	income
 ren user_educ		educ
 
 ren fQ7		work_impact
 ren fQ10		employer_encourage_ui
 ren fQ11		employer_says_rehire
 ren fQ15_1		job_satisfaction_manager
 ren fQ15_2		job_satisfaction_coworkers
 ren fQ15_3		job_satisfaction_wages
 ren fQ15_4		job_satisfaction_purpose
 ren Q25		apply_ui
 ren fQ26		why_notapply_ui
 
*Gender
 recode gender (2=1) (1=2)
 lab def gender 	1 "Male"	///
			2 "Female"	///
			3 "Non-binary"	///
			, replace
 lab val gender gender
 
*Married
 gen married=(user_marital==2)
 lab def married	0 "Unmarried"	///
			1 "Married", replace
 lab val married married
 
*Children
 gen any_children=(user_child==1)
 
 lab def any_children	0 "No children"		///
			1 "Have children"	, replace 
 lab val any_children any_children
 
*Female/child interaction
 gen family=.
 replace family=0 if any_children==0
 replace family=1 if gender==1 & any_children==1
 replace family=2 if gender==2 & any_children==1
 
 label def family	0 "No Children"				///
			1 "Presence of a child | male"		///
			2 "Presence of a child | female", 	///
			replace
 label val family family
 
*Race
 gen race = .
 replace race=1 if user_race_w==1
 replace race=2 if user_race_a==1 | user_race_p==1
 replace race=4 if user_race_i==1
 *replace race=5 if user_race_p==1
 replace race=3 if user_race_h==1
 replace race=5 if user_race_b==1

 lab def race 	1 "White" 		///
		2 "Asian/NH/PI"		///
		3 "Hispanic"		///
		4 "AI/AN"		///
		5 "Black", replace
 lab val race race
 
 gen nonwhite=race!=1
 lab def nonwhite	0 "White"	///
			1 "Non-white"	, replace
 lab val nonwhite nonwhite
 
 
*Education
 recode educ (5=4)
 lab def educ 	1	"LTHS"		///
		2	"HS"		///
		3	"S Coll"	///
		4	"BA", replace
 lab val educ educ	
 
 
 gen college = (inrange(educ,3,5))
 lab def college 	0 "HS or less"	///
			1 "Associate's degree or higher", replace
 lab val college college
 
 

*Expectations
 recode employer_says_rehire (2=0) (3=2)
 lab def employer_says_rehire 	0 "No" 		///
				1 "Yes"		///
				2 "Unsure", replace
 lab val employer_says_rehire employer_says_rehire
 
 tab fQ17, m
 tab fQ17, m nol
 recode fQ17 (6=1) (5=2) (4=3) (3=4) (2=5) (1=6), gen(workers_expect_rehire)
 lab def workers_expect_rehire 	1	"Very unlikely"		///
				2	"Unlikely"		///
				3	"Somewhat unlikely"	///
				4	"Somewhat likely"	///
				5	"Likely"		///
				6	"Very likely", replace
 lab val workers_expect_rehire workers_expect_rehire
 
 tab fQ17 workers_expect_rehire
 
 gen workers_expect_rehire_binary = (inrange(workers_expect_rehire,4,6))
 replace workers_expect_rehire_binary=. if workers_expect_rehire==.
 lab def workers_expect_rehire_binary 	0 "Unlikely"	///
					1 "Likely"	, replace
 lab val workers_expect_rehire_binary workers_expect_rehire_binary
 tab workers_expect_rehire workers_expect_rehire_binary
 
 lab def why_notapply_ui ///
	1 "Plan to file soon"										///
	2 "Voluntary left last job or retired"								///
	3 "Did not know benefits existed"								///
	4 "Did not know I was eligible or how to apply" 						///
	5 "Expect to start or return to a job soon"							///
	6 "Application was too confusing, technical or difficult to understand"				///
	7 "Tried to file by phone or internet but application system wasn't functioning" 		///
	8 "Too much like charity or welfare"								///
	9 "Worried filing might affect future job"							///
	10 "The amount of work required to file for unemployment wouldn't lead to the results I need"	///
	11 "No access to telephone, computer or internet, or application/assistance was not available in my language", replace
 lab val why_notapply_ui why_notapply_ui

 recode employer_encourage_ui (2=0)
 lab def employer_encourage_ui 	0 "No"	///
				1 "Yes", replace
 lab val employer_encourage_ui employer_encourage_ui
  
 keep 	person 			///
	gender 			///
	age 			///
	married			///
	income 			///
	college 		///
	educ 			///
	race 			///
	any_children 		///
	family			///
	nonwhite		///
	work_impact 		///
	apply_ui		///
	why_notapply_ui		///
	job_satisfaction*	///
	employer_says_rehire 	///
	employer_encourage_ui	///
	workers_expect_rehire 	///
	workers_expect_rehire_binary

tempfile survey
save `survey'
 


*******************************************************************************
* Open cleaned Homebase 2020 data
*******************************************************************************

use `workingdata'/homebase_raw_2020_update, clear
 compress


*******************************************************************************
* Clean company and location vars

 ren sthud	state
 ren stfipshud	stfips
 ren msachud	msac
 ren ind	industry
 ren company_id	firm
 ren user_id	person
 egen establishment = group(firm location_id)
 
 drop if state==.
 drop if msac==.

 gsort firm establishment person day
*unique firm
*unique establishment
*unique person

*isid firm establishment person day

 keep 	firm industry establishment person	         ///
	msac state stfips location_id 			 ///
	week day numdaysinweek  			 ///
	hours_worked level avg_hourly_wage_rate
	
********************************************************************************
* Prep


*Identify max week with at least seven days.
 sum week if numdaysinweek==7
 loc maxweek `r(max)'

 
*Keep obs with reasonable hours, those in the US, those with real stfps
 keep if inrange(hours_worked,0,20)
 keep if inrange(stfips,1,56) & stfips != 50


*Create new firm definition. Firms should be specific to industry and msa.
 egen new_firm = group(firm industry msac state), missing 	
 							
*drop firm   
*ren new_firm firm

 *isid firm week establishment person day
 keep if inrange(week,4,`maxweek')
 

/********************************************************************************
* Find firms operating in baseperiod

There are some rows where people have multiple different wages/positions. 
I delete the rows where someone has a 0 or missing wage and then just keep the 
first reported wage rate/position for now.
*******************************************************************************/

*Fix wage variable
 replace avg_hourly_wage_rate=-1 if (	avg_hourly_wage_rate==0 | ///
					avg_hourly_wage_rate==.)
 keep if hours_worked>0 & hours_worked!=.
 
*Create new variable tracking hours worked in base period by person 
 gsort firm state msac industry person day
 by firm state msac industry person: egen base_hours=total(hours_worked) if inlist(week,4,5)
 
 collapse 	(sum) 	hours_worked			///
		(max) 	dailywage=avg_hourly_wage_rate 	///
			base_hours			///
			level,				///
		by(new_firm firm week day person state msac industry)	


tempfile fulldata
save `fulldata'


*Construct firm/person/week dataset. 
*Add in the tenure and average hours variables.

 collapse 	(sum) 	hours_worked		///
		(max) 	dailywage 		///
			level,			///
		by(new_firm firm week person state msac industry)
		
tempfile firmpersonweek
save `firmpersonweek'


*Construct firm/week dataset
 gen anywork=(hours_worked>0)
 collapse 	(sum) 	hours_worked 		///
			anywork, 		///
		by(new_firm firm week state msac industry)
 isid new_firm week
 rename anywork numworkers
 compress
tempfile firmweek
save `firmweek'
 
 
*Identify base period hours
 gen baseperiod=(inlist(week,4,5))
 keep if baseperiod==1
 collapse 	(sum) 	hours_worked 			///
			numworkers,			///
		by(new_firm firm industry msac state)
 
 
*Keep only firms with at least 80 total hours in base period
 keep if hours_worked >= 80
 
 replace hours_worked	=hours_worked/2
 replace numworkers	=numworkers/2
 rename hours_worked 	firm_base_hours
 rename numworkers 	base_workers
 
 replace firm_base_hours=floor(firm_base_hours)
 
 unique new_firm
 loc baselinecount=`r(unique)'
 
tempfile baseperiod
save `baseperiod'


*Identify workers who worked both weeks in base period
use `firmpersonweek'
 
 
*Drop people who work for more than one firm
 merge m:1 new_firm firm industry msac state using `baseperiod'
 keep if _merge == 3
 gen iufirm = 0
 sort person new_firm
 by person new_firm: replace iufirm=1 if _n==1
 by person: egen nufirm=total(iufirm)
 keep if nufirm==1
 
 keep if inrange(week,4,5)
 collapse (sum) hours_worked, by(person)
 keep if hours_worked>0
 isid person 
tempfile baseworkers
save `baseworkers'
 
 
********************************************************************************
* Identify laid off workers and rehired workers
********************************************************************************

*Open firm/week dataset
use `firmpersonweek', clear

*Keep if operating in baseperiod 
 merge m:1 new_firm firm industry msac state using `baseperiod', assert(1 3) keep(3) nogen
 merge m:1 person using `baseworkers', keep(3) nogen
 
 
*Create observations, so we can record zero hours worked.
 fillin person week


 gsort person _fillin week
 foreach var in new_firm firm industry msac state level firm_base_hours {
  by person: replace `var'=`var'[1] if _fillin==1
 }
 replace hours_worked=0 if _fillin==1

 
 isid person week
 gen anywork=(hours_worked>0)

 
*Identify employent spells by person 
 egen personid=group(person)
 tsset personid week
 tsspell anywork
 
 gsort person week
 by person: egen totspell=max(_spell)
 
 
*Identify and keep firms that ever shutdown
 by person: gen always_employed=(anywork[1]==1) & (anywork[_N]==1) & (totspell==1)
 tab always_employed
 *keep if always_employed==0

 
*Identify laid off workers
 gen laidoff=((anywork==0) & (inrange(week,11,17)))
 *by person: egen ever_laidoff=max(laidoff)
 *keep if ever_laidoff==1
 
 
*Identify rehired workers
 gen rehired=(anywork==1 & inrange(week,17,`maxweek'))
 
 
 collapse 	(max)	rehired		 		///
			firm_base_hours 		///
		(sum)	totweekslaidoff=laidoff, 	///
	by(person new_firm firm msac state industry)
 
*keep if laidoff==1
 isid person new_firm firm msac state industry
 
 gen laidoff=totweekslaidoff>0
 replace rehired=. if laidoff==0
 
tempfile laidoff_rehired_tag
save `laidoff_rehired_tag'


********************************************************************************
*Restrict firm/person/day to workers at reopened firms
********************************************************************************

*Merge the reopened dataset into the full firm/person/day dataset 
use `fulldata'

 merge m:1 person new_firm firm msac state industry 				///
						using `laidoff_rehired_tag', 	///
						assert(1 3) 			///
						keep(3) 			///
						nogen
 
 
 
/*******************************************************************************
*Identify worker characteristics

We'll identify tenure-based variables over teh full 2020 calendar. 
But the other variables (wage, hours worked, level), we'll identify just using 
an extended base period - between late January and mid-March.

*******************************************************************************/
  
*First Day of Work (focusing just on first/last day right now instead of tenure)
 assert hours_worked!=.
 gen worked_today=(hours_worked>0)
 gsort new_firm person -worked_today day  
 by new_firm person: gen min_day=day[1]
 
 
*First Week of Work 
 by new_firm person: gen min_week=week[1]
 
 
*Last Day of Work 
 gsort new_firm person -worked_today -day 
 by new_firm person: gen max_day=day[1]
 
 
*Last Week of Work 
 by new_firm: gen max_week=week[1]
 
  
/*Restrict to initial employees. Exclude workers who only worked before base 
period or after shutdown. And Restrict to period before Covid for purposes of 
identifying wages, hours worked, and position.*/
 gsort new_firm person week


*Keep one observation per person and keep relevant vars
 
 collapse	(max)	dailywage			/// maximum hourly wage
			level				/// max level (position)
			max_day min_day			///
			max_week min_week		///
			rehired				///
			laidoff				///
			base_hours			///
		(sum) 	total_days=worked_today		/// number of days worked
			total_hours=hours_worked,	/// number of hours worked
		by(	new_firm firm 			///
			industry state msac 		///
			person)
	
 isid new_firm person	
 
	
*Identify level
 label def level 	1 "Employee"		///
			2 "Manager"		///
			3 "General Manager", replace
 label val level level
 
*Identify manager
 gen manager=inrange(level,2,3)
 label def manager	0 "Employee"		///
			1 "Manager", replace
 label val manager manager
 
*Create bins of hourly wages
 egen wage_binned=cut(dailywage), at(-1,0,5,10,15,20,100)
 label define wage_labels 	-1 "NA"		///
				0 "$0-5" 	///
				5 "$5-10" 	///
				10 "$10-15" 	///
				15 "$15-20" 	///
				20 "$20+" 	///
				100 "NA"	, replace
 label values wage_binned wage_labels
 replace wage_binned=100 if wage_binned==-1
 			

*Part-time vs full-time in base period
 gen status=0 if inrange(base_hours,1,40)
 replace status=1 if inrange(base_hours,41,280)
 lab def status 0 "Part-time"	///
		1 "Full-time", replace
 lab val status status

 egen total_hours_binned=cut(total_hours), at(0(100)800) ic label
 egen total_days_binned=cut(total_days), at(0(10)60) ic label
 
 lab def total_hours_binned	///
	0	"0-99"		///
	1	"100-199"	///
	2	"200-299"	///
	3	"300-399"	///
	4	"400-499"	///
	5	"500-599"	///
	6	"600-699"	///
	7	"700+", replace	
 lab val total_hours_binned total_hours_binned
 
 lab def total_days_binned	///
	0	"0-9"		///
	1	"10-19"		///
	2	"20-29"		///
	3	"30-39"		///
	4	"40-49"		///
	5	"50+", replace	
 lab val total_days_binned total_days_binned
 

*Merge in PPP and UI information
 merge m:1 state using `pppuic', assert(2 3) keep(3) nogen
 
 recode stfips 	///
		(9 23 25 33 44 50=1) 		///
		(34 36 42=2) 			///
		(18 17 26 39 55=3) 		///
		(19 20 27 29 31 38 46=4) 	///
		(10 11 12 13 24 37 45 51 54=5) 	///
		(1 21 28 47=6) 			///
		(5 22 40 48=7) 			///
		(4 8 16 35 30 49 32 56=8) 	///
		(2 6 15 41 53=9), 		///
		gen(division)
	
 lab def uic		///
	1	"125-141%"	///
	2	"141-152%"	///
	3	"152-161%"	///
	4	"161-177%", replace	
 lab val uic uic


 
/*******************************************************************************
*Merge in 2018-2019 data. Modify tenure vars.

2018 and 2019 history is created in 03a_workerhistory.do
*******************************************************************************/


*Merge in worker history information from 2018-2019
 merge 1:1 firm msac state industry person ///
		using "`results'/data/homebase_worker_history.dta", ///
		gen(merge_history)
 tab merge_history
 drop if merge_history==2
 

*Hours Worked Modification
 sort new_firm person 
 replace total_hours=total_hours+total_hours_1819

*Tenure Modification
 loc dec312019=date("12312019","MDY")
 gen start_day=min_day+`dec312019'
 replace start_day=first_day_1819 if merge_history==3
 
 gen end_day=max_day+`dec312019'
 
 format start_day	%td
 format end_day 	%td
 
 drop min_day max_day first_day_1819 last_day_1819
 
 sum start_day, detail
 loc earliest_startday=`r(min)'
 loc latest_startday=`r(max)'
 
 loc dec312017=date("12312017","MDY")
 loc dec312018=date("12312018","MDY")
 loc jan012019=date("01012019","MDY")
 loc jun302019=date("06302019","MDY")
 loc jul012019=date("07012019","MDY")
 loc dec312019=date("12312019","MDY")
 loc jan012020=date("01012020","MDY")
 loc jun302020=date("06302020","MDY")
 
 cap drop start_day_binned
 gen start_day_binned=.
 replace start_day_binned=0 if inrange(start_day,`dec312017',`dec312018')
 replace start_day_binned=1 if inrange(start_day,`jan012019',`jun302019')
 replace start_day_binned=2 if inrange(start_day,`jul012019',`dec312019')
 replace start_day_binned=3 if inrange(start_day,`jan012020',`jun302020')

 lab def start_day_binned 	0	"Before Dec 2018"	///
				1	"Jan-Jun 2019"		///
				2	"Jun-Dec 2019"		///
				3	"Since Jan 2020"	, replace
 
 lab val start_day_binned start_day_binned
 
 
********************************************************************************
*Merge in select variables from survey
********************************************************************************

 
 merge m:1 person using `survey', gen(survey_merge)
 drop if survey_merge==2
	

/*******************************************************************************
In the remaining part of the do file, we estimate a variety of logit models
and run margin analyses after those logits in order to identify whether certain 
characteritics are predictive of being laidoff or rehired.

We run 5 logits.

In the first, we use all demographics information in our survey. 
We output these results into Excel doc using put excel.

In the second, we use all the same demgoraphic info, but also incldue workers' 
expectation of being rehired. We output those into CSV file with esttab

In the third, we use job level characteristics: wage, tenure, position etc, 
along with survey info.

In the fourth, we use rehiring expctations with only state and industry fe.

In the fifth, we use state-level policies (UI and PPP) along wtih survey info,
and with and without Census divison FE.


*******************************************************************************/

********************************************************************************
* Logit #1: Survey data
********************************************************************************

putexcel set "`results'/table/alllaidoffworkers_prob_results_oct25.xlsx", mod

 foreach outcome in laidoff rehired {


 logit `outcome' 	i.gender 			///
			i.married			///
			ib2.age 			///
			ib2.educ 			///
			i.race				///
			i.family			///
			i.manager			///
			i.industry			///
			i.state				///
	if survey_merge==3				///
		, vce(cl new_firm)
		
 estimates store logit_results_`outcome'

 eststo marg_dems: margins if survey_merge==3,		///
			dydx(   gender 			///
				race			///
				age			///
				educ			///
				married			///
				manager			///
				family			///
				) 			///
			noestimcheck post atmeans 	///
			at( 	gender=1		///
				married=0		///
				race=1			///
				age=2			///
				state=5			///
				educ=2			///
				industry=7		///
				manager=0		///
				family=0		///
				)		
 

 mat coef`outcome'=e(b)
 mat se`outcome'=e(V)
 
 loc marg_n_`outcome'=string(`r(N)',"%9.0fc")

}

 count if laidoff==0 & survey_merge==3
 loc n=string(`r(N)',"%9.0fc")
 putexcel B31 ="`n'"
 
 count if laidoff==1 & survey_merge==3
 loc n=string(`r(N)',"%9.0fc")
 putexcel C31 ="`n'"

 count if rehired==1 & survey_merge==3
 loc n=string(`r(N)',"%9.0fc")
 putexcel D31 ="`n'"
 
 putexcel F31 = "`marg_n_laidoff'"
 putexcel H31 = "`marg_n_rehired'"

 mat coef = coeflaidoff\coefrehired

 cap program drop excelresults
 program def excelresults 
 
 syntax, 	dem(varname)		///
		value(real)		///
		row(real)		///
		matcell(real)
 
 
*notlaidoff
 count if `dem'!=. & laidoff==0
 loc tot=`r(N)' 
 
 count if `dem'==`value' & laidoff==0
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel B`row'="`avg'"
 
*laidoff
 count if `dem'!=. & laidoff==1
 loc tot=`r(N)' 
 
 count if `dem'==`value' & laidoff==1
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel C`row'="`avg'"
 
 
*rehired
 count if `dem'!=. & rehired==1
 loc tot=`r(N)' 
 
 count if `dem'==`value' & rehired==1
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel D`row'="`avg'"
 
  
 loc coeflaidoff=(abs(coef[1,`matcell']))
 loc coefrehired=(abs(coef[2,`matcell']))
 
 loc selaidoff=(sqrt(selaidoff[`matcell',`matcell']))
 loc serehired=(sqrt(serehired[`matcell',`matcell']))
 
 loc strcoeflaidoff=string(coef[1,`matcell'],"%9.3f")
 loc strcoefrehired=string(coef[2,`matcell'],"%9.3f")
  

 if `coeflaidoff' >= (2*`selaidoff') {
	putexcel F`row'="`strcoeflaidoff'", bold
 }
 else {
	putexcel F`row'="`strcoeflaidoff'"
 }
 
 if `coefrehired' >= (2*`serehired') {
	putexcel H`row'="`strcoefrehired'", bold
 }
 else {
	putexcel H`row'="`strcoefrehired'"
 }
 
 
 loc strselaidoff=string((sqrt(selaidoff[`matcell',`matcell'])),"%9.3f")
 loc strserehired=string((sqrt(serehired[`matcell',`matcell'])),"%9.3f")
 
 putexcel G`row'="(`strselaidoff')"
 putexcel I`row'="(`strserehired')"

end


***Race
*Black
 excelresults, 	dem(race) 		///
		value(5) 		///
		row(17)			///
		matcell(19)

*Asian
 excelresults, 	dem(race) 		///
		value(2) 		///
		row(18)			///
		matcell(16)

*AI/AN
 excelresults, 	dem(race) 		///
		value(3) 		///
		row(19)			///
		matcell(18)

*Hispanic
 excelresults, 	dem(race) 		///
		value(4) 		///
		row(20)			///
		matcell(17)
		
*** AGE
*18-25				
 excelresults, 	dem(age) 		///
		value(1) 		///
		row(5)			///
		matcell(6)
		
*38-49	
 excelresults, 	dem(age) 		///
		value(3) 		///
		row(7)			///
		matcell(8)
		
		
*50-64		
 excelresults, 	dem(age) 		///
		value(4) 		///
		row(8)			///
		matcell(9)
		
*65+
 excelresults, 	dem(age) 		///
		value(5) 		///
		row(9)			///
		matcell(10)
 

***EDUCATION
*less than hs			
 excelresults, 	dem(educ) 		///
		value(1) 		///
		row(11)			///
		matcell(11)
		
*some college		
 excelresults, 	dem(educ) 		///
		value(3) 		///
		row(13)			///
		matcell(13)
		
*BA plus
 excelresults, 	dem(educ) 		///
		value(4) 		///
		row(14)			///
		matcell(14)
 
 
 
***OTHER DEMOGRAPHICS
*married
 excelresults, 	dem(married) 		///
		value(1) 		///
		row(22)			///
		matcell(5)
 
*female
 excelresults, 	dem(gender) 		///
		value(2) 		///
		row(23)			///
		matcell(2)

		
*presence of children | male
 excelresults, 	dem(family) 	///
		value(1) 		///
		row(24)			///
		matcell(21)
		
*presence of children | female
 excelresults, 	dem(family) 	///
		value(2) 		///
		row(25)			///
		matcell(22)



*manager
 excelresults, 	dem(manager)		///
		value(1)		///
		row(27)			///
		matcell(24)
 
 
 
 cap program drop excelresults
 program def excelresults 
 
 syntax, 	dem(varname)		///
		value(real)		///
		row(real)		
 
*notlaidoff
 count if `dem'!=. & laidoff==0
 loc tot=`r(N)' 
 
 count if `dem'==`value' & laidoff==0
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel B`row'="`avg'"
 
*laidoff
 count if `dem'!=. & laidoff==1
 loc tot=`r(N)' 
 
 count if `dem'==`value' & laidoff==1
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel C`row'="`avg'"
 
 
*rehired
 count if `dem'!=. & rehired==1
 loc tot=`r(N)' 
 
 count if `dem'==`value' & rehired==1
 loc demcount=`r(N)'
 
 loc avg=string(`demcount'/`tot',"%9.2f")
 putexcel D`row'="`avg'"
 
 end
 
 excelresults, dem(race)	value(1) row(16)
 excelresults, dem(age)		value(2) row(6)
 excelresults, dem(educ)	value(2) row(12)

 
 
********************************************************************************
*Logit #2: HB survey data including rehire expectation
********************************************************************************
 
 logit rehired 		i.gender 			///
			i.married			///
			ib2.age 			///
			ib2.educ 			///
			i.race				///
			i.family			///
			i.manager			///
			i.industry			///
			i.state				///
			i.workers_expect_rehire_binary	///
	if survey_merge==3				///
		, vce(cl new_firm)
		
 estimates store logit_rehired_expect
 
 eststo marg_dems: margins if survey_merge==3,		///
			dydx(   gender 			///
				race			///
				age			///
				educ			///
				married			///
				manager			///
				family			///
				workers_expect_rehire_binary	///
				) 			///
			noestimcheck post atmeans 	///
			at( 	gender=1		///
				married=0		///
				race=1			///
				age=2			///
				state=5			///
				educ=2			///
				industry=7		///
				manager=0		///
				family=0		///
				workers_expect_rehire_binary=0	///
				)		
 
 mat coefrehired=e(b)
 mat serehired=e(V)
 
 loc marg_n_rehired=string(`r(N)',"%9.0fc")
 
 count if laidoff==1 & survey_merge==3 & workers_expect_rehire_binary!=.
 loc n=string(`r(N)',"%9.0fc")
 putexcel J31 ="`n'"

 cap program drop excelresults
 program def excelresults 
 
 syntax, 	row(real)		///
		matcell(real)
 
 loc coefrehired=(abs(coefrehired[1,`matcell']))
 loc serehired=(sqrt(serehired[`matcell',`matcell']))
 loc strcoefrehired=string(coefrehired[1,`matcell'],"%9.3f")
 loc strserehired=string((sqrt(serehired[`matcell',`matcell'])),"%9.3f")

 if `coefrehired' >= (2*`serehired') {
	putexcel J`row'="`strcoefrehired'", bold
 }
 else {
	putexcel J`row'="`strcoefrehired'"
 }
 
 putexcel K`row'="(`strserehired')"

end


***Race
*Black
 excelresults, 	row(17)			///
		matcell(19)

*Asian
 excelresults, 	row(18)			///
		matcell(16)

*AI/AN
 excelresults, 	row(19)			///
		matcell(18)

*Hispanic
 excelresults, 	row(20)			///
		matcell(17)
		
*** AGE
*18-25				
 excelresults, 	row(5)			///
		matcell(6)
		
*38-49	
 excelresults, 	row(7)			///
		matcell(8)
		
		
*50-64		
 excelresults, 	row(8)			///
		matcell(9)
		
*65+
 excelresults, 	row(9)			///
		matcell(10)
 

***EDUCATION
*less than hs			
 excelresults,  row(11)			///
		matcell(11)
		
*some college		
 excelresults, 	row(13)			///
		matcell(13)
		
*BA plus
 excelresults, 	row(14)			///
		matcell(14)
 
 
***OTHER DEMOGRAPHICS
*married
 excelresults, 	row(22)			///
		matcell(5)
 
*female
 excelresults, 	row(23)			///
		matcell(2)

		
*presence of children | male
 excelresults, 	row(24)			///
		matcell(21)
		
*presence of children | female
 excelresults, 	row(25)			///
		matcell(22)


*manager
 excelresults, 	row(27)			///
		matcell(24)
 
*expected to be rehired
 excelresults,  row(29)			///
		matcell(26)
 
 

		
	
********************************************************************************
*Logit #3: Demographics plus policies (PPP and UIC) with and without division FE
********************************************************************************	


 foreach outcome in laidoff rehired {

  logit `outcome' 	i.gender 			///
			i.married			///
			ib2.age 			///
			ib2.educ 			///
			i.race				///
			i.family			///
			i.manager			///
			i.industry			///
			i.loanc				///
			i.uic				///
	if survey_merge==3				///
		, vce(cl new_firm)
		
 estimates store logit_results_`outcome'

 eststo marg_dems: margins if survey_merge==3,		///
			dydx(   gender 			///
				race			///
				age			///
				educ			///
				married			///
				manager			///
				family			///
				uic			///
				loanc			///
				) 			///
			noestimcheck post atmeans 	///
			at( 	gender=1		///
				married=0		///
				race=1			///
				age=2			///
				educ=2			///
				industry=7		///
				manager=0		///
				family=0		///
				)		
 
 estimates save `results'/estimates/margins_policies_`outcome'_nodiv_oct25, replace
 
 

 logit `outcome' 	i.gender 			///
			i.married			///
			ib2.age 			///
			ib2.educ 			///
			i.race				///
			i.family			///
			i.manager			///
			i.industry			///
			i.division			///
			i.loanc				///
			i.uic				///
	if survey_merge==3				///
		, vce(cl new_firm)
		
 estimates store logit_results_`outcome'

 eststo marg_dems: margins if survey_merge==3,		///
			dydx(   gender 			///
				race			///
				age			///
				educ			///
				married			///
				manager			///
				family			///
				uic			///
				loanc			///
				) 			///
			noestimcheck post atmeans 	///
			at( 	gender=1		///
				married=0		///
				race=1			///
				age=2			///
				division=1		///
				educ=2			///
				industry=7		///
				manager=0		///
				family=0		///
				)		
 
 estimates save `results'/estimates/margins_policies_`outcome'_div_oct25, replace
 
}


		
********************************************************************************
*Merged CPS and HB policy output
********************************************************************************

 eststo clear
 foreach type in nodiv div {
 foreach outcome in laidoff rehired {
  estimates use `results'/estimates/margins_policies_`outcome'_`type'_oct25
  estimates store hb`outcome'`type'_oct25
 }
 }
 

*Create table with estimates for PPP and UI policies
  esttab 	hblaidoffnodiv_oct25 	///
		hblaidoffdiv_oct25 	///
		hbrehirednodiv_oct25 	///
		hbrehireddiv_oct25	///
	using "`results'/table/margins_policies_coefs_oct25.csv", ///
	keep(*.loanc *.uic) replace nostar b(%9.3fc) se(%9.3fc)
 

********************************************************************************

log close workerlevel_prob_log
********************************************************************************