/*******************************************************************************
The following .do file identifies how hours worked for our baseline firms change 
each week relative to the baseline period. We also decompose hours losses into 
shutdowns (where firms laid off all workers), layoffs at still open firms,
and hours reductions by retained workers.

*******************************************************************************/

cap log close decomposition_log

local date: display %tdCCYY-NN-DD =daily("`c(current_date)'", "DMY")
log using "/accounts/projects/jrothst/homebase/data/bpea_replication_archive/logs/01_decomposition_`date'.log", ///
	replace name(decomposition_log)

set more off
set varabbrev off

loc dir 	"/accounts/projects/jrothst/homebase/data/bpea_replication_archive/"
loc code	"`dir'/code/"
loc results	"`dir'/results/"
loc sourcedata	"`dir'/data_raw/"
loc workingdata	"`dir'/data_clean/"


*******************************************************************************/

*******************************************************************************
* Open cleaned file data
*******************************************************************************

* Compute total hours worked and total headcount by week
use `workingdata'/homebase_raw_2020_update, clear
 compress

*******************************************************************************
* Clean company and location vars
*******************************************************************************
 ren sthud	state
 ren stfipshud	stfips
 ren msachud	msac
 ren ind	industry
 ren company_id	firm
 ren user_id	person
 egen establishment = group(firm location_id)
 
 drop if state==.
 drop if msac==.

 gsort firm establishment person day
*unique firm
*unique establishment
*unique person
*isid firm establishment person day


 keep 	firm establishment person	///
	industry msac state stfips 	///
	week day numdaysinweek		///
	hours_worked 


********************************************************************************
* Prep
********************************************************************************

*Identify max week with at least seven days.
 sum week if numdaysinweek==7
 loc maxweek `r(max)'

*Keep obs with reasonable hours, those in the US, those with real stfps
 keep if inrange(hours_worked,0,20)
 drop if inlist(stfips,98,99)

*Create new firm definition. Firms are specific to industry and MSA.
 egen new_firm = group(firm industry msac state), missing 				
 drop firm industry  
 ren new_firm firm
 

*save `workingdata'/tmp_0603_decompfile, replace

*Relabel weeks
lab def week	1 "Jan 4"	///
		2 "Jan 11"	///
		3 "Jan 18"	///
		4 "Jan 25"	///
		5 "Feb 1"	///
		6 "Feb 8"	///
		7 "Feb 15"	///
		8 "Feb 22"	///
		9 "Feb 29"	///
		10 "Mar 7"	///
		11 "Mar 14"	///
		12 "Mar 21"	///
		13 "Mar 28"	///
		14 "Apr 4"	///
		15 "Apr 11"	///
		16 "Apr 18"	///
		17 "Apr 25"	///
		18 "May 2"	///
		19 "May 9"	///
		20 "May 16"	///
		21 "May 23"	///
		22 "May 30"	///
		23 "Jun 6"	///
		24 "Jun 13"	///
		25 "Jun 20"	///
		26 "Jun 27"	///
		27 "Jul 4"	///
		28 "Jul 11"	///
		29 "Jul 18"	///
		30 "Jul 25"	///
		31 "Aug 1"	///
		32 "Aug 8"	///
		33 "Aug 15", replace
lab val week week

local weeklab: label week `maxweek'
			


********************************************************************************
* Analysis
********************************************************************************


*Sum hours worked by firm/person/week
 collapse (sum) hours_worked, by(firm week person state msac)
 isid firm week person
*Create anywork indicator
 gen anywork=(hours_worked>0)
*Total hours worked and total employees at each firm each week
 collapse (sum) hours_worked anywork, by(firm week state msac)
 isid firm week
 rename anywork numworkers
 compress
tempfile firmweek
save `firmweek'

 
*Base period total hours
 use `firmweek', clear
 gen baseperiod=(inlist(week,4,5))
 keep if baseperiod==1
 collapse (sum) hours_worked numworkers, by(firm)
 
*Keep only firms with at least 80 total hours in base period
 keep if hours_worked >= 80
 
 replace hours_worked=hours_worked/2
 replace numworkers=numworkers/2
 rename hours_worked base_hours
 rename numworkers base_workers
tempfile baseperiod
save `baseperiod'



*Merge basehours to firmweek.
*Keep baseline firms
use `firmweek', clear

 gen firmopen=(numworkers>0)
 keep firm state msac week firmopen numworkers hours_worked
 merge m:1 firm using `baseperiod', 	assert(1 3) 	///
					nogen 		///
					keep(3) 	///
					keepusing(firm base_hours base_workers)

*Fill in for all firm weeks
 fillin firm week
 
 replace hours_worked=0 if _fillin==1
 replace numworkers=0 if _fillin==1
 replace firmopen=0 if _fillin==1
 
 gsort firm _fillin
 by firm: replace state=state[1] if _fillin==1
 by firm: replace msac=msac[1] if _fillin==1
 by firm: replace base_hours=base_hours[1] if _fillin==1
 by firm: replace base_workers=base_workers[1] if _fillin==1
 
 
*Identify four sources of hours changes
 gsort firm week
 
*0. Change in number of workers
 gen delta_workers=(numworkers-base_workers)
 
*1. Fewer workers * their share of base hours
 gen fewer_workers=	(base_hours*			///
			(delta_workers/base_workers)*	///
			(delta_workers<0)*		///
			firmopen)
			
*2. More workers * their share of base hours			
 gen more_workers=	base_hours*			///
			(delta_workers/base_workers)*	///
			(delta_workers>0)*		///
			firmopen

*3. Shutdowns
 gen closing=(base_hours*(firmopen==0))*-1
 
*4. Retained workers lose hours (ie. change in avg hours worked by retained workers)
 gen change_avghrs=	((hours_worked/numworkers)-(base_hours/base_workers))*	///
			numworkers*						///
			firmopen
 assert firmopen==0 if change_avghrs==.
 replace change_avghrs=0 if firmopen==0 & change_avghrs==.

*Identify actual total hours changes
 gen change_hours=hours_worked-base_hours
 
*Ensure four sources of hours changes = total hours changes
 gen difference	= change_hours-				///
				(closing+		///
				(fewer_workers)+	///
				more_workers+		///
				change_avghrs)
 
 assert inrange(difference,-1,1)
 
 
* Sum up all forms of hours change
 collapse (sum) base_hours 	///
		closing 	///
		fewer_workers 	///
		more_workers 	///
		change_avghrs, by(week)
		
*Net difference in hours change from change in workers
 gen change_workers = fewer_workers+more_workers


*Create shares of hours change due to each source
 sum base_hours
 local avgbase=r(mean)
 
 foreach v in closing change_workers change_avghrs {
  gen indx_`v'=(`v'/`avgbase')*100
 }
 
local weeklab: label week `maxweek'

gen bar1length=abs(indx_change_avghrs)
gen bar1top=max(0,indx_change_avghrs)
gen bar1bottom=min(0,indx_change_avghrs)

gen bar2length=abs(indx_change_workers)
gen bar2top=min(bar1bottom, 0) if indx_change_workers<0
gen bar2bottom=bar2top-bar2length if indx_change_workers<0
replace bar2bottom=bar1top if indx_change_workers>0
replace bar2top=bar2bottom+bar2length if indx_change_workers>0

assert indx_closing<=0
gen bar3top=min(bar1bottom, bar2bottom)
gen bar3bottom=bar3top+indx_closing

gen total=indx_change_avghrs+indx_change_workers+indx_closing

keep if inrange(week,6,`maxweek')

twoway	(rbar bar1bottom bar1top week 						///
				if ~inlist(week,16,22,27),			///
				color("0 50 98") barwidth(0.7) fi(inten80)) 	///
	(rbar bar2bottom bar2top week						///
				if ~inlist(week,16,22,27),			///
				color("59 126 161") barwidth(0.7) fi(inten80)) 	///
	(rbar bar3bottom bar3top week						///
				if ~inlist(week,16,22,27),			///
				color("196 130 14") barwidth(0.7) fi(inten80)) 	///
	(rbar bar1bottom bar1top week 						///
				if inlist(week,16,22,27),			///
				color("0 50 98") barwidth(0.7) fi(inten40)) 	///
	(rbar bar2bottom bar2top week						///
				if inlist(week,16,22,27),			///
				color("59 126 161") barwidth(0.7) fi(inten40)) 	///
	(rbar bar3bottom bar3top week						///
				if inlist(week,16,22,27),			///
				color("196 130 14") barwidth(0.7) fi(inten40)) 	///	
	(scatter total week, 							///
				msymbol(+) msize(medium) mcolor(gs4)), 	///
	ylab(-60(20)8, labs(small) grid gstyle(minor) ) 			///
	yti(	"Percent change in hours attributable"				///
		"to each form of contraction", size(medsmall))			///
	yscale(titlegap(*5)) 							/// 
	yline(0, lc(red) lw(thin) lp(dot))					///
	xti(" ", size(tiny))	///
	xlab( 	6 "Feb 8"	///
		8 "Feb 22"	///
		10 "Mar 7"	///
		12 "Mar 21"	///
		14 "Apr 4"	///
		16 "Apr 18"	///
		18 "May 2"	///
		20 "May 16"	///
		22 "May 30"	///
		24 "Jun 13"	///
		26 "Jun 27"	///
		28 "Jul 11", notick labs(small))				///
	legend(off)								///
	scheme(s1color)								///
	/*note("Data updated through `weeklab'", size(vsmall))*/		///
 text(-24 10 "{bf:Layoffs}", color("59 126 161"))				///
 text(-42 24 "{bf:Shutdowns}", color("196 130 14"))				///
 text(3 16 "{bf:Hours cuts}", color("0 50 98"))					///
 text(-58 24.9 " {bf:+} indicates net percent change in hours.", size(vsmall))	///
 text(-60 23.6 "Lighly shaded bars represent weeks including holidays.", size(vsmall))

gr save `results'/figures/decomposition_hoursreduction_byweek_withtextlabel, replace
gr export "`results'/figures/decomposition_hoursreduction_byweek_withtextlabel.png", replace


 keep indx_change_avghrs indx_closing indx_change_workers week
save "`results'/data/decomposition.dta", replace


********************************************************************************

log close decomposition_log
********************************************************************************
