function [out] = ...
    ramsey_unconstrained_K_comp(g, params)
% This function computes the solution of the unconstrained Ramsey problem
% and main equilibrium objects.
% Inputs: labor wedge, government spending, index guess
out.params = params;
out.g = g;

index_guess = params.index_guess;
indices = params.indices;
grid_ak = (params.A) ^ ((1 - params.lambda)/params.lambda) * ...
    params.grid_ak_ex_A;
grid_al = params.grid_al;
lambda = params.lambda;
B = params.B;
elast_ell = params.elast_ell;
D = params.D;
elast_k = params.elast_k;
varrho = params.varrho;
varrho_k = params.varrho_k;
maxIter = params.maxIter;
grid_theta = params.grid_theta;
alphaa = params.alpha;
delta = params.delta;
E = params.E;

%% Method 1: Iterate on task index
theta_index = index_guess;
start_ell = params.guess_l;
start_K = params.guess_k;
index_series = index_guess;
stop = 1;
iter = 0;

while iter<maxIter && stop>0
    %Step 1: Solve system for given taxes and guess
    options = optimset('Display','off');
    obj = @(x) objective_planner_KC(exp(x), theta_index, params); % planner's objective
    ic = @(x) implementability_condition_KC(exp(x), varrho, g, theta_index, params); % constraint
    [x_sol,~,~,~,lambda_mult,~,~] = ...
        fmincon(obj,log([start_ell start_K]),[],[],[],[],[], [], ic,options); 
    ell_sol = exp(x_sol(1));
    K_sol = exp(x_sol(2));


    opts_min_grid = struct();
    
    fun = @(theta_ind) - output_raw_K_comp(ell_sol, K_sol, theta_ind, params);
    [new_index, ~] = fmin_grid(fun, theta_index, grid_theta, opts_min_grid);
    y = - fun(new_index);

    %Step 3 iterate
    iter = iter+1;
    index_series = [index_series new_index];
    start_ell = ell_sol;
    start_K = K_sol;
    stop = abs(theta_index-new_index);
    theta_index = new_index;
end

if iter == maxIter
    warning('Maximum number of iterations reached, check index_series')
end



%Assign results
ell = ell_sol;
K = K_sol;
k = allocate_K(ell, K, theta_index, params);

%Infer taxes and multipliers
ak = grid_ak(theta_index);
al = grid_al(theta_index);

r_firms = ak * (y/k)^(1/lambda);
w_firms = (y - r_firms * K) / ell;

tau_ell = 1 - (B * ell^(elast_ell)) / ((1-varrho) * w_firms);
tau_k = 1 - (E * D * K^(elast_k)) / ((1-varrho_k) * (r_firms - delta));

revenue = tau_ell * ell * w_firms + ...
    tau_k * K * (r_firms - delta);


mu_alt = lambda_mult.ineqnonlin; % value of the multipliers from fmincon

% Sanity check that the solutions for tau_ell and tau_k are correct:
% compute multipliers inverting their formulas for mu
mu_ell = (tau_ell/(1-tau_ell) + varrho)/(elast_ell - (tau_ell/(1-tau_ell))); %ic multiplier obtained inverting the formula for tau^ell

mu_k = (tau_k/(1-tau_k))/(elast_k-(tau_k/(1-tau_k))); % value of the multiplier obtained using the opitmal capital taxes

% Compute labor share, rental rate, wages
labor_share = ell * w_firms / y;


%% PLANNER'S OBJECTIVE
    function objective = objective_planner_KC(x, theta_index, params)
    % the planner chooses (k, ell, theta)

    elll = x(1);
    kk = x(2);
    yy = output_raw_K_comp(elll, kk, theta_index, params);

    %Computes minus consumption
    objective = B * elll^(1 + elast_ell)/( 1 + elast_ell) + ...
        D * E * ( kk^(1+elast_k) / (1+elast_k))  - (yy  - delta * kk);
    end

%% IMPLEMENTABILITY CONDITION

    function [c , ceq] = ...
            implementability_condition_KC(x, varrho, g, theta_index, params)

    elll = x(1);
    kk = x(2);
    yy = output_raw_K_comp(elll, kk, theta_index, params);

    %Computes IC: government BC + optimal factor allocations
    c = - (yy - delta * kk) + B * elll^(1 + elast_ell) / (1 - varrho) +...
        D *(E *  kk^(1 + elast_k))/(1 - varrho_k)  + g ;
    ceq = [];
    end


out.ell = ell;
out.k = k;
out.K = K;
out.y = y;
out.theta_index = theta_index;
out.tau_k = tau_k;
out.tau_ell = tau_ell;
out.revenue = revenue;
out.obj_value = objective_planner_KC([ell K],  theta_index, params);
out.constr_resid = implementability_condition_KC([ell K], varrho, g, theta_index, params);


out.labor_share = labor_share;
out.r = r_firms;
out.w = w_firms;
out.index_series = index_series;

out.mu_ell = mu_ell;
out.mu_k = mu_k;
out.mu = mu_alt;


out.theta = params.grid_theta(theta_index);
% out.implicit_tax= ...
%     1 - (r_firms / w_firms) * ...
%     params.grid_phi_l(theta_index) / params.grid_phi_k(theta_index);
out.implicit_tax= ...
    1 - (r_firms / ( (w_firms/alphaa)^alphaa * (r_firms/(1 - alphaa))^(1 - alphaa) )) * ...
    params.grid_phi_l(theta_index) / params.grid_phi_k(theta_index);


out.welfare = (out.y + (1- delta) * out.K) - ...
    params.B * (out.ell^ ( 1 + params.elast_ell)) / ( 1 + params.elast_ell) - ...
    params.D * ( params.E * out.K ^ ( 1 + params.elast_k)) / ( 1 + params.elast_k) - out.K;
out.net_k_share =  (r_firms - delta ) * out.K / (out.y - delta * out.K) ;   
out.k_ratio = out.K/out.y; 
out.net_output = out.y - delta * out.K; 
% out.MRTS = compute_MRTS(ell, k, theta_index, params);
% out.MRTS_prime = compute_MRTS(ell, 1.01*k, theta_index, params);
% out.el_subs = - ...
%     (( log(out.MRTS_prime) - log(out.MRTS) )/...
%     log(1.01) ) ^ (-1);


end