function [params] = setParams(override)
% This function produces a structure containing a default calibration as in
% the default specified below, overriding parameters if the input override
% is provided.

% DEFAULTS
% TASKS
params.N_tasks=10000; % grid points for tasks
% Factor advantage parameters
params.A = 20; % default scaling parameter for the comparative advantage
params.zeta = 2; % exponent in the comparative advantage term
% Scaling exponent in the comparative advantage term above
params.nu = 1;     % exponent in the labor absolute advantage is zeta * nu, 
% regulates how comp. adv. relates to absolute
params.lambda = 0.5;      %elasticity of subs across tasks
% depreciation rate: set at the average for equipment in the economy over
% 1981-2017
params.delta = 0.055;
% scaling of utility function
params.E = 1;
% DISTORTIONS
params.varrho = 0.15;    % Labor ex-taxes: moves in .15 - .25
params.varrho_k = 0;
params.tau_current_labor = 0.255; % average labor tax
params.tau_current_capital = 0.1; % average capital wedge on equipment
params.tau_bar = params.tau_current_labor; %lower bound on labor taxation

% FACTOR SUPPLIES

params.elast_ell = 1/.7;    % Labor el. 
params.D = 1;            %Normalized since units of capital free
params.elast_k = 1/0.65;    % Capital elasticity

% Iterations for equilibrium
params.maxIter = 100;
params.min_theta = 1e-5;

% share of labor in complementary tasks
params.alpha = .75;

% Replace fields with custom override
if nargin < 1 % do nothing
else % replace parameter fields
    ovveride_fields = fieldnames(override);
    for ifield = 1: length(ovveride_fields)
        ffield = ovveride_fields{ifield};
        if isfield(params, ffield)
            params.(ffield) = override.(ffield);
        else
            error(['field ' ffield ' not in params, check spelling'])
        end
    end
    
end
params.index_guess = params.N_tasks/2; % guess for the optimal division of tasks

% PARAMETERS COMPUTED BASED ON THE ABOVE
% FACTOR SUPPLIES
params.B = 1 - params.varrho;     %cancel effect of wedge on quantities
% GRIDS
% GRID for theta
params.grid_theta = linspace(params.min_theta, 1, params.N_tasks);
params.indices = linspace(1, params.N_tasks, params.N_tasks);

%Note that this needs to be normalized using the number of tasks, otherwise things explode
params.grid_phi_l = ...
    params.N_tasks ^ (1/(1 - params.lambda)) * ...
    params.grid_theta .^ (params.nu * params.zeta); 
% grid for phi_k_theta multiplied by A
params.grid_phi_k_ex_A = params.grid_phi_l./...
    (params.grid_theta .^ params.zeta); 
params.grid_phi_k =  params.grid_phi_k_ex_A/ params.A;

% Factor multiplying k or l; tasks up to "index" in grid_ak are produced using
% capital and those above are produced using labor
% grid for ak divided by A ^ ((1 - lambda)/lambda)
params.grid_ak_ex_A = cumsum(...
    params.grid_phi_k_ex_A .^ (params.lambda - 1) ).^(1/params.lambda);
params.grid_al = cumsum(...
    params.grid_phi_l.^(params.lambda - 1), 'reverse').^(1/params.lambda);

% params.grid_ak = (params.A) ^ ((1 - params.lambda)/params.lambda) * ...
%     params.grid_ak_ex_A;

% %for reference: labor reform constraint binds if mu<threshold mu :
params.threshold_mu = ...
    (params.tau_bar/(1 - params.tau_bar) + params.varrho) / ...
    (params.elast_ell - params.tau_bar / (1 - params.tau_bar));
params.threshold_tau_k = (params.varrho + params.tau_current_labor / ...
    (1 - params.tau_current_labor) ) *...
    params.elast_k / (...
    (params.varrho + params.tau_current_labor / ...
    (1 - params.tau_current_labor) ) * params.elast_k  + params.varrho ...
    + params.elast_ell);


end