#------------------------------------------------------------------------------#
# File:        run.hlw.ae.R
#
# Description: This the main file for HLW, which rund the HLW procedure with Adavanced Economies data
#              (1) Prepares data to be used in estimation
#              (2) Runs the three-stage HLW estimation for each economy
#              (3) Saves output.
#------------------------------------------------------------------------------#
rm(list=ls())  

#------------------------------------------------------------------------------#
# Prepare data to be used in estimation.
#
# Output will be saved in the inputData folder.
#
# Set the data start and end dates manually in each prepare.rstar.data file
#------------------------------------------------------------------------------#
#source("prepare.rstar.data.us.R")
#source("prepare.rstar.data.ca.R")
#source("prepare.rstar.data.ea.R")
#source("prepare.rstar.data.uk.R")

#------------------------------------------------------------------------------#
# Load required packages and source all programs to be used in HLW estimation.
#------------------------------------------------------------------------------#
if (!require("tis")) {install.packages("tis"); library("tis")} # Time series package
if (!require("mFilter")) {install.packages("mFilter"); library("mFilter")} # HP filter
if (!require("nloptr")) {install.packages("nloptr"); library("nloptr")} # Optimization
if (!require("seasonal")) {install.packages("seasonal"); library("seasonal")} # SA

# Source all R programs; see code guide for details of each
source("calculate.covariance.R")
source("format.output.R")
source("kalman.log.likelihood.R")
source("kalman.standard.errors.R")
source("kalman.states.R")
source("kalman.states.wrapper.R")
source("log.likelihood.wrapper.R")
source("median.unbiased.estimator.stage1.R") 
source("median.unbiased.estimator.stage2.R")
source("rstar.stage1.R")
source("rstar.stage2.R")
source("rstar.stage3.R")
source("run.hlw.estimation.R") 
source("unpack.parameters.stage1.R") 
source("unpack.parameters.stage2.R") 
source("unpack.parameters.stage3.R") 
source("utilities.R")


#------------------------------------------------------------------------------#
# Define variables
#------------------------------------------------------------------------------#
# Upper bound on a_3 parameter (slope of the IS curve)
a3.constraint <- -0.0025

# Lower bound on b_2 parameter (slope of the Phillips curve)
b2.constraint <- 0.025

# Set the start and end dates of the estimation sample (format is c(year,quarter))
sample.start <- c(1981,1) 
sample.end   <- c(2017,2)

# Set the estimation sample start date for the Euro Area 
# ea.sample.start <- c(1972,1)

# The estimation process uses data beginning 4 quarters prior to the sample start
data.start    <- shiftQuarter(sample.start,-4)
# ea.data.start <- shiftQuarter(ea.sample.start,-4)

# Set start index for y
g.pot.start.index <- 1 + ti(shiftQuarter(sample.start,-3),'quarterly')-ti(data.start,'quarterly')

# Set column names for CSV output
output.col.names <- c("Date","rstar","g","z","output gap","","All results are output from the Stage 3 model.",rep("",8),"Standard Errors","Date","y*","r*","g","","rrgap")

# Set number of iterations for Monte Carlo standard error procedure
niter <- 5000

# Because the MC standard error procedure is time consuming, we include a run switch
# Set run.se to TRUE to run the procedure
run.se <- FALSE

#------------------------------------------------------------------------------#
# World: Read in data, run estimation, and save output
#------------------------------------------------------------------------------#
# Read in the input data
ae.data <- read.table("inputData/rstar.data.ae.king.short.csv",
                      sep = ',', na.strings = ".", header=TRUE, stringsAsFactors=FALSE)

ae.log.output             <- ae.data$gdp.log
ae.inflation              <- ae.data$inflation
ae.inflation.expectations <- ae.data$inflation.expectations
ae.nominal.interest.rate  <- ae.data$interest
ae.real.interest.rate     <- ae.nominal.interest.rate - ae.inflation.expectations

#Seasonally adjust output and inflation

ts.output <- ts(ae.log.output, frequency = 4, start = c(1980, 1))
out.s <- seas(ts.output, x11="")
ae.log.output <- as.numeric(final(out.s))

ts.inflation <- ts(ae.inflation, frequency = 4, start = c(1980, 1))
inf.s <- seas(ts.inflation, x11="")
ae.inflation <- as.numeric(final(inf.s))

# Adjust inflation for the price of oil (note that in the global sample high inflation driven not by oil but by EMEs inflating away - see IMF WEO Ch4 2001)
#x.oil <- matrix(ae.oil)
#y.oil <- matrix(ae.inflation)
#b.oil <- solve(t(x.oil) %*% x.oil, t(x.oil) %*% y.oil)
#inf.fitted = x.oil %*% b.oil
#inf.no.oil <- y.oil - x.oil %*% b.oil

# Run HLW estimation for the World
ae.estimation <- run.hlw.estimation(ae.log.output, ae.inflation, ae.real.interest.rate, ae.nominal.interest.rate,
                                    a3.constraint = a3.constraint, b2.constraint = b2.constraint, run.se = run.se)

# One-sided (filtered) estimates
one.sided.est.ae <- cbind(ae.estimation$out.stage3$rstar.filtered,
                          ae.estimation$out.stage3$trend.filtered,
                          ae.estimation$out.stage3$z.filtered,
                          ae.estimation$out.stage3$output.gap.filtered)

# Save one-sided estimates to CSV
write.table(one.sided.est.ae, 'output/one.sided.est.ae.king.short.csv', row.names = FALSE, col.names = c("rstar","g","z","output gap"), quote = FALSE, sep = ',', na = ".")

# Save output to CSV
output.ae <- format.output(ae.estimation, one.sided.est.ae, ae.real.interest.rate, sample.start, sample.end, run.se = run.se)
write.table(output.ae, 'output/output.ae.king.short.csv', col.names = output.col.names, quote=FALSE, row.names=FALSE, sep = ',', na = '')
