/*This do file contains replication code to produce figures showing expenditures
by poverty status (Figures 6, 7, 11, and 13 from
Hoynes and Schanzenbach (2018)). The called data is first created by running
create_data.do (See readme for additional details):
*/

*Prelims
clear all
set more off, perm
cap log close 

*Set directories and log file
cap cd "./replication/"

log using "./log/create_fig_poverty.log", replace


********************************************************************************
********************************************************************************
*Figures 6 -> on use calculations from create_data.do file
*********************************************************************************
********************************************************************************
*********************************************************************************
*Collect data from previous calculations
*********************************************************************************

*SNAP
import excel "./out/timeseries.xlsx", clear firstrow sheet("SNAPatti_spm") 
	keep year totfs*
	reshape long totfs, i(year) j(bin)
tempfile snap
	save `snap', replace

*TANF
import excel "./out/timeseries.xlsx", clear firstrow sheet("TANF") 
	keep year totpcttanfatti_spm*
	reshape long totpcttanfatti_spm, i(year) j(bin)
tempfile tanf
	save `tanf', replace

*Medicaid
import excel "./out/timeseries.xlsx", clear firstrow sheet("mcaidatti_spm") 
	keep year tot*
	reshape long totmcaid, i(year) j(bin)
tempfile mcaid
	save `mcaid', replace
	
*SSI
import excel "./out/timeseries.xlsx", clear firstrow sheet("ssiatti_spm") 
	keep year tot*
	reshape long totssi, i(year) j(bin)
tempfile ssi
	save `ssi', replace
	
*Public housing
import excel "./out/timeseries.xlsx", clear firstrow sheet("publicatti_spm") 
	keep year tot*
	reshape long totpublic, i(year) j(bin)
tempfile public
	save `public', replace

*EITC and ACTC/CTC
foreach cred in eic actcctc  {
	import excel "./out/timeseries.xlsx", clear firstrow sheet("`cred'atti_spmamt") 
	keep year `cred'*
	reshape long `cred'atti_spm, i(year) j(bin)
	tempfile `cred'
	save ``cred'', replace
}
use `snap', clear
	merge 1:1 year bin using `tanf', keep(match) nogen
	merge 1:1 year bin using `eic', keep(match) nogen
	merge 1:1 year bin using `ssi', keep(match) nogen
	merge 1:1 year bin using `public', keep(match) nogen
	merge 1:1 year bin using `actcctc', keep(master match) nogen //CTC started in 1998
	merge 1:1 year bin using `mcaid', keep(master match) nogen //just select years for Medicaid
	merge 1:1 year bin using "./dta/pctkidbin.dta", keep(master match) nogen //pct kids in each bin
	merge m:1 year using "./dta/pop_le18.dta", keep(master match) nogen //number kids total
	
	*Get # kids in each bin
	foreach p in spm {
		g numkid`p' = pct`p'*pop
			lab var numkid`p' "# kids in each bin"
			}

*Clean up variable names
rename *atti_spm *
rename tot* *
rename actcctc ctc
rename fs snap
rename eic eitc
rename pcttanf tanf

*Get total amount in each bin
*Fill in missings with 0s
foreach var in snap tanf eitc ctc {
	replace `var' = 0 if `var' == . //CTC started 1998
	}

*0 out CTC amounts going to > 200%
replace ctc = 0 if bin == 5
	
*Area graphs for each bin and then overall -- programs other than CTC: all expenditures
foreach prog in snap tanf eitc mcaid ctc ssi public {
		egen all`prog' = total(`prog'), by(year) //summing x-bins
			g `prog'pc = (`prog'*1000000000)/numkidspm
		g all`prog'pc = (all`prog'*1000000000)/pop
		}

*********************************************************************************
*FIGURE 6: Area graphs, Spending on Children, by Program, 1990-2015
*********************************************************************************
*Export numbers to excel
preserve
keep year allmcaid allssi allctc alleitc allsnap alltanf allpublic
	duplicates drop
	export excel using "./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure6_raw") sheetreplace
restore

*********************************************************************************
*Figure 7A: Spending on Children, by Family Income, 1990-2015
*Figure 7B: Share of Total Federal Spending, 1990-2015

*Figure 11: Per-child Federal Spending on Children, by Family Income, 1990-2015
*a. Per-child spending
*b. Growth in per-child spending (created from excel output)

*Appendix Figure 1: Spending on Children by Family Income to Poverty, including
*Medicaid, Public Housing, and SSI
*a. Per-child spending
*b. Growth in per-child spending (created from excel output)
*********************************************************************************
preserve //preserve to aggregate, then get back to programs

cap drop pcttotal* 
cap drop total*
foreach pc in "" pc  { //loop over per-capita
g total`pc' = snap`pc' + tanf`pc' + eitc`pc' + ctc`pc'
	tab year, su(total`pc')
g totalmcaid`pc' = snap`pc' + tanf`pc' + eitc`pc' + ctc`pc' + mcaid`pc' + ssi`pc' + public`pc'
}

foreach m in "" mcaid  { //loop over w/ and wo/Medicaid
*Get annual total (summing across all bins
egen yeartotal`m' = total(total`m'), by(year)
*Each bin's share of total
g pcttotal`m' = total`m'/yeartotal`m'
	}

drop totalmcaidpc
keep pcttotal* year total* bin
	reshape wide total pcttotal totalpc totalmcaid pcttotalmcaid, i(year) j(bin)
	
*Export to excel
export excel year total1 total2 total3 total4 total5 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure7a_raw") sheetreplace
		
export excel year pcttotal1 pcttotal2 pcttotal3 pcttotal4 pcttotal5 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure7b_raw") sheetreplace

export excel year totalpc1 totalpc2 totalpc3 totalpc4 totalpc5 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure11_raw") sheetreplace
		
export excel year totalmcaid1 totalmcaid2 totalmcaid3 totalmcaid4 totalmcaid5 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure_A1a_raw") sheetreplace
		
export excel year pcttotalmcaid1 pcttotalmcaid2 pcttotalmcaid3 pcttotalmcaid4 pcttotalmcaid5 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure_A1b_raw") sheetreplace
		
*Save as temp file to be used for Appendix Table 3 comparing admin totals w/ CPS
cap drop totalpc* totalmcaidpc*
	tempfile adminpov
		save `adminpov', replace

restore //bring back program totals

*********************************************************************************
*FIGURE 13: Spending on Children, by Poverty Status, 1990 and 2015
*********************************************************************************
*Just keep years we need
tab year, mi
keep if year == 1990 | year == 2015

*Agggregate bins into > or < 100% poverty
keep year bin eitc ctc tanf snap mcaid numkidspm
	reshape wide eitc ctc tanf snap mcaid numkidspm, i(year) j(bin)

foreach var in eitc ctc tanf snap mcaid {
	
*Aggregate amounts going to each bin	
*For CTC: Limit to expenditures below 200% poverty
if "`var'" == "ctc" {
	replace `var'5 = 0
	}
	g `var'lt100 = `var'1 + `var'2
	g `var'ge100 =  `var'3 + `var'4 + `var'5
	
*Per capita amounts by aggregated bin	
	g `var'lt100pc = (`var'lt100/(numkidspm1 + numkidspm2))*1000000000
	g `var'ge100pc = (`var'ge100/(numkidspm3 + numkidspm4 + numkidspm5))*1000000000
		
} //end credit loop

*Export all to excel
order year eitc* ctc* mcaid* tanf* snap*
	keep year *100
export excel year *100 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure13_raw") sheetreplace
		
		
********************************************************************************
*Appendix Figure 3: Spending on Children by Family Income, Comparison 
*of Administrative Data to CPS
*a. Aggregate spending
*b. Share of total spending
*********************************************************************************	

*First, bring in and create CPS version
	use "./dta/pctkidbin.dta", clear //pct kids in each bin
	merge m:1 year using "./dta/pop_le18.dta", keep(master match) nogen //number kids total
		
*Get # kids in each bin
	g numkidspm = pctspm*pop
		lab var numkidspm "# kids in each bin"
	rename bin spmbin
	keep numkid year spmbin
	tempfile pop
		save `pop', replace

use "./dta/cps_BPEA.dta", clear

*Get fraction going to each ATTI:SPM bin (using the Bitler-Hoynes ATTI measure)
*income:poverty ratios 
rename h_inc_alt1a_spm spm_inc_pov
summ spm_inc_pov, d
bys year: summ spm_inc_pov

	g spmbin = 1 if spm_inc_pov < 0.5
	replace spmbin = 2 if spm_inc_pov >= 0.5 & spm_inc_pov < 1
	replace spmbin = 3 if spm_inc_pov >= 1 & spm_inc_pov < 1.5
	replace spmbin = 4 if spm_inc_pov >= 1.5 & spm_inc_pov < 2
	replace spmbin = 5 if spm_inc_pov >= 2 & spm_inc_pov !=.

tab year spmbin, row mi


*Cut off CTC at 200% poverty
	replace ctc = 0 if spmbin == 5

*Weigh by households
g one = 1 //counter var
	bys hhid year: g n = _n //just keep 1st observation in each hh
	tab n //make sure hh size looks ok
	
	keep if n == 1
*Collapse to get aggregate benefit amounts going to each poverty bin, one obs per hh
collapse (sum) h_fsval h_inc_pa ctc heitc h_numkidcaid [pw = h_weight], ///
	by(spmbin calyear)
	*rename calyear to match other data
	rename calyear year
	*Couple of variable labels to keep track of everything
	lab var h_fsval "SNAP amt"
	lab var h_inc_pa "cash welf amt"
	lab var ctc "CTC amt"
	lab var heitc "EITC amt"
	lab var h_numkidcaid "num kids Mcaid"
	
	drop if spmbin == .
	*Get total for each benefit across all bins
	foreach var in h_fsval h_inc_pa ctc heitc h_numkidcaid  {
		local lab: variable label `var'
		egen `var'total = total(`var'), by(year)
		g pct`var' = `var'/`var'total
			*Fill in 0 values before CTC started
			replace pct`var' = 0 if `var'total == 0
			lab var pct`var' "Pct `lab'"
		}
		keep h* ctc pct* year spmbin
		*Merge to population data
		merge 1:1 year spmbin using `pop', keep(master match) 
			assert _merge >= 3 //make sure everything matched
			drop _merge
	
		reshape wide h_fsval h_inc_pa ctc heitc h_numkidcaid ///
			pcth_fsval pcth_inc_pa pctctc pctheitc pcth_numkidcaid ///
			numkidspm, i(year) j(spmbin)
			
*Make sure this looks reasonable
foreach var in pcth_fsval pcth_inc_pa pctctc pctheitc pcth_numkidcaid {
forvalues n = 1/5 {
di "This is `var' in poverty bucket `n' 1990-2015"
	tab year, su(`var'`n')
	}
	}
	
*Rename a couple of benefits
rename *h_numkidcaid* *mcaid*
rename *heitc* *eitc*
rename *h_inc_pa* *tanf*
rename *h_fsval* *snap*

*Merge to CPI and put in 2016 dollars
merge 1:1 year using "./dta/cpiu.dta", keep(match master) 
	assert _merge >= 3 
	drop _merge //make sure everything ok
*Put everything in 2015 dollars
summ cpiu if year == 2015, d
	local cpi2015 = r(mean)
	foreach prog in eitc snap tanf ctc { 
		forvalues n = 1/5{	
			replace `prog'`n' = `prog'`n'*(`cpi2015'/cpiu)
			}
			}
		
foreach prog in eitc snap tanf ctc { 
forvalues n = 1/5{		
g `prog'`n'_pc = (`prog'`n')/numkidspm`n' 
		lab var `prog'`n'_pc "Per child expend `lab' Q`n', 2015 $"
		}
		}
		
*Now total amounts
forvalues n = 1/5 {
	g total`n' = eitc`n' + snap`n' + tanf`n' + ctc`n'
		//leaving out medicaid for now
		lab var total`n' "Total amt on Q`n'"
	}

forvalues n = 1/5 { //create pct of total
	g pcttotal`n' = total`n'/(total1 + total2 + total3 + total4 + total5)
	}
	
*Put dollar amounts in billions
forvalues n = 1/5 { //create pct of total
	replace total`n' = total`n'/1000000000
	}
	
rename pcttotal* pcttotalcps*
rename total* totalcps*

*Merge to admin data
merge 1:1 year using `adminpov'

cap drop totalpc*
*Binned PC amounts
foreach cps in cps "" {
g total`cps'_u100 = (total`cps'1 + total`cps'2)
g total`cps'_o100 = (total`cps'3 + total`cps'4 + total`cps'5)
}

	
foreach dta in cps "" {
*Bin up < and > 100% poverty
	g pct`dta'_u100 = pcttotal`dta'1 + pcttotal`dta'2
	g pct`dta'_o100 = pcttotal`dta'3 + pcttotal`dta'4 +pcttotal`dta'5
	}

*Export all variables to excel (can always aggregate up)
export excel year *100 using ///
	"./out/BPEAtablenumbers.xlsx", firstrow(variables) ///
		sheet("Figure_A3_raw") sheetreplace

cap log close
