clear
set more off
cd 

************************************************************************
* How many lives are at stake? Assessing 2030 Sustainable Development Goal
*    trajectories for maternal and child health

* John W McArthur * Krista Rasmussen * Gavin Yamey
* February 2018 BMJ, 360, k373 doi: https://doi.org/10.1136/bmj.k373
************************************************************************

************************************
* ANALYSIS AND FIGURES
* 1. Maternal Mortality (MMR)
* 2. Under-5 Child Mortality (CMR)
* 4. Figures and Tables
* 5. Appendix Files
************************************


**********************************************************************
**********************************************************************
* 1. Maternal Mortality (MMR)
**********************************************************************
**********************************************************************

use "SDG_mmr2017.dta", clear

*******************************************************************************
**A. Calculate average annual rate of decline from 2005-2015
*******************************************************************************

 * Flipping direction of slope so higher rate of decline = improvement
gen rp0515 = 1 - (n2015/n2005)^(1/10)

*******************************************************************************
**B. Calculate scenarios: 
   * 1. Business as usual
   * 2. Global MMR = 70
   * 3. Each country reaches MMR = 70
*******************************************************************************

*** 1. Scenario 1: Business as usual (BAU) 
 
 * Extrapolate BAU trajectory out to 2030
	for num 2015/2030: gen ex0515_X = n2015*((1-rp0515)^(X-2015))
	for num 2015/2030: label variable ex0515_X "MMR X trajectory from 2005-15 rates"

 * Check if country is on track for meeting MMR <=70 by 2030 on 2005-2015 trajectory
	gen sdgyes2030_0515 = (ex0515_2030<=70) if ex0515_2030!=.
	label variable sdgyes2030_0515 "Meets MMR <=70 by 2030 on rp05-15"

 
*** 2. Scenario 2: World reaches overall MMR of 70 by 2030: each country accelerates by 5.3725 pct. points above recent rate of decline
 
 * Calculate implied rate of decline under Scenario 2
	gen mmr_rp0515_4 = rp0515 + .053725
	
 * Extrapolate Scenario 2 MMR trajectory out to 2030
	for num 2015/2030: gen SDG_x5_4mmrX = n2015*((1-mmr_rp0515_4)^(X-2015))
	for num 2015/2030: label variable SDG_x5_4mmrX "MMR in X if world meets 70 MMR in 2030 (all countries accel 5.3725 pct. points, Scenario 2)"

	
*** 3. Scenario 3: Each country reaches MMR <=70: off-track countries accelerate to meet 70 by 2030 
 
 * Calculate rate of decline needed for each country to reach MMR = 70 by 2030 under Scenario 3
	gen mmr_rpSDGneed = 1 - (70/n2015)^(1/15)

	* Because some countries have already achieved goal, above calculation 
	* would have them go in the opposite direction
	replace mmr_rpSDGneed = 0 if n2015 <=70

 * Calculate each country's percentage point acceleration required to achieve scenario
	gen mmr_pctpSDGneed = mmr_rpSDGneed - rp0515
	 replace mmr_pctpSDGneed = 0 if mmr_pctpSDGneed < 0
	 replace mmr_pctpSDGneed = 0 if ex0515_2030 <=70
	 label variable mmr_pctpSDGneed "Pct. point increase required to reach MMR=70 by 2030"
 
 * Extrapolate Scenario 3 MMR trajectory out to 2030
	for num 2015/2030: gen SDG_x70mmrX = n2015*((1-mmr_rpSDGneed)^(X-2015))
	for num 2015/2030: label variable SDG_x70mmrX "MMR in X if all countries meet <= 70 MMR (Scenario 3)"
	* For on-track countries, instead use current rate to extrapolate out to 2030 (2005-2015)
		for num 2015/2030: replace SDG_x70mmrX = ex0515_X if ex0515_2030<=70 
 
 
* Rename to ease merge
foreach var of varlist sdgyes2030_0515 n* rp0515 ex0515_20* {
rename `var' mmr_`var'
}

*******************************************************************************
**C. Calculate country-level accelerations during MDG period
*******************************************************************************

 * Previous rates of decline
gen rp9000 = 1- (mmr_n2000/mmr_n1990)^(1/10)
gen rp0015 = 1- (mmr_n2015/mmr_n2000)^(1/15)

 * Percentage point increase in rate of decline during MDG period
gen dif9000_0015 = rp0015 - rp9000

save "SDG_mmr_graphdata2017.dta", replace

*******************************************************************************
**D. Find number of additional lives saved under SDG scenarios 
*******************************************************************************

use "SDG_mmr_graphdata2017.dta", clear

*** 1. Scenario 1: Business as usual (BAU)
 
* 1.a. Extrapolate BAU deaths to 2030
for num 2017/2030: gen bau0515_xmdX = mmr_ex0515_X * (birthX / 100000)
 for num 2017/2030: label variable bau0515_xmdX "Maternal deaths in X under BAU"
	 
* 1.b. Cumulative BAU deaths from 2017 to 2030 for each country
egen cul1730_mmrdeathsBAU = rowtotal(bau0515_xmd2017 - bau0515_xmd2030), missing
 label variable cul1730_mmrdeathsBAU "Sum of maternal deaths 2017-2030, BAU trajectory"

	 
*** 2. Scenario 2: World reaches overall MMR of 70 by 2030

* 2.a. Extrapolate Scenario 2 deaths to 2030 
for num 2017/2030: gen SDG_x5_4mdX = SDG_x5_4mmrX * (birthX / 100000) 
 for num 2017/2030: label variable SDG_x5_4mdX "Maternal deaths in X if world meets 70 MMR (Scenario 2)"

* 2.b. Cumulative Scenario 2 deaths from 2017 to 2030 for each country
egen cul1730_mmrdeathsSDGx5_4 = rowtotal(SDG_x5_4md2017 - SDG_x5_4md2030), missing
 label variable cul1730_mmrdeathsSDGx5_4 "Sum of maternal deaths 2017-2030 if world meets 70 MMR (Scenario 2)"	
	 
* 2.c. Lives at stake (difference between BAU and Scenario 2) for each country
for num 2017/2030: gen dif_bau0515_sdgyes5_4_X = bau0515_xmdX - SDG_x5_4mdX
 for num 2017/2030: label variable dif_bau0515_sdgyes5_4_X "Lives at stake in X, dif. between BAU and if world meets 70 MMR (Scenario 2)"
	 
* 2.d. Cumulative lives at stake from 2017 to 2030 for Scenario 2 for each country
egen cul1730_mmrstake5_4 = rowtotal(dif_bau0515_sdgyes5_4_2017 - dif_bau0515_sdgyes5_4_2030), missing
 label variable cul1730_mmrstake5_4 "Lives at stake total 2017-2030, dif. between BAU and if world meets 70 MMR (Scenario 2)"

	 
*** 3. Scenario 3: Each country reaches <= 70 MMR by 2030
	
* 3.a. Extrapolate Scenario 3 deaths to 2030
for num 2017/2030: gen SDG_x70mdX = SDG_x70mmrX * (birthX / 100000)
 for num 2017/2030: label variable SDG_x70mdX "Maternal deaths in X if all countries meet <= 70 MMR (Scenario 3)"
	 
* 3.b. Cumulative Scenario 3 deaths from 2017 to 2030 for each country
egen cul1730_mmrdeathsSDG70 = rowtotal(SDG_x70md2017 - SDG_x70md2030), missing
 label variable cul1730_mmrdeathsSDG70 "Sum of maternal deaths 2017-2030 if all countries meet <=70 MMR (Scenario 3)"
	 
* 3.d. Lives at stake (difference between BAU and Scenario 3) for each country
for num 2017/2030: gen dif_bau0515_sdgyes70_X = bau0515_xmdX - SDG_x70mdX
 for num 2017/2030: label variable dif_bau0515_sdgyes70_X "Lives at stake in X, dif. between BAU and if all countries meet <=70 MMR (Scenario 3)"

* 3.d. Cumulative lives at stake from 2017 to 2030 for Scenario 3 for each country
egen cul1730_mmrstake = rowtotal(dif_bau0515_sdgyes70_2017 - dif_bau0515_sdgyes70_2030), missing
 label variable cul1730_mmrstake "Lives at stake total 2017-2030, dif. between BAU and if all countries meet <=70 MMR (Scenario 3)"

	 
save "SDG_mmr_livesstake2017.dta", replace

**********************************************************************
**********************************************************************


**********************************************************************
**********************************************************************
* 2. Under-5 Child Mortality Rate (CMR)
**********************************************************************
**********************************************************************

use "SDG_u5mr2017.dta", clear

*******************************************************************************
**A. Calculate average annual rate of decline from 2006-2016
*******************************************************************************

 * Flipping direction of slope so higher rate of decline = improvement
gen rp0616 = 1- (n2016/n2006)^(1/10)

*******************************************************************************
**B. Calculate scenarios: 
   * 1. Business as usual
   * 2. Each country reaches CMR <25.5
*******************************************************************************

*** 1. Scenario 1: Business as usual (BAU)

 * Extrapolate BAU trajectory out to 2030
	for num 2017/2030: gen ex0616_X = n2016*((1-rp0616)^(X-2016))
	for num 2017/2030: label variable ex0616_X "CMR X trajectory from 2006-16 rates"

 * Check if country is on track for meeting CMR <25.5 by 2030 on 2006-2016 trajectory
	gen sdgyes2030_0616 = (ex0616_2030<25.5) if ex0616_2030!=.
	label variable sdgyes2030_0616 "Meets CMR target by 2030 on rp06-16"

	
*** 2. Scenario 2: Each country reaches CMR <25.5: off-track countries accelerate to meet <25.5 by 2030 

 * Calculate rate of decline needed for each country to reach CMR <25.5 by 2030
	gen u5mr_rpSDGneed = 1 - (25.49/n2016)^(1/14)

	* Because some countries have already achieved goal, above calculation 
	* would have them go in the opposite direction
	replace u5mr_rpSDGneed = 0 if n2016 < 25.5 & n2016!=.

 * Calculate each country's percentage point acceleration required to achieve scenario
	gen u5mr_pctpSDGneed = u5mr_rpSDGneed - rp0616
	 replace u5mr_pctpSDGneed = 0 if u5mr_pctpSDGneed < 0
	 replace u5mr_pctpSDGneed = 0 if ex0616_2030 <25.5
	 label variable u5mr_pctpSDGneed "Pct. point increase required to reach CMR <25.5 by 2030"

 * Extrapolate Scenario 2 CMR trajectory out to 2030
	for num 2017/2030: gen SDG_xcmrX = n2016*((1-u5mr_rpSDGneed)^(X-2016))
	for num 2017/2030: label variable SDG_xcmrX "CMR in X with CMR <25.5 in 2030 trajectory"
	* For on-track countries, instead use current rate to extrapolate out to 2030 (2006-2016)
	 for num 2017/2030: replace SDG_xcmrX = ex0616_X if ex0616_2030<25.5 
	 

* Rename to ease merge
foreach v of varlist sdgyes2030_0616 n2000 n2016 rp0616 ex0616_20* {
rename `v' u5mr_`v'
}

save "SDG_u5mr_graphdata2017.dta", replace

*******************************************************************************
**C. Find number of additional lives saved under SDG scenarios
*******************************************************************************

use "SDG_u5mr_graphdata2017.dta", clear

*** 1. Scenario 1: Business as usual (BAU)

* 1.a. Extrapolate BAU deaths to 2030
for num 2017/2030: gen bau0616_xcdX = u5mr_ex0616_X * (birthX / 1000)
 for num 2017/2030: label variable bau0616_xcdX "Child deaths in X under BAU"
 
* 1.b. Cumulative BAU deaths from 2017 to 2030 for each country
egen cul1730_u5mrdeathsBAU = rowtotal(bau0616_xcd2017 - bau0616_xcd2030), missing
 label variable cul1730_u5mrdeathsBAU "Sum of child deaths 2017-2030, BAU trajectory"

	 
*** 2. Scenario 2: Each country reaches <25.5 CMR by 2030

* 2.a. Extrapolate Scenario 2 deaths to 2030
for num 2017/2030: gen SDG_xcdX = SDG_xcmrX * (birthX / 1000)
 for num 2017/2030: label variable SDG_xcdX "Child deaths in X if all countries meet <25.5 CMR (Scenario 2)"
	 
* 2.b. Cumulative Scenario 2 deaths from 2017 to 2030 for each country
egen cul1730_u5mrdeathsSDG = rowtotal(SDG_xcd2017 - SDG_xcd2030), missing
 label variable cul1730_u5mrdeathsSDG "Sum of child deaths 2017-2030 if all countries meet <25.5 CMR (Scenario 2)"
	 
* 2.c. Lives at stake (difference between BAU and Scenario 2) for each country
for num 2017/2030: gen dif_bau0616_sdgyes_X = bau0616_xcdX - SDG_xcdX
 for num 2017/2030: label variable dif_bau0616_sdgyes_X "Lives at stake in X, dif. between BAU and if all countries meet <25.5 CMR (Scenario 2)"
	 
* 2.d. Cumulative lives at stake from 2017 to 2030 for Scenario 2 for each country
egen cul1730_u5mrlives_stake = rowtotal(dif_bau0616_sdgyes_2017 - dif_bau0616_sdgyes_2030), missing
 label variable cul1730_u5mrlives_stake "Lives at stake total 2017-2030, dif. between BAU and if all countries meet <25.5 CMR (Scenario 2)"

	 
save "SDG_u5mr_livesstake2017.dta", replace

**********************************************************************
**********************************************************************


**********************************************************************
**********************************************************************
* 3. Figures and Tables
**********************************************************************
**********************************************************************

**********************************************************************
* Merge data for graph 
**********************************************************************

 * Merge MMR and CMR datasets
use "SDG_mmr_livesstake2017.dta", clear 
merge 1:1 CountryCode using "SDG_u5mr_livesstake2017.dta"
drop _merge
save "SDG_merged2017.dta", replace

 * By country, count number of targets for which each country is on track
gen SDGyes_two = mmr_sdgyes2030_0515 + u5mr_sdgyes2030_0616
 label variable SDGyes_two "Num. targets on track for 2030"

save "SDGcount2017.dta", replace


*******************************************************************************
**Figure 1 Map of countries that are off-track for 2030 targets for maternal or child mortality
*******************************************************************************

use "SDGcount2017.dta", clear

 * For GIS, isolate each category
* a. off-track only for MMR
gen offmmr_only = 1 if mmr_sdgyes2030_0515==0 & SDGyes_two!=0 & SDGyes_two!=.
* b. off-track only for CMR
gen offu5mr_only = 1 if u5mr_sdgyes2030_0616==0 & SDGyes_two!=0 & SDGyes_two!=.
* c. off-track for both
gen offboth = 1 if mmr_sdgyes2030_0515==0 & u5mr_sdgyes2030_0616==0 & SDGyes_two!=.
* d. missing data for at least one indicator
gen missing = 1 if mmr_sdgyes2030_0515==. | u5mr_sdgyes2030_0616==.
* e. on-track for both
gen ontrackboth = 1 if mmr_sdgyes2030_0515==1 & u5mr_sdgyes2030_0616==1 & SDGyes_two!=.

keep CountryName CountryCode off* missing on*


*******************************************************************************
**Figure 2 Scenarios for global MMR trajectory to 2030
*******************************************************************************

use "SDG_mmr_livesstake2017.dta", clear

 ** Reshape long
reshape long SDG_x70mmr mmr_ex0515_  SDG_x5_4mmr mmr_n birth, i(CountryName CountryCode Region) j(year)
drop SDG_x70md*

 * Collapse all countries - extrapolated to 2030 data with added RP and BAU, birth-weighted 
collapse (mean) mmr_n SDG_x70 mmr_ex0515_ SDG_x5_4mmr [weight=birth], by(year)
label variable mmr_n "Actual" 
label variable mmr_ex0515_ "EX 05-15"
drop if year<2000

twoway (line mmr_n year, lcolor(navy) lwidth(thick)) (line mmr_ex0515_ year, ///
lcolor(navy) lwidth(medthick) lpattern(dash)) (line SDG_x5_4mmr year, lpattern(dash)) (line SDG_x70 year, lpattern(dash)), ylabel(0(50 ///
)400, angle(horizontal) valuelabel) legend(order(1 "Actual" 2 "Business as Usual (BAU)" 3 "BAU + 5.4 pct. points" 4 "All countries meet 70 MMR")) ///
scheme(s1mono)

save "SDG_mmr_graphcollapse2017.dta", replace


*******************************************************************************
**Figure 3 Distribution of average annual rates of decline required to meet 2030 SDG for CMR among countries that have not yet met target
*******************************************************************************

use "SDG_u5mr_livesstake2017.dta", clear

* Keep countries that have not yet met CMR <25.5 as of 2016
keep if u5mr_n2016>=25.5 & u5mr_n2016!=. & u5mr_sdgyes2030!=.

* Create categories by rate required to meet CMR <25.5 (0-3%; 3-6%; 6-9%; 9-12%)
gen rpneed_cat = .
foreach num of numlist 0(.03).09{
local lista = `num' + .03
replace rpneed_cat = `num' if u5mr_rpSDGneed >= `num' & u5mr_rpSDGneed <`lista'
}
* Count the number of countries and 2030 births in each rate required category
gen count = 1
collapse (sum) count birth2030, by(rpneed_cat)

save "SDG_u5mr_bargraph2017.dta", replace


****************************************************************************************
**Table 1 Estimated maternal lives at stake, cumulative 2017-2030
****************************************************************************************

use "SDG_mmr_livesstake2017.dta", clear

br CountryName cul*
* Identify top 10 countries with most maternal lives at stake
gsort -cul1730_mmrstake
gen top10 = _n
* Identify remaining off-track countries
gen offtrack_nottop10 = 1 if cul1730_mmrstake!=0 & cul1730_mmrstake!=. & top10>10

* Sum maternal deaths under SDG scenarios in countries that are: 1) not in top 10 countries with highest no. of lives at stake, & 2) not on track for MMR <=70 
foreach var of varlist cul1730_mmrdeathsBAU cul1730_mmrdeathsSDG70 cul1730_mmrstake{
egen sum_not10`var' = sum(`var') if offtrack_nottop10==1
}

* Sum maternal deaths under SDG scenarios in all off-track countries
foreach var of varlist cul1730_mmrdeathsBAU cul1730_mmrdeathsSDG70 cul1730_mmrstake{
egen tol`var' = sum(`var') if mmr_sdgyes2030_0515==0
}
save "SDG_mmr_livesstaketop10.dta", replace



*******************************************************************************
**Table 2 Estimated under-5 children's lives at stake, cumulative 2017-2030
*******************************************************************************

use "SDG_u5mr_livesstake2017.dta", clear

br CountryName cul* u5mr_sdgyes2030_0616
* Identify top 10 countries with most child lives at stake
gsort -cul1730_u5mrlives_stake
gen top10 = _n
* Identify remaining off-track countries
gen offtrack_nottop10 = 1 if cul1730_u5mrlives_stake!=0 & cul1730_u5mrlives_stake!=. & top10>10

* Sum child deaths under SDG scenarios in countries that are: 1) not in top 10 countries with highest no. of lives at stake, & 2) not on track for CMR <25.5
foreach var of varlist cul1730_u5mrdeathsBAU cul1730_u5mrdeathsSDG cul1730_u5mrlives_stake{
egen sum_not10`var' = sum(`var') if offtrack_nottop10==1
}

* Sum child deaths under SDG scenarios in all off-track countries
foreach var of varlist cul1730_u5mrdeathsBAU cul1730_u5mrdeathsSDG cul1730_u5mrlives_stake{
egen tol`var' = sum(`var') if u5mr_sdgyes2030_0616==0
}
save "SDG_u5mr_livesstaketop10.dta", replace

**********************************************************************
**********************************************************************
* 4. Appendix Files
**********************************************************************
**********************************************************************

*******************************************************************************
**Appendix 1: MMR and CMR trends - average annual rates of decline over most recent available decade and rates of decline required to meet SDG 2030 target thresholds
*******************************************************************************

use "SDG_u5mr_livesstake2017.dta", clear
keep CountryName CountryCode u5mr_n2016 u5mr_rp0616 u5mr_ex0616_2030 u5mr_rpSDGneed
tostring u5mr_rpSDGneed, force replace 
replace u5mr_rpSDGneed =" " if u5mr_ex0616_2030<25.5 & u5mr_ex0616_2030!=.
save "SDG_u5mr_supplementalTable2017.dta", replace

use "SDG_mmr_livesstake2017.dta", clear
keep CountryName CountryCode Region mmr_n2015 mmr_rp0515 mmr_ex0515_2030 mmr_rpSDGneed
tostring mmr_rpSDGneed, force replace
replace mmr_rpSDGneed =" " if mmr_ex0515_2030<=70 & mmr_ex0515_2030!=.
merge 1:1 CountryCode using "SDG_u5mr_supplementalTable2017.dta"

sort CountryName
drop CountryCode _merge Region
order CountryName mmr_n2015 mmr_rp0515 mmr_ex0515_2030 mmr_rpSDGneed u5mr_n2016 u5mr_rp0616 u5mr_ex0616_2030 u5mr_rpSDGneed


*******************************************************************************
**Appendix 2: Estimated maternal and child deaths and lives at stake for 2017-2030
*******************************************************************************

use "SDG_u5mr_livesstake2017.dta", clear
keep CountryName CountryCode cu*
save "SDG_u5mr_supplementalTableLIVES2017.dta", replace

use "SDG_mmr_livesstake2017.dta", clear
keep CountryName CountryCode cu*
drop cul1730_mmrdeathsSDGx5_4 cul1730_mmrstake5_4
merge 1:1 CountryCode using "SDG_u5mr_supplementalTableLIVES2017.dta"

sort CountryName
drop CountryCode _merge
order CountryName cul1730_mmrdeathsBAU cul1730_mmrdeathsSDG cul1730_mmrstake cul1730_u5mrdeathsBAU cul1730_u5mrdeathsSDG cul1730_u5mrlives_stake
 

*******************************************************************************
**Appendix 3: Average annual declines in neonatal mortality rate for 2006-2016, trajectories for 2030, and estimated deaths and lives at stake for 2017-2030
*******************************************************************************

use "SDG_nmr2017.dta", clear

*******************************************************************************
**A. Calculate average annual rate of decline from 2006-2016
*******************************************************************************

 * Flipping direction of slope so higher rate of decline = improvement
gen rp0616 = 1- (n2016/n2006)^(1/10)

*******************************************************************************
**B. Calculate scenarios: 
   * 1. Business as usual
   * 2. Each country reaches NMR <12.5
*******************************************************************************

*** 1. Scenario 1: Business as usual (BAU)

 * Extrapolate out to 2030
	for num 2017/2030: gen ex0616_X = n2016*((1-rp0616)^(X-2016))
	for num 2017/2030: label variable ex0616_X "NMR X trajectory from 2006-16 rates"

 * Check if country is on track for meeting NMR <12.5 by 2030 on 2006-2016 trajectory
	gen sdgyes2030_0616 = (ex0616_2030<12.5) if ex0616_2030!=.
	label variable sdgyes2030_0616 "Meets NMR target <12.5 by 2030 on rp06-16"


*** 2. Scenario 2: Each country reaches NMR <12.5: off-track countries accelerate to meet <12.5 by 2030

 * Calculate rate of decline needed for each country to reach NMR <12.5 by 2030 under Scenario 2
	gen nmr_rpSDGneed = 1 - (12.49/n2016)^(1/14)

	* Because some countries have already achieved goal, above calculation would have them go in the opposite direction
	replace nmr_rpSDGneed = 0 if n2016 < 12.5

 * Calculate each country's percentage point acceleration required to achieve scenario
	gen nmr_pctpSDGneed = nmr_rpSDGneed - rp0616
 	 replace nmr_pctpSDGneed = 0 if nmr_pctpSDGneed < 0
	 replace nmr_pctpSDGneed = 0 if ex0616_2030 < 12.5 & n2016!=.
	 label variable nmr_pctpSDGneed "pct. point increase required to reach NMR <12.5 by 2030"

 * Extrapolate Scenario 2 NMR trajectory out to 2030
	for num 2017/2030: gen SDG_xnmrX = n2016*((1-nmr_rpSDGneed)^(X-2016))
	for num 2017/2030: label variable SDG_xnmrX "NMR in X with NMR <12.5 in 2030 trajectory"
 	* For on-track countries, instead use current rate to extrapolate out to 2030 (2006-2016)
	for num 2017/2030: replace SDG_xnmrX = ex0616_X if ex0616_2030<12.5 

	
 * Rename to ease merge
foreach v of varlist n2000 n2016 rp0616 ex0616_20* sdgyes2030_0616 {
rename `v' nmr_`v'
}

save "SDG_nmr_graphdata2017.dta", replace


*******************************************************************************
**C. Find number of additional lives saved under SDG scenarios
*******************************************************************************

use "SDG_nmr_graphdata2017.dta", clear

*** 1. Scenario 1: Business as usual (BAU)
 
* 1.a. Extrapolated BAU deaths to 2030
for num 2017/2030: gen bau0616_xneodX = nmr_ex0616_X * (birthX / 1000)
 for num 2017/2030: label variable bau0616_xneodX "Neonatal deaths in X under BAU"
	 
* 1.b. Cumulative BAU deaths from 2017 to 2030 for each country
egen cul1730_nmrdeathsBAU = rowtotal(bau0616_xneod2017 - bau0616_xneod2030), missing
 label variable cul1730_nmrdeathsBAU "Sum of neonatal deaths 2017-2030, BAU trajectory"

	 
*** 2. Scenario 2: All countries reach <=12.5 NMR by 2030

* 2.a. Extrapolate Scenario 2 deaths to 2030
for num 2017/2030: gen SDG_xneodX = SDG_xnmrX * (birthX / 1000)
 for num 2017/2030: label variable SDG_xneodX "Neonatal deaths in X if all countries meet <12.5 NMR (Scenario 2)"

* 2.b. Cumulative Scenario 2 deaths from 2017 to 2030 for each country
egen cul1730_nmrdeathsSDG = rowtotal(SDG_xneod2017 - SDG_xneod2030), missing
 label variable cul1730_nmrdeathsSDG "Sum of neonatal deaths 2017-2030 if all countries meet <12.5 NMR (Scenario 2)" 

* 2.c. Lives at stake (difference between BAU and Scenario 2) for each country
for num 2017/2030: gen dif_bau0616_sdgyes_X = bau0616_xneodX - SDG_xneodX
 for num 2017/2030: label variable dif_bau0616_sdgyes_X "Lives at stake in X dif. between BAU and if all countries meet <12.5 NMR (Scenario 2)"

* 2.d. Cumulative lives at stake from 2017 to 2030 for Scenario 2 for each country
egen cul1730_nmrlives_stake = rowtotal(dif_bau0616_sdgyes_2017 - dif_bau0616_sdgyes_2030), missing
 label variable cul1730_nmrlives_stake "Lives at stake total 2017-2030, dif. between BAU and if all countries meet <12.5 NMR (Scenario 2)"

 
save "SDG_nmr_livesstake2017.dta", replace


*******************************************************************************
**D. Clean for Appendix 3
*******************************************************************************

use "SDG_nmr_livesstake2017.dta", clear

keep CountryName nmr_n2016 nmr_rp0616 nmr_ex0616_2030 nmr_rpSDGneed cu*
tostring nmr_rpSDGneed, force replace 
replace nmr_rpSDGneed =" " if nmr_ex0616_2030<12.5 & nmr_ex0616_2030!=.
save "SDG_nmr_livesstake_clean2017.dta", replace

sort CountryName
order CountryName nmr_n2016 nmr_rp0616 nmr_ex0616_2030 nmr_rpSDGneed cul1730_nmrdeathsBAU cul1730_nmrdeathsSDG cul1730_nmrlives_stake

