clear
set more off
cd 

****************************************************************************
****************************************************************************
**  2. INFECTIOUS DISEASE INDICATORS
** tuberculosis deaths * malaria deaths * HIV deaths * antiretoviral therapy coverage

** John W. McArthur & Krista Rasmussen
** January 2018
****************************************************************************
****************************************************************************

****************************************************************************
***TUBERCULOSIS DEATHS
****************************************************************************
*I.  Counterfactual calculations - if 2001 deaths remained constant
*II. Incremental lives saved compared to 2001 level
****************************************************************************

***************************************************
**I. COUNTERFACTUAL CALCULATION
***************************************************
use "TB_deaths.dta", clear

gen IndicatorName = "Tuberculosis deaths, estimated total (excludes death due to HIV)"

 ** Counterfactual - compare actual to constant 2001 value
**************************************
 * CFD = if number of deaths in 2001 continued out to 2015
for num 2002/2015: gen CFD_2001_deathX = n2001
for num 2002/2015: label variable CFD_2001_deathX "TB deaths X extrapolated as 2001 value"
save "TB_deaths_rpfull", replace


***************************************************
**II. LIVES SAVED COMPARED TO 2001 LEVEL
***************************************************
*No acceleration tests due to pre-2000 missing data and widespread increasing number of deaths.
use "TB_deaths_rpfull.dta", clear

 ** Assume constant 2001 value - estimated difference between 2001 value and actual
*************************
for num 2002/2015: gen dif_mdg_cfd_X = CFD_2001_deathX - nX
for num 2002/2015: label variable dif_mdg_cfd_X "Counterfactual: est. lives affected in X, dif. 2001 and actual"

 ** Calculate total lives affected in different groups
 *1. sum difference between CF and actual for each group, by year
 *2. sum across all years to find total lives affected from 2002 - 2015
 
**All developing countries
for num 2002/2015: egen dev_cfd_lives_X = sum(dif_mdg_cfd_X) 
for num 2002/2015: label variable dev_cfd_lives_X "Counterfactual: developing world total lives affected in X"
egen dev_cfd_lives_all = rowtotal(dev_cfd_lives_2002-dev_cfd_lives_2015), missing
label variable dev_cfd_lives_all "Counterfactual: developing, total 2002-2015"

**By Region
for num 2002/2015: egen region_cfd_lives_X = sum(dif_mdg_cfd_X), by(Region)
for num 2002/2015: label variable region_cfd_lives_X "Counterfactual: by region total lives affected in X"
egen region_cfd_lives_all = rowtotal(region_cfd_lives_2002-region_cfd_lives_2015), missing
label variable region_cfd_lives_all "Counterfactual: by region, total 2002-2015"

**LIC
for num 2002/2015: egen LIC_cfd_lives_X = sum(dif_mdg_cfd_X) if IncomeGroup=="L"
for num 2002/2015: label variable LIC_cfd_lives_X "Counterfactual: LIC total lives affected in X"
egen LIC_cfd_lives_all = rowtotal(LIC_cfd_lives_2002-LIC_cfd_lives_2015), missing
label variable LIC_cfd_lives_all "Counterfactual: LIC, total 2002-2015"

**LIC ex. India
for num 2002/2015: egen LICxind_cfd_lives_X = sum(dif_mdg_cfd_X) if IncomeGroup=="L" & CountryCode!="IND"
for num 2002/2015: label variable LICxind_cfd_lives_X "Counterfactual: LIC ex-India total lives affected in X"
egen LICxind_cfd_lives_all = rowtotal(LICxind_cfd_lives_2002-LICxind_cfd_lives_2015), missing
label variable LICxind_cfd_lives_all "Counterfactual: LIC ex-India total 2002-2015"

**MIC
for num 2002/2015: egen MIC_cfd_lives_X = sum(dif_mdg_cfd_X) if IncomeGroup!="L"
for num 2002/2015: label variable MIC_cfd_lives_X "Counterfactual: MIC total lives affected in X"
egen MIC_cfd_lives_all = rowtotal(MIC_cfd_lives_2002-MIC_cfd_lives_2015), missing
label variable MIC_cfd_lives_all "Counterfactual: MIC, total 2002-2015"

**MIC ex. China
for num 2002/2015: egen MICxchn_cfd_lives_X = sum(dif_mdg_cfd_X) if IncomeGroup!="L" & CountryCode!="CHN"
for num 2002/2015: label variable MICxchn_cfd_lives_X "Counterfactual: MIC ex-China total lives affected in X"
egen MICxchn_cfd_lives_all = rowtotal(MICxchn_cfd_lives_2002-MICxchn_cfd_lives_2015), missing
label variable MICxchn_cfd_lives_all "Counterfactual: MIC ex-China, total 2002-2015"

**Ex. China & India
for num 2002/2015: egen xChiInd_cfd_lives_X = sum(dif_mdg_cfd_X) if CountryCode!="CHN" & CountryCode!="IND"
for num 2002/2015: label variable xChiInd_cfd_lives_X "Counterfactual: ex-China & India total lives affected in X"
egen xChiInd_cfd_lives_all = rowtotal(xChiInd_cfd_lives_2002-xChiInd_cfd_lives_2015), missing
label variable xChiInd_cfd_lives_all "Counterfactual: ex-China & India, total 2002-2015"

**China
for num 2002/2015: gen China_cfd_lives_X = dif_mdg_cfd_X if CountryCode=="CHN"
for num 2002/2015: label variable China_cfd_lives_X "Counterfactual: China total lives affected in X"
egen China_cfd_lives_all = rowtotal(China_cfd_lives_2002-China_cfd_lives_2015), missing
label variable China_cfd_lives_all "Counterfactual: China, total 2002-2015"

**India
for num 2002/2015: gen India_cfd_lives_X = dif_mdg_cfd_X if CountryCode=="IND"
for num 2002/2015: label variable India_cfd_lives_X "Counterfactual: India total lives affected in X"
egen India_cfd_lives_all = rowtotal(India_cfd_lives_2002-India_cfd_lives_2015), missing
label variable India_cfd_lives_all "Counterfactual: India, total 2002-2015"

**Rest of World (ex. China, India & SSA)
for num 2002/2015: egen ROW_cfd_lives_X = sum(dif_mdg_cfd_X) if CountryCode!="IND" & CountryCode!="CHN" & Region!="SSA"
for num 2002/2015: label variable ROW_cfd_lives_X "Counterfactual: Rest of World total lives affected in X"
egen ROW_cfd_lives_all = rowtotal(ROW_cfd_lives_2002-ROW_cfd_lives_2015), missing
label variable ROW_cfd_lives_all "Counterfactual: Rest of World, total 2002-2015"

save "TB_deaths_analysis.dta", replace
 



****************************************************************************
***MALARIA DEATHS
****************************************************************************
*I. Rate of progress and counterfactual calculations
	*A. Pre-MDG average annual proportional rate of progress 
	*B. Later-MDG average annual proportional rate of progress
	*C. Difference in rate of progress
	*D. Extrapolate pre-MDG rate of progress to find counterfactual
*II. Country count acceleration test and lives saved
	*A. Acceleration test - country count
	*B. Lives improved compared to BAU trajectory
****************************************************************************

***************************************************
**I. RATE OF PROGRESS AND COUNTERFACTUAL CALCULATIONS
***************************************************
***Initial year = 2000
***Final year of data available = 2013

use "malaria_deaths_WHO.dta", clear

**************************************
 * A * Pre-MDG proportional rate of change
**************************************
   *use rate from 2000-2005 with assumption that rate is unchanged from pre - 2000s
   *flipping direction of slope so higher rate of change = improvement
   *change 0's to .05 to enable ARC calculations for countries achieving zero deaths in 2013
foreach v of varlist n2000-n2013{
replace `v'=.05 if `v'==0
}
gen rp0005 = 1- (n2005/n2000)^(1/5) 

**Exclude countries from rate of progress calculations if they have a value in 2000 of <100 malaria deaths
replace rp0005 = . if n2000<100 & n2000!=.

**************************************
 * B * Later-MDG proportional rate of change
**************************************
gen rp0513 = 1- (n2013/n2005)^(1/8)

 * Return .05's to 0's now that ARC calculation is complete
foreach v of varlist n2000-n2013{
replace `v'=0 if `v'<1 & `v'!=.
}
**Exclude countries from rate of progress calculations if they have a value in 2000 of <100 malaria deaths
replace rp0513 = . if n2000<100 & n2000!=.

**************************************
 * C * Find the difference in rate of progress before/after MDG
**************************************
 
*2000-2005 v. 2005-2013
local listb "0005" // years 2000-2005
local listc "0513" // years 2005-2013

foreach pre of local listb{
foreach pos of local listc{

gen difrp`pre'_`pos' = rp`pos'-rp`pre' 
label var difrp`pre'_`pos' "dif in rate of progress, (`pos')-(`pre')"

gen fulldata`pre'_`pos'=0
replace fulldata`pre'_`pos' = 1 if rp`pre'!=. & rp`pos'!=.
label var fulldata`pre'_`pos' "# countries with data for all years `pre' `pos'"
}
}
**************************************
 * D. Use rate of progress from 2000-2005 to generate counterfactual for 2006-2013
**************************************
   * at 2000-2005 rates
local listb "0005"

foreach pre of local listb{
for num 2006/2013: gen cf`pre'_X = n2005*((1-rp`pre')^(X-2005))
for num 2006/2013: label variable cf`pre'_X "Counterfactual B: Malaria deaths X extrapolation from `pre' rates"
}
save "malaria_deaths_WHO_rpfull.dta", replace


***************************************************
**II. COUNTRY COUNT ACCELERATION TEST AND LIVES IMPROVED 
***************************************************

use "malaria_deaths_WHO_rpfull.dta", clear

**************************************************
 * A * Acceleration test - simple country count
**************************************************
* Diference in speed test - accerlation, using 2000-2005 as baseline
 **Any acceleration
gen difrp0005_0513_any = 0
replace difrp0005_0513_any=1 if difrp0005_0513>0
replace difrp0005_0513_any = . if difrp0005_0513==.
replace difrp0005_0513_any = 0 if rp0513<=0 & rp0513!=. & difrp0005_0513!=. //exclude countries from acceleration count that still have increasing malaria deaths (even if at a slower rate)

 **"real" acceleration >= .01 (1 percentage point difference between two periods)
gen difrp0005_0513_1 = 0
replace difrp0005_0513_1 = 1 if difrp0005_0513>=.01
replace difrp0005_0513_1 = . if difrp0005_0513==.
replace difrp0005_0513_1 = 0 if rp0513<=0 & rp0513!=. & difrp0005_0513!=. //exclude countries from acceleration count that still have increasing malaria deaths (even if at a slower rate)

save "malaria_deaths_WHO_cfmdg.dta", replace

************************************************** 
 * B. * Use counterfactuals to determine number of lives saved above BAU trajectory
**************************************************
   *1. find difference between counterfactual and actual in 2013
   *2. sum difference to find lives saved
 
 ** Counterfactual: 2000-2005 - estimated difference between counterfactual estrapolated from RP00-05 and actual
*************************
gen dif_mdg_cf_2013 = cf0005_2013 - n2013 
label variable dif_mdg_cf_2013 "Counterfactual: est. lives affected in 2013, extrap 2000-2005"

 * Calculate total lives affected in different groups
**All developing countries
egen dev_cf_lives_2013 = sum(dif_mdg_cf_2013)
label variable dev_cf_lives_2013 "Counterfactual: developing world total lives affected in 2013"

**By Region
bysort Region: egen region_cf_lives_2013 = sum(dif_mdg_cf_2013)
label variable region_cf_lives_2013 "Counterfactual: by region, total lives affected in 2013"

**LIC
egen LIC_cf_lives_2013 = sum(dif_mdg_cf_2013) if IncomeGroup=="L" 
label variable LIC_cf_lives_2013 "Counterfactual: LIC total lives affected in 2013"

**LIC ex. India
egen LICxind_cf_lives_2013 = sum(dif_mdg_cf_2013) if IncomeGroup=="L" & CountryCode!="IND"
label variable LICxind_cf_lives_2013 "Counterfactual: LIC ex-India total lives affected in 2013"

**MIC
egen MIC_cf_lives_2013 = sum(dif_mdg_cf_2013) if IncomeGroup!="L" 
label variable MIC_cf_lives_2013 "Counterfactual: MIC total lives affected in 2013"

**MIC ex. China
egen MICxchn_cf_lives_2013 = sum(dif_mdg_cf_2013) if IncomeGroup!="L" & CountryCode!="CHN"
label variable MICxchn_cf_lives_2013 "Counterfactual: MIC ex-China total lives affected in 2013"

**Ex. China & India
egen xChiInd_cf_lives_2013 = sum(dif_mdg_cf_2013) if CountryCode!="CHN" & CountryCode!="IND"
label variable xChiInd_cf_lives_2013 "Counterfactual: ex-China & India total lives affected in 2013"

**Rest of World (ex. China, India, SSA)
egen ROW_cf_lives_2013 = sum(dif_mdg_cf_2013) if CountryCode!="IND" & CountryCode!="CHN" & Region!="SSA"
label variable ROW_cf_lives_2013 "Counterfactual: ROW total lives saved in 2013"

replace IndicatorName = "Estimated number of malria deaths" if IndicatorName==""
save "malaria_deaths_analysis.dta", replace




****************************************************************************
***ANTIRETROVIRAL THERAPY COVERAGE
****************************************************************************
*I. Rate of progress and counterfactual calculations
	*A. Pre/Post-MDG average annual percentage point rate of progress 
	*B. Difference in rate of progress
	*C. Extrapolate pre-MDG rate of progress to find counterfactual
*II. Country count acceleration test and deaths averted
	*A. Acceleration test - country count 
	*B. Deaths averted due to access to treatment
****************************************************************************

***************************************************
**I. RATE OF PROGRESS AND COUNTERFACTUAL CALCULATIONS
***************************************************
**INITIAL YEAR = 2000
**FINAL YEAR = 2015

local lista "antiretro_pct"
foreach p of local lista{

use "`p'_clean", clear

**************************************
* A. Calculate Rate of Progress: average annual percentage point change 
**************************************
**since no 1990s data, use 2000-2002 to calculate initial rate of progress

*Pre-MDG Average Annual Percentage Point Change
gen rp0002 = (n2002-n2000)/2

*Post-MDG Average Annual Percentage Point Change
gen rp0215 = (n2015-n2002)/13
 
**************************************
 * B * Find the difference in rate of progress before/after MDG
**************************************

*uses early 2000s at rate of progress
local listb "0002" //years 00-02
local listc "0215" //years 02-15

foreach pre of local listb{
foreach pos of local listc{

gen difrp`pre'_`pos' = rp`pos'-rp`pre' 
label var difrp`pre'_`pos' "dif in rate of progress, (`pos')-(`pre')"

gen fulldata`pre'_`pos'=0
replace fulldata`pre'_`pos' = 1 if rp`pre'!=. & rp`pos'!=.
label var fulldata`pre'_`pos' "# countries with data for all years `pre' `pos'"
}
}
**************************************
* C. Use rate of progress to generate counterfactual for 2003-2015
**************************************
** at 1990-2000 and 
local listb "0002"

foreach pre of local listb{
for num 2003/2015: gen cf`pre'_X = n2002+(rp`pre'*(X - 2002)) if rp`pre'!=.
for num 2003/2015: replace cf`pre'_X = 1 if cf`pre'_X > 1 & cf`pre'_X!=. //can't have more than 100%
for num 2003/2015: label variable cf`pre'_X "`p' X extrapolation from `pre' rates"
}

save "`p'_rpfull", replace
}

***************************************************
**II. COUNTRY COUNT ACCELERATION TEST AND DEATHS AVERTED
***************************************************

use "antiretro_pct_rpfull.dta", clear

**************************************
 * A * Acceleration test - simple country count
**************************************

* Difference in rate of progress - acceleration from 2000-2002 v. 2002-2015
**Any acceleration
gen difrp0002_0215_any = 0
 replace difrp0002_0215_any = 1 if difrp0002_0215>0
 replace difrp0002_0215_any = . if difrp0002_0215==.
 replace difrp0002_0215_any = 0 if rp0215<=0 & difrp0002_0215_any!=.  
 
**"Real" acceleration = .33 percentage points difference 
gen difrp0002_0215_1 = 0
 replace difrp0002_0215_1 = 1 if difrp0002_0215>=.0033
 replace difrp0002_0215_1 = . if difrp0002_0215==.
 replace difrp0002_0215_1 = 0 if rp0215<=0 & difrp0002_0215_1!=.  

save "antiretro_pct_analysis.dta", replace

**************************************************************************************
 * B * Deaths averted due to treatment (using UNAIDS measure)
**************************************************************************************
*For lives saved calculations, instead of using a counterfactual, we use UNAIDS "deaths averted due to treatment", which is available at the developing country aggregate
*1) To estimate the total number of people with access to treatment, we use UNAIDS data
*2) Calculate each country's share of developing country total with access to ART
*3) Apply each country's access share to UNAIDS "deaths averted due to treatment" total
*4) For lives saved tables, we scale this number down by 2.5% to reduce the estimated overlap with deaths in children under-5 in 2002

use "antiretro_pct_analysis.dta", clear

*1) Find total number of people in sample countries with data that have access to treatment and then find each country's share of this total
egen total_2015dev = sum(ARTppl2015)
gen countryshare_treatment = ARTppl2015/total_2015dev

*2) Apply each country's share with treatment to UNAIDS total deaths averted
 **Total deaths averted from 2001-2015 = 7,889,000
 **Reduce total by 2.5% to account for overlap with deaths in children under 5: 7,691,000 deaths averted from 2001-2015
gen deathsaverted = countryshare_treatment * 7691000
 label variable deathsaverted "Approximate HIV/AIDS deaths averted due to treatment, scaled down by 2.5% to remove overlap with deaths in children under 5"

*3) Calculate total deaths averted in relevant categories of interest
	**A. all developing
egen dev_lives = sum(deathsaverted)
 label variable dev_lives "All developing approximate HIV/AIDS deaths averted due to treatment"

	**B. developing ex. China and India
egen xChiInd_lives = sum(deathsaverted) if CountryCode!="CHN" & CountryCode!="IND"
 label variable xChiInd_lives "Developing ex. China and India approximate HIV/AIDS deaths averted due to treatment"

	**C. By Region
bysort Region: egen reg_lives = sum(deathsaverted)
 label variable reg_lives "By Region approximate HIV/AIDS deaths averted due to treatment"

	**D. LIC ex. India
egen LICxInd_lives = sum(deathsaverted) if IncomeGroup=="L" & CountryCode!="IND"
 label variable LICxInd_lives "LIC ex. India approximate HIV/AIDS deaths averted due to treatment"

	**E. MIC ex. China
egen MICxInd_lives = sum(deathsaverted) if IncomeGroup!="L" & CountryCode!="CHN"
 label variable MICxInd_lives "MIC ex. China approximate HIV/AIDS deaths averted due to treatment"

	**F. Rest of World (ex. China, India, SSA)
egen ROW_lives = sum(deathsaverted) if CountryCode!="CHN" & CountryCode!="IND" & Region!="SSA"
 label variable ROW_lives "Rest of World (ex. China India SSA) approximate HIV/AIDS deaths averted due to treatment"

save "antiretro_pct_deathsaverted.dta", replace


