function [range,freqlist] = dbrange(d,list,varargin)
% dbrange  Find a range that encompasses the ranges of the listed tseries objects.
%
% Syntax
% =======
%
%     [RANGE,FREQ] = dbrange(D,...)
%     [RANGE,FREQ] = dbrange(D,LIST,...)
%
% Input arguments
% ================
%
% * `D` [ struct ] - Input database.
%
% * `LIST` [ char | cellstr | `Inf` ] - List of tseries objects that will
% be included in the range search; `Inf` means all tseries objects existing in
% the input databases will be included.
%
% Output arguments
% =================
%
% * `RANGE` [ numeric | cell ] - Range that encompasses the observations of
% the tseries objects in the input database; if tseries objects with
% different frequencies exist, the ranges are returned in a cell array.
%
% * `FREQ` [ numeric ] - Date frequencies of the returned ranges.
%
% Options
% ========
%
% * `'startDate='` [ *`'maxRange'`* | `'minRange'` ] - `'maxRange'` means
% the `range` will start at the earliest start date of all tseries included
% in the search; `'minRange'` means the `range` will start at the latest
% start date found.
%
% * `'endDate='` [ *`'maxRange'`* | `'minRange'` ] - `'maxRange'` means the
% `range` will end at the latest end date of all tseries included in the
% search; `'minRange'` means the `range` will end at the earliest end date.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright  2007-2012 Jaromir Benes.

if ~exist('list','var') || isequal(list,Inf)
    list = fieldnames(d);
elseif ischar(list)
    list = regexp(list,'\w+','match');
end

% Validate input arguments.
P = inputParser();
P.addRequired('d',@isstruct);
P.addRequired('list',@iscellstr);
P.parse(d,list);

% Validate options.
options = passvalopt('data.dbrange',varargin{:});

%**************************************************************************

freqlist = [1,2,4,6,12,0];
startdates = cell(1,6);
enddates = cell(1,6);
range = cell(1,6);
nlist = numel(list);

for i = 1 : nlist
    if isfield(d,list{i}) && istseries(d.(list{i}))
        x = d.(list{i});
        freqindex = freq(x) == freqlist;
        if any(freqindex)
            startdates{freqindex}(end+1) = startdate(x);
            enddates{freqindex}(end+1) = enddate(x);
        end
    end
end

if isanychari(options.startdate,{'maxrange','unbalanced'})
    startdates = cellfun(@min,startdates,'uniformOutput',false);
else
    startdates = cellfun(@max,startdates,'uniformOutput',false);
end

if isanychari(options.enddate,{'maxrange','unbalanced'})
    enddates = cellfun(@max,enddates,'uniformOutput',false);
else
    enddates = cellfun(@min,enddates,'uniformOutput',false);
end

for i = find(~cellfun(@isempty,startdates))
    range{i} = startdates{i} : enddates{i};
end

nonempty = ~cellfun(@isempty,range);
if sum(nonempty) == 0
    range = [];
    freqlist = [];
elseif sum(nonempty) == 1
    range = range{nonempty};
    freqlist = freqlist(nonempty);
else
    range = range(nonempty);
    freqlist = freqlist(nonempty);
end

end