function this = subsasgn(this,s,x)
% subsasgn  Subscripted assignment for rhsmodel objects.
%
% Syntax for assigning parameterisations from other model objects
% ================================================================
%
%     b(index) = c
%
% Syntax for deleting specified parameterisations
% ================================================
%
%     b(index) = []
%
% Syntax to assign parameter values
% ==================================
%
%     b.name = x
%     b.name(index) = x
%
% Input arguments
% ================
%
% * `b` [ model ] - rhsmodel object that will be assigned new
% parameterisations or new parameter values.
%
% * `c` [ model ] - rhsmodel object compatible with `b` whose
% parameterisations will be assigned (copied) into `b`.
%
% * `index` [ numeric ] - Index of parameterisations that will be assigned
% or deleted.
%
% * `name` [ char ] - Name of a variable, shock, or parameter.
%
% * `x` [ numeric ] - A value (or a vector of values) that will be assigned
% to a parameter or variable named `name`.
%
% Output arguments
% =================
%
% * `b` [ model ] - rhsmodel object with newly assigned or deleted
% parameterisations, or with newly assigned parameters.
%
% Description
% ============
%
% Example
% ========
%
% Expand the number of parameterisations in a rhsmodel object that has
% initially just one parameterisation:
%
%     b(1:10) = b;
%
% The parameterisation is simply copied ten times within the model object.

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

%**************************************************************************

if isequal(s(1).type,'()')
   if length(s) > 1 || (~isa(x,'rhsmodel') && ~isempty(x))
      utils.error('rhsmodel', ...
         'Invalid subscripted assignment to rhsmodel object.');
   end
   lhsindex = s(1).subs{1};
   if islogical(lhsindex)
      nlhs = sum(lhsindex);
   else
      nlhs = length(lhsindex);
   end
   nrhs = size(x.Assign,3);
   if isempty(x)
      this.Assign(:,:,lhsindex) = [];
      this.stdcorr(:,:,lhsindex) = [];
   else
      if nlhs > nrhs
         x.Assign(:,:,end+1:nlhs) = x.Assign(:,:,end*ones([1,nlhs-end]));
         x.stdcorr(:,:,end+1:nlhs) = x.stdcorr(:,:,end*ones([1,nlhs-end]));
      end
      this.Assign(:,:,lhsindex) = x.Assign;
      this.stdcorr(:,:,lhsindex) = x.stdcorr;
   end
elseif isequal(s(1).type,'.')
   name = s(1).subs;
   [assignpos,stdcorrpos] = mynameposition(this,name);
   if sum(assignpos) + sum(stdcorrpos) > 1
      % More than one name matches found; this is probably a regexp call,
      % which is not allowed in `subsasgn` or `subsref`.
      utils.error('rhsmodel', ...
         'Invalid subscripted assignment to rhsmodel object.');
   end
   s(1) = [];
   index = assignpos;
   if any(assignpos)
      if isempty(s)   
         this.Assign(1,index,:) = x;
      else
         this.Assign(1,index,:) = ...
            subsasgn(this.Assign(1,index,:),s,x);
      end
   elseif any(stdcorrpos)
      if isempty(s)   
         this.stdcorr(1,stdcorrpos,:) = x;
      else
         this.stdcorr(1,stdcorrpos,:) = ...
            subsasgn(this.Assign(1,stdcorrpos,:),s,x);
      end
   else      
      utils.error('rhsmodel', ...
         'This name does not exist in the rhsmodel object: ''%s''.', ...
         name);
   end
end

end