function THIS = subsref(THIS,s)
% subsref  Subscripted reference for plan objects.
%
% Syntax
% =======
%
%     P = P(STARTDATE:ENDDATE)
%
% Input arguments
% ================
%
% * `P` [ plan ] - Simulation plan.
%
% Output aguments
% ================
%
% * `P` [ plan ] - Simulation plan reduced or expanded to the new range,
% `STARTDATE:ENDDATE`.
%
% * `STARTDATE` [ numeric ] - New start date for the simulation plan.
%
% * `ENDDATE` [ numeric ] - New end date for the simulation plan.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

%**************************************************************************

if isequal(s(1).type,'{}') ...
        && length(s(1).subs) == 1 && length(s(1).subs{1}) == 1
    shift = s(1).subs{1}(1);
    THIS.startdate = THIS.startdate + shift;
    THIS.enddate = THIS.enddate + shift;
    s(1) = [];
    if ~isempty(s)
        THIS = builtin('subsref',THIS,s);
    end
elseif isequal(s(1).type,'()') ...
        && length(s(1).subs) == 1
    newrange = s(1).subs{1};
    if ~datcmp(THIS.startdate,newrange(1)) ...
            || ~datcmp(THIS.enddate,newrange(end))
        dochkfreq();
        dochngrange();
        THIS.startdate = newrange(1);
        THIS.enddate = newrange(end);
    end
elseif isequal(s(1).type,'.')
    THIS = builtin('subsref',THIS,s(1));
    s(1) = [];
    if ~isempty(s)
        THIS = builtin('subsref',THIS,s);
    end
else
    utils.error('plan', ...
        'Invalid subscripted reference to a plan object.');
end

% Nested functions.

%**************************************************************************
    function dochkfreq()
        if ~freqcmp(THIS.startdate,newrange(1)) ...
                || ~freqcmp(THIS.enddate,newrange(end))
            utils.error('plan', ...
                ['Invalid date frequency of the new range ', ...
                'in subscripted reference to plan object.']);
        end
    end
% dochkfreq().

%**************************************************************************
    function dochngrange()
        nx = size(THIS.xanchors,1);
        nnreal = size(THIS.nanchorsreal,1);
        nnimag = size(THIS.nanchorsimag,1);
        nc = size(THIS.canchors,1);
        nq = size(THIS.qanchors,1);
        
        if ~isinf(newrange(1))
            if newrange(1) < THIS.startdate
                npre = round(THIS.startdate - newrange(1));
                THIS.xanchors = [false(nx,npre),THIS.xanchors];
                THIS.nanchorsreal = [false(nnreal,npre),THIS.nanchorsreal];
                THIS.nanchorsimag = [false(nnimag,npre),THIS.nanchorsimag];
                THIS.canchors = [false(nc,npre),THIS.canchors];
                THIS.qanchors = [false(nq,npre),THIS.qanchors];
            elseif newrange(1) > THIS.startdate
                npre = round(newrange(1) - THIS.startdate);
                THIS.xanchors = THIS.xanchors(:,npre+1:end);
                THIS.nanchorsreal = THIS.nanchorsreal(:,npre+1:end);
                THIS.nanchorsimag = THIS.nanchorsimag(:,npre+1:end);
                THIS.canchors = THIS.canchors(:,npre+1:end);
                THIS.qanchors = THIS.qanchors(:,npre+1:end);
            end
        end
        
        if ~isinf(newrange(end))
            if newrange(end) > THIS.enddate
                npost = round(newrange(end) - THIS.enddate);
                THIS.xanchors = [THIS.xanchors,false(nx,npost)];
                THIS.nanchorsreal = [THIS.nanchorsreal,false(nnreal,npost)];
                THIS.nanchorsimag = [THIS.nanchorsimag,false(nnimag,npost)];
                THIS.canchors = [THIS.canchors,false(nc,npost)];
                THIS.qanchors = [THIS.qanchors,false(nq,npost)];
            elseif newrange(end) < THIS.enddate
                npost = round(THIS.enddate - newrange(end));
                THIS.xanchors = THIS.xanchors(:,1:end-npost);
                THIS.nanchorsreal = THIS.nanchorsreal(:,1:end-npost);
                THIS.nanchorsimag = THIS.nanchorsimag(:,1:end-npost);
                THIS.canchors = THIS.canchors(:,1:end-npost);
                THIS.qanchors = THIS.qanchors(:,1:end-npost);
            end
        end
        
    end
% dochkfreq().

end