function THIS = nonlinearise(THIS,varargin)
% nonlinearise  Select equations for simulation in an exact non-linear mode.
%
% Syntax
% =======
%
%     P = nonlinearise(P)
%     P = nonlinearise(P,RANGE)
%     P = nonlinearise(P,LIST,RANGE)
%
% Input arguments
% ================
%
% * `P` [ plan ] - Simulation plan object.
%
% * `LIST` [ cellstr ] - List of labels for equations that will be simulated in
% an exact non-linear mode; all selected equations must be marked by an
% [`=#`](modellang/exactnonlin) mark in the model file. If `LIST` is not
% specified, all equations marked in the model file will be simulated in a
% non-linear mode.
%
% * `RANGE` [ numeric ] - Date range on which the equations will be
% simulated in an exact non-linear mode; currently, the range must start at
% the start of the plan range. If `RANGE` is not specified, the equations
% are non-linearised over the whole simulation range.
%
% Output arguments
% =================
%
% * `P` [ plan ] - Simulation plan with information on exact non-linear
% simulation included.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

if isempty(varargin)
    list = Inf;
    dat = THIS.startdate : THIS.enddate;
else
    % List of equations or Inf for all.
    if isequal(varargin{1},Inf) || ischar(varargin{1}) || iscellstr(varargin{1})
        list = varargin{1};
        varargin(1) = [];
    else
        list = Inf;
    end
    % Dates.
    dat = varargin{1};
end

% Parse input arguments.
ip = inputParser();
ip.addRequired('P',@(x) isa(x,'plan'));
ip.addRequired('LIST',@(x) isequal(x,Inf) || ischar(x) || iscellstr(x));
ip.addRequired('DAT', ...
    @(x) isnumeric(x) && all(datfreq(x) == datfreq(THIS.startdate)));
ip.parse(THIS,list,dat);

%**************************************************************************

nper = round(THIS.enddate - THIS.startdate + 1);
datindex = round(dat - THIS.startdate + 1);
if any(datindex < 1 | datindex > nper)
    utils.error('plan', ...
        'Some of the non-linearised dates are out of plan range.');
end

if isequal(list,Inf)
    THIS.qanchors(:,datindex) = true;
    return
end

if ischar(list)
    list = {list};
end

nlist = length(list);
found = true(1,nlist);
for i = 1 : nlist
    usrlabel = list{i};
    nusrlabel = length(usrlabel);
    if nusrlabel > 3 && strcmp(usrlabel(end-2:end),'...')
        eqtnindex = strncmp(THIS.qlist,usrlabel,nusrlabel-3);
    else
        eqtnindex = strcmp(THIS.qlist,usrlabel);
    end
    if any(eqtnindex)
        THIS.qanchors(eqtnindex,datindex) = true;
    else
        found(i) = false;
    end
end

if any(~found)
    utils.error('plan', ...
        'Cannot non-linearise this equation: ''%s''.', ...
        list{~found});
end

end