classdef VAR < userdataobj
    % VAR  Vector autoregressions: VAR objects and functions.
    %
    % VAR methods:
    %
    % Constructor
    % ============
    %
    % * [`VAR`](VAR/VAR) - Create new reduced-form VAR object.
    %
    % Getting information about VAR objects
    % ======================================
    %
    % * [`comment`](VAR/comment) - Get or set user comments in an IRIS object.
    % * [`companion`](VAR/companion) - Matrices of first-order companion VAR.
    % * [`eig`](VAR/eig) - Eigenvalues of a VAR process.
    % * [`get`](VAR/get) - Query VAR object properties.
    % * [`isexplosive`](VAR/isexplosive) - True if any eigenvalue is outside unit circle.
    % * [`isstationary`](VAR/isstationary) - True if all eigenvalues are within unit circle.
    % * [`length`](VAR/length) - Number of alternative parameterisations in VAR object.
    % * [`mean`](VAR/mean) - Mean of VAR process.
    % * [`sspace`](VAR/sspace) - Quasi-triangular state-space representation of VAR.
    % * [`userdata`](VAR/userdata) - Get or set user data in an IRIS object.
    %
    % Referencing VAR objects
    % ========================
    %
    % * [`subsasgn`](VAR/subsasgn) - Subscripted assignment for VAR objects.
    % * [`subsref`](VAR/subsref) - Subscripted reference for VAR objects.
    %
    % Simulation and forecasting
    % ===========================
    %
    % * [`ferf`](VAR/ferf) - Forecast error response function.
    % * [`forecast`](VAR/forecast) - Unconditional or conditional forecasts.
    % * [`instrument`](VAR/instrument) - Define conditioning instruments for VAR model.
    % * [`resample`](VAR/resample) - Resample from a VAR object.
    % * [`simulate`](VAR/simulate) - Simulate VAR model.
    %
    % Manipulating VARs
    % ==================
    %
    % * [`alter`](VAR/alter) - Expand or reduce number of alternative parameterisations.
    % * [`backward`](VAR/backward) - Backward VAR process.
    % * [`demean`](VAR/demean) - Remove constant from VAR object and demean associated data.
    % * [`horzcat`](VAR/horzcat) - Combine two compatible VAR objects in one object with multiple parameterisations.
    % * [`integrate`](VAR/integrate) - Integrate VAR process and data associated with it.
    %
    % Stochastic properties
    % ======================
    %
    % * [`acf`](VAR/acf) - Autocovariance and autocorrelation functions for VAR variables.
    % * [`fmse`](VAR/fmse) - Forecast mean square error matrices.
    % * [`vma`](VAR/vma) - Matrices describing the VMA representation of a VAR process.
    % * [`xsf`](VAR/xsf) - Power spectrum and spectral density functions for VAR variables.
    %
    % Estimation, identification, and statistical tests
    % ==================================================
    %
    % * [`estimate`](VAR/estimate) - Estimate a reduced-form VAR or BVAR.
    % * [`lrtest`](VAR/lrtest) - Likelihood ratio test for VAR models.
    % * [`portest`](VAR/portest) - Portmanteau test for autocorrelation in VAR residuals.
    %
    % Getting on-line help on VAR functions
    % ======================================
    %
    %     help VAR
    %     help VAR/function_name
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        A = []; % Transition matrix.
        K = []; % Constant vector.
        G = []; % Coefficients at co-integrating vector in VEC form.
        Omega = []; % Cov of reduced-form residuals.
        Sigma = []; % Cov of parameters.
        T = []; % Shur decomposition of the transition matrix.
        U = []; % Schur transformation of the variables.
        range = []; % User range.
        fitted = []; % Effective estimation sample.
        aic = []; % Akaike info criterion.
        sbc = []; % Schwartz bayesian criterion.
        Rr = []; % Parameter restrictions.
        nhyper = NaN; % Number of estimated hyperparameters.
        eigval = []; % Vector of eigenvalues.
        ynames = {}; % Names of endogenous variables.
        enames = {}; % Names of residuals.
        inames = {}; % Names of conditioning instruments.
        ieqtn = {}; % Expressions for conditioning instruments.
        Zi = []; % Measurement matrix for conditioning instruments.
    end
    
    methods
        varargout = acf(varargin)
        varargout = backward(varargin)
        varargout = companion(varargin)
        varargout = demean(varargin)
        varargout = eig(varargin)
        varargout = estimate(varargin)
        varargout = ferf(varargin)
        varargout = fmse(varargin)
        varargout = forecast(varargin)
        varargout = get(varargin)
        varargout = horzcat(varargin)
        varargout = instrument(varargin)
        varargout = integrate(varargin)
        varargout = isexplosive(varargin)
        varargout = isstationary(varargin)
        varargout = length(varargin)
        varargout = lrtest(varargin)
        varargout = mean(varargin)
        varargout = portest(varargin)
        varargout = resample(varargin)
        varargout = simulate(varargin)
        varargout = sspace(varargin)
        varargout = vma(varargin)
        varargout = xsf(varargin)
        varargout = subsref(varargin)
        varargout = subsasgn(varargin)
    end
    
    methods (Hidden)
        varargout = disp(varargin)
        varargout = end(varargin)
        varargout = isempty(varargin)
        varargout = saveobj(varargin)
        varargout = specget(varargin)
        varargout = SVAR(varargin)
        varargout = mysystem(varargin)
    end
    
    methods (Access=protected,Hidden)
        varargout = myinfocrit(varargin)
        varargout = myschur(varargin)
        varargout = mysubsalt(varargin)
        varargout = sizeof(varargin)
        varargout = size(varargin)
    end
    
    methods (Static,Hidden)
        error(varargin)
        warning(varargin)
        varargout = loadobj(varargin)
        
        varargout = glsq(varargin)
        varargout = restrict(varargin)
        varargout = stackdata(varargin)
        varargout = inputdata(varargin)
        varargout = datarequest(varargin)
        varargout = outputdata(varargin)
    end
    
    methods
        
        function this = VAR(varargin)
            % VAR  Create new reduced-form VAR object.
            %
            % Syntax
            % =======
            %
            %     V = VAR()
            %     V = VAR(YNAMES)
            %     V = VAR(YNAMES,ENAMES)
            %
            % Output arguments
            % =================
            %
            % * `V` [ VAR ] - New empty VAR object.
            %
            % * `YNAMES` [ cellstr | char | function_handle ] - Names of VAR variables.
            %
            % * `ENAMES` [ cellstr | char | function_handle ] - Names of VAR residuals.
            %
            % Description
            % ============
            %
            % This function creates a new empty VAR object. It is usually followed by
            % the [estimate](VAR/estimate) function to estimate the VAR parameters on
            % data.
            %
            % Example
            % ========
            %
            % To estimate a VAR, you first need to create an empty VAR object, and then
            % run the [VAR/estimate](VAR/estimate) function on it, e.g.
            %
            %     V = VAR();
            %     [V,D] = estimate(V,D,list,range);
            %
            % or simply
            %
            %     [V,D] = estimate(V,D,list,range);
            %
            
            % -IRIS Toolbox.
            % -Copyright (c) 2007-2012 Jaromir Benes.
            
            this = this@userdataobj();
            if nargin == 0
                return
            elseif nargin == 1
                if isa(varargin{1},'VAR')
                    this = varargin{1};
                    return
                elseif isstruct(varargin{1})
                    % Convert struct to VAR object.
                    list = properties(this);
                    for i = 1 : length(list)
                        try %#ok<TRYNC>
                            this.(list{i}) = varargin{1}.(list{i});
                        end
                    end
                    % Populate triangular representation.
                    if isempty(this.T) || isempty(this.U) || isempty(this.eigval)
                        this = myschur(this);
                    end
                    % Populate information criteria.
                    if isempty(this.aic) || isempty(this.sbc)
                        this = myinfocrit(this);
                    end
                    return
                elseif iscellstr(varargin{1}) || ischar(varargin{1})
                    this = setnames(this,varargin{1});
                end
            elseif nargin == 2 ...
                    && (iscellstr(varargin{1}) || ischar(varargin{1})) ...
                    && (iscellstr(varargin{2}) || ischar(varargin{2}))
                this = setnames(this,varargin{1},varargin{2});
            else
                options = passvalopt('VAR.VAR',varargin{:});
                if ~isempty(options.userdata)
                    this = userdata(this,options.userdata);
                end
            end
        end
    end
    
end