classdef figure < report.tabularobj
    % figure  Start new figure in report.
    %
    % Syntax
    % =======
    %
    %     P.figure(CAPTION,...)
    %     P.figure(CAPTION,FIG)
    %
    % Input arguments
    % ================
    %
    % * `P` [ struct ] - Report object created by the
    % [`report.new`](report/new) function.
    %
    % * `CAPTION` [ char | cellstr ] - Title and subtitle displayed at the top
    % of the figure.
    %
    % * `FIG` [ numeric ] - Handle to a graphics figure created by the user
    % that will be captured and inserted in the report; the report figure
    % object must have no childern-graphs in this case; if no handle is
    % specified, the current figure will be captured.
    %
    % Options
    % ========
    %
    % * `'captionTypeface='` [ cellstr | char | *`'\large\bfseries'`* ] - LaTeX
    % format commands for typesetting the figure caption; the subcaption format
    % can be entered as the second cell in a cell array.
    %
    % * `'close='` [ *`true`* | `false` ] - (Inheritable from parent objects) Close
    % Matlab the underlying figure window when finished.
    %
    % * `'separator='` [ char | *`'\medskip\par'`* ] - (Inheritable from parent
    % objects) LaTeX commands that will be inserted after the figure.
    %
    % * `'figureOptions='` [ cell | *empty* ] - Figure options that will be
    % applied to the figure handle after all graphs are drawn.
    %
    % * `'figureScale='` [ numeric | *`0.85`* ] - (Inheritable from parent objects)
    % Scale of the figure in the LaTeX document.
    %
    % * `'footnote='` [ char | *empty* ] - Footnote at the figure title; only
    % shows if the title is non-empty.
    %
    % * `'sideways='` [ `true` | *`false`* ] - (Inheritable from parent objects)
    % Print the table rotated by 90 degrees.
    %
    % * `'style='` [ struct | *empty* ] - Apply this cascading style structure
    % to the figure; see [`qstyle`](qreport/qstyle).
    %
    % * `'subplot='` [ numeric | *`'auto'`* ] - (Inheritable from parent objects)
    % Subplot division of the figure.
    %
    % * `'typeface='` [ char | *empty* ] - (Not inheritable from parent objects)
    % LaTeX code specifying the typeface for the figure as a whole; it must use
    % the declarative forms (such as `\itshape`) and not the command forms
    % (such as `\textit{...}`).
    %
    % * `'visible='` [ `true` | *`false`* ] - (Inheritable from parent objects)
    % Visibility of the underlying Matlab figure window.
    %
    % Generic options
    % ================
    %
    % See help on [generic options](report/Contents) in report objects.
    %
    % Description
    % ============
    %
    % Figures are top-level report objects and cannot be nested within other
    % report objects, except [`align`](report/align). Figure objects can have
    % the following types of children:
    %
    % * [`graph`](report/graph);
    % * [`empty`](report/empty).
    %
    % Example
    % ========
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        savefig = [];
        figfile = '';
        handle = [];
    end
    
    methods
        
        function this = figure(varargin)
            this = this@report.tabularobj(varargin{:});
            this.childof = {'report','align'};
            this.default = [this.default,{ ...
                'close',true,@islogicalscalar,true, ...
                'figureoptions',{},@(x) iscell(x) && iscellstr(x(1:2:end)),true, ...
                'figurescale','auto', ...
                @(x) isnumericscalar(x) || strcmpi(x,'auto'),true, ...
                'subplot','auto', ...
                @(x) (isnumeric(x) && numel(x) == 2) || strcmpi(x,'auto'),true, ...
                'visible',false,@islogical,true, ...
                'separator','\medskip\par',@ischar,true, ...
                'style',[],@(x) isempty(x) || isstruct(x),true, ...
                'typeface','',@ischar,false, ...                
                }];
        end
        
        % Process class-specific input arguments.
        function [THIS,varargin] = specargin(THIS,varargin)
            % Capture the figure handle in case this figure object will
            % have no children. Do not use gcf() though because that would
            % create a new empty figure if none exists -- we do not want
            % that.
            if ~isempty(varargin) && length(varargin{1}) == 1 ...
                    && all(ishghandle(varargin{1})) ...
                    && strcmp(get(varargin{1}(1),'type'),'figure')
                % If user enters a figure handle, we create a hardcopy of
                % the figure and store it in binary form in THIS object.
                h = varargin{1};
                varargin(1) = [];
                THIS.figfile = [tempname(cd()),'.fig'];
                saveas(h,THIS.figfile);
                fid = fopen(THIS.figfile);
                THIS.savefig = fread(fid);
                fclose(fid);
            else
                % If user does not enter a figure handles, we just capture
                % the current figure if one exists. This is ready for use
                % if THIS object has no children, but no hardcopy is
                % stored. This features is mainly for bkw compatibility and
                % is not documented.
                THIS.handle = get(0,'CurrentFigure');
            end
        end
        
        function THIS = setoptions(THIS,varargin)
            THIS = setoptions@report.tabularobj(THIS,varargin{:});
            THIS.options.long = false;
        end
        
        varargout = speclatexcode(varargin)
        
    end

    methods (Access=protected,Hidden)
        
        varargout = mycompilepdf(varargin)
        varargout = myplot(varargin)
        varargout = mysubplot(varargin)
        
    end
    
end