classdef band < report.series    
% band  Add new data with lower and upper bounds to graph or table.
%
% Syntax
% =======
%
%     P.series(CAP,X,LOW,HIGH,...)
%
% Input arguments
% ================
%
% * `P` [ struct ] - Report object created by the
% [`report.new`](report/new) function.
%
% * `CAP` [ char ] - Caption used as a default legend entry in a graph,
% or in the leading column in a table.
%
% * `X` [ tseries ] - Input data with the centre of the band.
%
% * `LOW` [ tseries ] - Input data with lower bounds; can be specified
% either relative to the centre or absolute, see the option `'relative='`.
%
% * `HIGH` [ tseries ] - Input data with upper bounds; can be specified
% either relative to the centre or absolute, see the option `'relative='`.
%
% Options for table and graph bands
% ==================================
%
% * `'low='` [ char | *'Low'* ] - (Inheritable from parent objects) Mark
% used to denote the lower bound.
%
% * `'high='` [ char | *'High'* ] - (Inheritable from parent objects) Mark
% used to denote the upper bound.
%
% * `'relative='` [ *`true`* | `false` ] - (Inheritable from parent objects) If
% true, the data for the lower and upper bounds are relative to the centre,
% i.e. the bounds will be added to the centre (in this case, `LOW` must be
% negative numbers and `HIGH` must be positive numbers). If false, the
% bounds are absolute data (in this case `LOW` must be lower than `X`, and
% `HIGH` must be higher than `X`).
%
% Options for table bands
% ========================
%
% * `'bandTypeface='` [ char | *`'\footnotesize'`* ] - (Inheritable from parent
% objects) LaTeX format string used to typeset the lower and upper bounds.%
%
% Options for graph bands
% ========================
%
% * `'plotType='` [ `'errorbar'` | *`'patch'`* ] - Type of plot used to draw
% the band.
%
% * `'relative='` [ *`true`* | `false` ] - (Inheritable from parent objects) If
% true the lower and upper bounds will be, respectively, subtracted from
% and added to to the middle line.
%
% * `'white='` [ numeric | *`0.8`* ] - (Inheritable from parent objects)
% Proportion of white colour mixed with the line colour and used to fill
% the area that depicts the band.
%
% See help on [`report/series`](report/series) for other options available.
%
% Generic options
% ================
%
% See help on [generic options](report/Contents) in report objects.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        low = [];
        high = [];
    end
    
    methods
        
        function this = band(varargin)
            this = this@report.series(varargin{:});
            this.default = [this.default,{...
                'bandformat',[],@(x) isempty(x) || ischar(x),false,...
                'bandtypeface','\footnotesize',@ischar,true,...
                'low','Low',@ischar,true,...
                'high','High',@ischar,true,...
                'relative',true,@islogicalscalar,true, ...
                'white',0.8,@(x) isnumeric(x) ...
                    && all(x >= 0) && all(x <= 1),true, ...
                'plottype','patch',@(x) any(strcmpi(x,{'errorbar','line','patch'})),true,...
            }];
        end
        
        function [this,varargin] = specargin(this,varargin)
            [this,varargin] = specargin@report.series(this,varargin{:});
            if ~isempty(varargin)
                this.low = varargin{1};
                if isa(this.low,'tseries')
                    this.low = {this.low};
                end
                varargin(1) = [];
            end
            if ~isempty(varargin)
                this.high = varargin{1};
                if isa(this.high,'tseries')
                    this.high = {this.high};
                end
                varargin(1) = [];
            end
        end
        
        function THIS = setoptions(THIS,varargin)
            THIS = setoptions@report.series(THIS,varargin{:});
            if ischar(THIS.options.bandformat)
                utils.warning('report', ...
                    ['The option ''bandformat'' in report/band is obsolete ', ...
                    'and will be removed from future IRIS versions. ', ...
                    'Use ''bandtypeface'' instead.']);
                THIS.options.bandtypeface = THIS.options.bandformat;
            end
            % Check consistency of `LOW` and `HIGH` relative to `X`. This function
            % needs to be finished.
            chkconsistency(THIS);
        end
        
        varargout = speclatexcode(varargin)
        varargout = latexonerow(varargin)                
        varargout = plot(varargin)
        varargout = chkconsistency(varargin)
        
    end
    
end