function f = normal(mu,sgm)
% normal  Create function proportional to log of normal distribution.
%
% Syntax
% =======
%
%     F = logdist.normal(MEAN,STD)
%
% Input arguments
% ================
%
% * `MEAN` [ numeric ] - Mean of the normal distribution.
%
% * `STD` [ numeric ] - Std dev of the normal distribution.
%
% Output arguments
% =================
%
% * `F` [ function_handle ] - Handle to a function returning a value that
% is proportional to the log of the normal density.
%
% Description
% ============
%
% See [help on the logdisk package](logdist/Contents) for
% details on using the function handle `F`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

a = mu;
b = sgm;
mode = mu;
f = @(x,varargin) xxnormal(x,a,b,mu,sgm,mode,varargin{:});

end

% Subfunctions.

%**************************************************************************
function y = xxnormal(x,a,b,mu,sgm,mode,varargin)

if isempty(varargin)
    y = -0.5 * ((x - mu)./sgm).^2;
    return
end

switch lower(varargin{1})
    case {'proper','pdf'}
        y = 1/(sgm*sqrt(2*pi)) .* exp(-(x-mu).^2/(2*sgm^2));
    case 'info'
        y = 1/(sgm^2)*ones(size(x));
    case {'a','location'}
        y = a;
    case {'b','scale'}
        y = b;
    case 'mean'
        y = mu;
    case {'sigma','sgm','std'}
        y = sgm;
    case 'mode'
        y = mode;
    case 'name'
        y = 'normal';
    case 'draw'
        y = mu + sgm*randn(varargin{2:end});
end

end
% xxnormal().
