function f = gamma(mu,sgm)
% gamma  Create function proportional to log of gamma distribution.
%
% Syntax
% =======
%
%     F = logdist.gamma(MEAN,STD)
%
% Input arguments
% ================
%
% * `MEAN` [ numeric ] - Mean of the gamma distribution.
%
% * `STD` [ numeric ] - Std dev of the gamma distribution.
%
% Output arguments
% =================
%
% * `F` [ function_handle ] - Handle to a function returning a value that
% is proportional to the log of the gamma density.
%
% Description
% ============
%
% See [help on the logdisk package](logdist/Contents) for details on
% using the function handle `F`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

b = sgm^2/mu;
a = mu/b;
if a >= 1
    mode = (a - 1)*b;
else
    mode = NaN;
end
f = @(x,varargin) xxgamma(x,a,b,mu,sgm,mode,varargin{:});

end

% Subfunctions.

%**************************************************************************
function y = xxgamma(x,a,b,mu,sgm,mode,varargin)

y = zeros(size(x));
index = x > 0;
x = x(index);
if isempty(varargin)
    y(index) = (a-1)*log(x) - x/b;
    y(~index) = -Inf;
    return
end

switch lower(varargin{1})
    case {'proper','pdf'}
        y(index) = x.^(a-1).*exp(-x/b)/(b^a*gamma(a));
    case 'info'
        y(index) = -(a - 1)/x.^2;
    case {'a','location'}
        y = a;
    case {'b','scale'}
        y = b;
    case 'mean'
        y = mu;
    case {'sigma','sgm','std'}
        y = sgm;
    case 'mode'
        y = mode;
    case 'name'
        y = 'gamma';
    case 'draw'
        y = gamrnd(a,b,varargin{2:end});
end

end
% xxgamma().