function config = irisconfig()
% IRISCONFIG  [Not a public function] Default values for IRIS config preferences.
%
% Backend IRIS function.
% No help provided.

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

config = struct();

%**************************************************************************

% Extensions associated with Matlab Editor.
config.extensions = {'model','s','q'};

% Date preferences.
config.freqletters = 'YHQBM';
config.dateformat = 'YFP';
config.plotdateformat = 'Y:P';
config.months = { ...
    'January','February','March','April','May','June', ...
    'July','August','September','October','November','December'};
config.standinmonth = 'first';

% Reporting preferences.
config.figureposition = [0,0,500*[1.7,1]];

% Obsolete, not used.
%config.publishpreamble = '';
%config.reportpreamble = '';

% Tseries preferences.
config.tseriesformat = '';

% TeX/LaTeX paths to executables.
% Use `kpswhich` to find TeX components.
[config.latexpath,folder] = findtexmf('latex');
config.dvipspath = locatefile('dvips',folder);
config.dvipdfmpath  = locatefile('dvipdfm',folder);
config.epstopdfpath = locatefile('epstopdf',folder);
config.ps2pdfpath = locatefile('ps2pdf',folder);
config.pdflatexpath = locatefile('pdflatex',folder);

% Model code preferences.
% Highest character allowed in model codes.
config.highcharcode = 1999;

% Execute the user's configuration file.
if exist('irisuserconfig.m','file')
    config = irisuserconfig(config);
    config.userconfigpath = which('irisuserconfig.m');
else
    config.userconfigpath = '';
end

% Validate the required options in case the user modified their values.
config.validate = struct( ...
    'extensions',@iscellstr, ...
    'freqletters',@(x) (ischar(x) && numel(x) == numel(unique(x)) && numel(x) == 5) ...
    || isequal(x,'config'), ...
    'dateformat',@(x) ischar(x) || iscellstr(x), ...
    'plotdateformat',@(x) ischar(x), ...
    'months',@(x) (iscellstr(x) && numel(x) == 12) ...
    || isequal(x,'config'), ...
    'standinmonth',@(x) (isnumeric(x) && numel(x) == 1 && x > 0) || isequal(x,'first') || isequal(x,'last') ...
    || isequal(x,'config'), ...
    ... 'reportpreamble',@ischar, ...
    ... 'publishpreamble',@ischar, ...
    'tseriesformat',@ischar, ...
    'latexpath',@ischar, ...
    'dvipspath',@ischar, ...
    'dvipdfmpath',@ischar, ...
    'epstopdfpath',@ischar, ...
    'ps2pdfpath',@ischar, ...
    'pdflatexpath',@ischar, ...
    'highcharcode',@(x) isempty(x) ...
    || (isnumeric(x) && numel(x) == 1 && x > 255));

list = fieldnames(config.validate);
invalid = {};
missing = {};
for i = 1 : numel(list)
    if isfield(config,list{i})
        if ~config.validate.(list{i})(config.(list{i}));
            invalid{end+1} = list{i}; %#ok<AGROW>
        end
    else
        missing{end+1} = list{i}; %#ok<AGROW>
    end
end

% Report the options that have been assigned invalid values.
if ~isempty(invalid)
    x = struct();
    x.message = sprintf(...
        '\n*** IRIS cannot start because the value supplied for this IRIS option is invalid: ''%s''.', ...
        invalid{:});
    x.identifier = 'iris:config';
    x.stack = dbstack();
    x.stack = x.stack(end);
    error(x);
end

% Report the options that are missing (=have been removed by the user).
if ~isempty(missing)
    x = struct();
    x.message = sprintf(...
        '\n*** IRIS cannot start because this IRIS option is missing from configuration struct: ''%s''.', ...
        missing{:});
    x.identifier = 'iris:config';
    x.stack = dbstack();
    x.stack = x.stack(end);
    error(x);
end

% Options that cannot be customised.
% IRIS root folder.
tmp = cd();
cd(fileparts(which('irisstartup.m')));
config.irisroot = cd();
cd(tmp);

% Read IRIS version. The IRIS version is stored in the root Contents.m
% file, and is displayed by the Matlab ver() command.
x = ver();
index = strcmp('IRIS Toolbox',{x.Name});
if any(index)
    config.version = x(index).Version;
else
    utils.warning('config', ...
        'Cannot determine the current version of IRIS.');
    config.version = '???';
end

% User cannot change these properties.
config.protected = { ...
    'userconfigpath', ...
    'irisroot', ...
    'version', ...
    'validate', ...
    'protected', ...
    };

end

% Subfunctions.

%**************************************************************************
function fpath = locatefile(file,folder)

if exist('folder','var') && ~isempty(folder)
    list = dir(fullfile(folder,[file,'.*']));
else
    list = [];
end
if length(list) == 1
    fpath = fullfile(folder,list.name);
else
    fpath = findtexmf(file);
end

end
% locatefile().