function dmc = dbminuscontrol(this,d,c)
% dbminuscontrol  Create simulation-minus-control database.
%
% Syntax
% =======
%
%    D = dbminuscontrol(M,D,C)
%
% Input arguments
% ================
%
% * `M` [ model ] - Model object on which the databases `D` and `C` are
% based.
%
% * `D` [ struct ] - Simulation database.
%
% * `C` [ struct ] - Control database.
%
% Output arguments
% =================
%
% * `D` [ struct ] - Simulation-minus-control database, in which all
% log-variables are `d.x/c.x`, and all other variables are `d.x-c.x`.
%
% Description
% ============
%
% Example
% ========
%
% We run a shock simulation in full levels using a steady-state (or
% balanced-growth-path) database as input, and then compute the deviations
% from the steady state.
%
%     d = sstatedb(m,1:40);
%     ... % Set up a shock or shocks here.
%     s = simulate(m,d,1:40);
%     s = dbextend(d,s);
%     s = dbminuscontrol(m,s,d);
%
% Note that this is equivalent to running
%
%     d = zerodb(m,1:40);
%     ... % Set up a shock or shocks here.
%     s = simulate(m,d,1:40,'deviation',true);
%     s = dbextend(d,s);
%

% The IRIS Toolbox.
% Copyright (c) 2007-2012 Jaromir Benes

IP = inputParser();
IP.addRequired('m',@(x) isa(x,'metaobj'));
IP.addRequired('d',@isstruct);
IP.addRequired('c',@isstruct);
IP.parse(this,d,c);

%**************************************************************************

list = [get(this,'yList'),get(this,'xList'),get(this,'eList')];
islog = get(this,'log');

dmc = d;
remove = false(size(list));
for i = 1 : length(list)
    if isfield(d,list{i}) && isfield(c,list{i})
        try
            if islog.(list{i})
                dmc.(list{i}) = dmc.(list{i}) / c.(list{i});
            else
                dmc.(list{i}) = dmc.(list{i}) - c.(list{i});
            end
        catch %#ok<CTCH>
            remove(i) = true;
        end
    else
        remove(i) = true;
    end
end

if any(remove)
    dmc = rmfield(dmc,list(remove));
end

end