function varargout = llf(x,dates,varargin)
% llf  Local linear trend (or random-walk-plus-noise) filter with tunes.
%
% Syntax
% =======
%
%     [T,C] = llf(X)
%     [T,C] = llf(X,RANGE,...)
%
% Syntax with output arguments swapped
% =====================================
%
%     [C,T] = llf2(X)
%     [C,T] = llf2(X,RANGE,...)
%
% Input arguments
% ================
%
% * `X` [ tseries ] - Input tseries object that will be filtered.
%
% * `RANGE` [ numeric ] - Date range on which the input data will be
% filtered; `RANGE` can be `Inf`, `[startdata,Inf]`, or `[-Inf,enddate]`;
% if not specifired, `Inf` (i.e. the entire available range of the input
% series) is used.
%
% Output arguments
% =================
%
% * `T` [ tseries ] - Lower-frequency (trend) component.
%
% * `C` [ tseries ] - Higher-frequency (cyclical) component.
%
% Options
% ========
%
% * `'drift='` [ numeric | tseries | *`0`* ] - Deterministic drift in the
% trend.
%
% * `'lambda='` [ numeric | *`10 freq`* ] - Smoothing parameter; needs to
% be specified for tseries objects with indeterminate frequency.
%
% * `'level='` [ tseries ] - Hard tunes on the level of the trend.
%
% * `'change='` [ tseries ] - Hard tunes on the change of the trend.
%
% * `'log='` [ `true` | *`false`* ] - Logarithmise the data before filtering,
% de-logarithmise afterwards.
%
% Description
% ============
%
% Design of the optimisation problem
% ------------------------------------
%
% The `llf` function solves the following constrained optimisation problem:
% $$
% \min_{\bar y_t} \lambda \sum_t
% \left( \Delta \bar y_t - \delta_t \right)^2
% + \sum_{t\in\Omega} \left( \bar y_t - y_t \right)^2,
% $$
% where `\Delta` is the first-difference operator, `\lambda` is the
% smoothing parameter, `\delta_t` is a user-specified drift, `\bar y_t` is
% the computed trend, `y_t` are the actual data, and `\Omega` is the set of
% periods in which the observations are available, subject to
%
% * constraints on the level of the trend:
% $$
% \bar y_\tau = a_\tau, \quad \tau \in \Sigma_1,
% $$
% where `a_\tau` are the user-specified hard tunes, and `\Sigma_1` is the
% set of periods in which the user has imposed constraints on the level of
% the trend;
% * constraints on the change in trend:
% $$
% \Delta \bar y_\tau = b_\tau, \quad \tau \in \Sigma_2.
% $$
% where `b_\tau` are the user-specified hard tunes and `\Sigma_2` is the
% set of periods in which re the user has imposed constraints on the change
% in the trend.
%
% Default smoothing parameters
% ------------------------------
%
% If you do not specify the smoothing parameter using the `'lambda='`
% option, a default value is used. The default value is based on the date
% frequency of the input time series, and is calculated as
% `\lambda = 10 \cdot f`, where `f` is the frequency (yearly=1,
% half-yearly=2, quarterly=4, bi-monthly=6, monthly=12). This gives the
% following default values:
%
% * 10 for yearly time series,
% * 20 for half-yearly time series,
% * 40 for quarterly time series,
% * 60 for bi-monthly time series,
% * 120 for monthly time series.
%
% Note that there is no default value for data with indeterminate or daily
% frequency. For these time series, you must always supply the `'lambda='`
% option.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

% BWF, HPF, LLF

if ~exist('dates','var')
    dates = Inf;
end

if isempty(dates)
    [varargout{1:nargout}] = deal(empty(x));
    return
end

%**************************************************************************

[varargout{1:nargout}] = myfilter('llf',x,dates,[],varargin{:});

end
