function varargout = hpf(x,range,varargin)
% hpf  Hodrick-Prescott filter with tunes.
%
% Syntax
% =======
%
%     [T,C] = hpf(X)
%     [T,C] = hpf(X,RANGE,...)
%
% Syntax with output arguments swapped
% =====================================
%
%     [C,T] = hpf2(X)
%     [C,T] = hpf2(X,RANGE,...)
%
% Input arguments
% ================
%
% * `X` [ tseries ] - Input tseries object that will be filtered.
%
% * `RANGE` [ numeric ] - Date range on which the input data will be
% filtered; `RANGE` can be `Inf`, `[startdata,Inf]`, or `[-Inf,enddate]`;
% if not specifired, `Inf` (i.e. the entire available range of the input
% series) is used.
%
% Output arguments
% =================
%
% * `T` [ tseries ] - Lower-frequency (trend) component.
%
% * `C` [ tseries ] - Higher-frequency (cyclical) component.
%
% Options
% ========
%
% * `'lambda='` [ numeric | *`100 freq^2`* ] - Smoothing parameter; needs
% to be specified for tseries objects with indeterminate frequency.
%
% * `'level='` [ tseries ] - Hard tunes on the level of the trend.
%
% * `'change='` [ tseries ] - Hard tunes on the change of the trend.
%
% * `'log='` [ `true` | *`false`* ] - Logarithmise the data before filtering,
% de-logarithmise afterwards.
%
% Description
% ============
%
% Design of the optimisation problem
% ------------------------------------
%
% The `hpf` function solves the following constrained optimisation problem:
% $$
% \min_{\bar y_t} \lambda \sum_t 
% \left( \Delta \bar y_t - \Delta \bar y_t \right)^2
% + \sum_{t\in\Omega} \left( \bar y_t - y_t \right)^2,
% $$
% where `\Delta` is the first-difference operator, `\lambda` is the
% smoothing parameter, `\bar y_t` is the computed trend, `y_t` are the
% actual data, and `\Omega` is the set of periods in which the observations
% are available, subject to
%
% * constraints on the level of the trend:
% $$
% \bar y_\tau = a_\tau, \quad \tau \in \Sigma_1,
% $$
% where `a_\tau` are the user-specified hard tunes, and `\Sigma_1` is the
% set of periods in which the user has imposed constraints on the level of
% the trend;
% * constraints on the change in trend:
% $$
% \Delta \bar y_\tau = b_\tau, \quad \tau \in \Sigma_2.
% $$
% where `b_\tau` are the user-specified hard tunes and `\Sigma_2`
% is the set of periods in which the user has imposed constraints on the
% change in the trend.
%
% Default smoothing parameters
% ------------------------------
%
% If the user does not specify the smoothing parameter using the `'lambda='`
% option, a default value is used. The default value is based on the date
% frequency of the input time series, and is calculated as
% $\lambda = 100 \cdot f^2$, where $f$ is the frequency (yearly=1,
% half-yearly=2, quarterly=4, bi-monthly=6, monthly=12). This gives the
% following default values:
%
% * 100 for yearly time series,
% * 400 for half-yearly time series,
% * 1600 for quarterly time series,
% * 3600 for bi-monthly time series,
% * 14400 for monthly time series.
%
% Note that there is no default value for data with indeterminate or daily
% frequency. For these time series, you must supply the `'lambda='`
% option.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

% BWF, HPF, LLF

if ~exist('range','var')
    range = Inf;
end

if isempty(range)
    [varargout{1:nargout}] = deal(empty(x));
    return
end

%**************************************************************************

[varargout{1:nargout}] = myfilter('hpf',x,range,[],varargin{:});

end