function varargout = get(this,varargin)
% get  Query tseries object property.
%
% Syntax
% =======
%
%     value = get(x,query)
%     [value,value,...] = get(x,query,query,...)
%
% Input arguments
% ================
%
% * `x` [ model ] - Tseries object.
%
% * `query` [ char ] - Name of the queried property.
%
% Output arguments
% =================
%
% * `value` [ ... ] - Value of the queried property.
%
% Valid queries on tseries objects
% =================================
%
% * `'end='` Returns [ numeric ] the date of the last observation.
%
% * `'freq='` Returns [ numeric ] the frequency (periodicity) of the time
% series.
%
% * `'nanEnd='` Returns [ numeric ] the last date at which observations are
% available in all columns; for scalar tseries, this query always returns
% the same as `'end'`.
%
% * `'nanRange='` Returns [ numeric ] the date range from `'nanstart'` to
% `'nanend'`; for scalar time series, this query always returns the same as
% `'range'`.
%
% * `'nanStart='` Returns [ numeric ] the first date at which observations are
% available in all columns; for scalar tseries, this query always returns
% the same as `'start'`.
%
% * `'range='` Returns [ numeric ] the date range from the first observation to the
% last observation.
%
% * `'start='` Returns [ numeric ] the date of the first observation.
%
% Description
% ============
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

P = inputParser();
P.addRequired('x',@istseries);
P.addRequired('name',@iscellstr);
P.parse(this,varargin);

%**************************************************************************

varargout = cell(size(varargin));
varargin = strtrim(varargin);
n = length(varargin);
flag = true([1,n]);
for iarg = 1 : n
   [varargout{iarg},flag(iarg)] = get_(lower(varargin{iarg}));
end

% Report invalid queries.
if any(~flag)
   utils.error('model', ...
      'This is not a valid tseries object query: ''%s''.', ...
      varargin{~flag});
end

   %***********************************************************************
   % Nested function get_().
   function [x,flag] = get_(query)
      
      x = [];
      flag = true;
      
      % Check for function calls inside the GET query.
      tokens = regexp(query,'^([A-Za-z]\w*)\((.*?)\)$','tokens','once');
      transform = [];
      if ~isempty(tokens)
         query = tokens{2};
         transform = str2func(tokens{1});
      end
      
      switch query
         case {'range','first2last','start2end','first:last','start:end'}
            x = range(this);
         case {'min','minrange','nanrange'}
            sample = all(~isnan(this.data(:,:)),2);
            x = range(this);
            x = x(sample);
         case {'start','startdate','first'}
            x = this.start;
         case {'nanstart','nanstartdate','nanfirst','allstart','allstartdate'}
            sample = all(~isnan(this.data(:,:)),2);
            if isempty(sample)
               x = NaN;
            else
               x = this.start + find(sample,1,'first') - 1;
            end
         case {'end','enddate','last'}
            x = this.start + size(this.data,1) - 1;
         case {'nanend','nanenddate','nanlast','allend','allenddate'}
            sample = all(~isnan(this.data(:,:)),2);
            if isempty(sample)
               x = NaN;
            else
               x = this.start + find(sample,1,'last') - 1;
            end
         case {'freq','frequency','per','periodicity'}
            x = datfreq(this.start);
         case {'data','value','values'}
            % Not documented. Use x.data directly.
            x = this.data;
         case {'comment','comments'}
            % Not documented. User x.Comment directly.
            x = comment(this);
         otherwise
            flag = false;
      end
      
      if flag && ~isempty(transform)
         x = transform(x);
      end
      
   end
   % End of nested function get_().

end

