function [h1,h2,range,data] = errorbar(varargin)
% errorbar  Line plot with error bars.
%
% Syntax
% =======
%
%     [LL,EE,RANGE] = errorbar(X,B,...)
%     [LL,EE,RANGE] = errorbar(RANGE,X,B,...)
%     [LL,EE,RANGE] = errorbar(AA,RANGE,X,B,...)
%     [LL,EE,RANGE] = errorbar(X,LO,HI,...)
%     [LL,EE,RANGE] = errorbar(RANGE,X,LO,HI,...)
%     [LL,EE,RANGE] = errorbar(AA,RANGE,X,LO,HI,...)
%
% Input arguments
% ================
%
% * `AA` [ numeric ] - Handle to axes in which the graph will be plotted; if
% not specified, the current axes will used.
%
% * `RANGE` [ numeric ] - Date range; if not specified the entire range of
% the input tseries object will be plotted.
%
% * `X` [ tseries ] - Tseries object whose data will be plotted as a line
% graph.
%
% * `B` [ tseries ] - Width of the bands that will be plotted around the
% lines.
%
% * `LO` [ tseries ] - Width of the band below the line.
%
% * `HI` [ tseries ] - Width of the band above the line.
%
% Output arguments
% =================
%
% * `LL` [ numeric ] - Handles to lines plotted.
%
% * `EE` [ numeric ] - Handles to error bars plotted.
%
% * `RANGE` [ numeric ] - Actually plotted date range.
%
% Options
% ========
%
% * `'dateformat='` [ char | *irisget('plotdateformat')* ] - Date format for
% the tick marks on the x-axis.
%
% * `'datetick='` [ numeric | *`Inf`* ] - Vector of dates locating tick marks
% on the x-axis; Inf means they will be created automatically.
%
% * `'relative='` [ *`true`* | `false` ] - If true, the data for the lower and
% upper bounds are relative to the centre, i.e. the bounds will be added to
% the centre (in this case, `LOW` must be negative numbers and `HIGH` must
% be positive numbers). If false, the bounds are absolute data (in this
% case `LOW` must be lower than `X`, and `HIGH` must be higher than `X`).
%
% See help on built-in `plot` function for other options available.

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

if nargin == 0
    h1 = [];
    h2 = [];
    return
end

if all(ishghandle(varargin{1}))
    ax = varargin{1}(1);
    varargin(1) = [];
else
    ax = gca();
end

if isnumeric(varargin{1})
    range = varargin{1};
    varargin(1) = [];
else
    range = Inf;
end

x = varargin{1};
varargin(1) = [];

if isempty(varargin) || ~isa(varargin{1},'tseries')
    low = x;
    low.data = low.data(:,2:2:end);
    low.Comment = low.comment(:,2:2:end);
    high = low;
    x.data = x.data(:,1:2:end);
    x.Comment = x.comment(:,1:2:end);
else
    low = varargin{1};
    varargin(1) = [];
    if ~isempty(varargin) && isa(varargin{1},'tseries')
        high = varargin{1};
        varargin(1) = [];
    else
        high = low;
    end
end

[opt,varargin] = passvalopt('tseries.errorbar',varargin{:});

%**************************************************************************

[h1,range,data,time] = ...
    tseries.gplot(@plot,ax,range,x,varargin{:});

status = get(gca(),'nextPlot');
set(gca(),'nextPlot','add');
low = mygetdata(low,range);
high = mygetdata(high,range);
h2 = tseries.myjusterrorbars(time,data,low,high,opt.relative);
set(gca(),'nextPlot',status);

% link = cell(size(h1));
for i = 1 : numel(h1)
    set(h2(i),'color',get(h1(i),'color'));
    % link{i} = linkprop([h1(i),h2(i)],'color');
end
% setappdata(ax,'link',link);
grfun.excludefromlegend(h2);

end