function daily(THIS)
% DAILY   Calendar view of a daily tseries object.
%
% Syntax
% =======
%
%     daily(X)
%
% Input arguments
% ================
%
% * `X` [ tseries ] - Tseries object with indeterminate frequency whose
% date ticks will be interpreted as Matlab serial date numbers.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

   
if datfreq(THIS.start) ~= 0
   error('iris:tseries', ...
   'Function DAILY can be used only on series with indeterminate frequency.');
end

%**************************************************************************

% Display header.
strfun.loosespace();
mydispheader(THIS);

% Re-arrange data into a 2D matrix.
THIS.data = THIS.data(:,:);

% Display data, one month per row
[x,rowstart] = xxcalendar(THIS);
output = ' ';
blanks(1:size(THIS.data,2)-1) = {' '};
for i = 1 : length(rowstart)
   output = char(output,datestr(rowstart(i),'    mmm-YYYY:'),blanks{:});
end
output = strjust(output,'right');
divider = ' ';
divider = divider(ones([size(output,1),1]));
output = [output,divider(:,[1,1])];
for i = 1 : 31
   tmp = strjust(char(sprintf('[%g]',i),num2str(x(:,i))),'right');
   output = [output,tmp]; %#ok<AGROW>
   if i < 31
      output = [output,divider(:,[1,1,1,1])]; %#ok<AGROW>
   end
end
disp(output);

% Display comment.
disp(THIS.Comment);

% [startyear,startmonth,startday] = datevec(this.start);

end

% Subfunctions.

%**************************************************************************
function [x,rowstart] = xxcalendar(this)

   if isnan(this.start) || isempty(this.data)
      x = [];
      rowstart = NaN;
      return
   end
   
   [nper,ncol] = size(this.data);
   [startyear,startmonth,startday] = datevec(this.start);
   [endyear,endmonth,endday] = datevec(this.start + nper - 1);
   data = this.data;
   
   % Pad missing observations at the beginning of the first month
   % and at the end of the last month with NaNs.
   tmp = eomday(endyear,endmonth);
   data = [nan([startday-1,ncol]);data;nan([tmp-endday,ncol])];
   
   % Start-date and end-date of the calendar matrixt.
   % startdate = datenum(startyear,startmonth,1);
   % enddate = datenum(endyear,endmonth,tmp);
   
   year = startyear : endyear;
   nyear = length(year);
   year = year(ones([1,12]),:);
   year = year(:);
   
   month = 1 : 12;
   month = transpose(month(ones([1,nyear]),:));
   month = month(:);
   
   year(1:startmonth-1) = [];
   month(1:startmonth-1) = [];
   year(end-(12-endmonth)+1:end) = [];
   month(end-(12-endmonth)+1:end) = [];
   nper = length(month);
   
   lastday = eomday(year,month);
   x = [];
   for t = 1 : nper
      tmp = nan(ncol,31);
      tmp(:,1:lastday(t)) = transpose(data(1:lastday(t),:));
      x = [x;tmp]; %#ok<AGROW>
      data(1:lastday(t),:) = [];
   end
   
   rowstart = datenum(year,month,1);
   
end
% xxcalendar().