function varargout = bwf(x,order,cutoff,dates,varargin)
% bwf  Butterworth filter with tunes.
%
% Syntax
% =======
%
%     [T,C] = bwf(X,ORDER,CUTOFF)
%     [T,C] = bwf(X,ORDER,CUTOFF,RANGE,...)
%
% Syntax with output arguments swapped
% =====================================
%
%     [C,T] = bwf2(X,ORDER,CUTOFF)
%     [C,T] = bwf2(X,ORDER,CUTOFF,RANGE,...)
%
% Input arguments
% ================
%
% * `X` [ tseries ] - Input tseries object that will be filtered.
%
% * `ORDER` [ numeric ] - Order of the Butterworth filter; note that
% `ORDER` = 2 reproduces the Hodrick-Prescott filter.
%
% * `CUTOFF` [ numeric ] - Cut-off periodicity.
%
% * `RANGE` [ numeric ] - Date range on which the input data will be
% filtered; `RANGE` can be `Inf`, `[startdata,Inf]`, or `[-Inf,enddate]`;
% if not specifired, `Inf` (i.e. the entire available range of the input
% series) is used.
%
% Output arguments
% =================
%
% * `T` [ tseries ] - Lower-frequency (trend) component.
%
% * `C` [ tseries ] - Higher-frequency (cyclical) component.
%
% Options
% ========
%
% * `'level='` [ tseries ] - Hard tune on the level of the trend.
%
% * `'change='` [ tseries ] - Hard tune on the change of the trend.
%
% * `'log='` [ `true` | *`false`* ] - Logarithmise the data before
% filtering, de-logarithmise afterwards.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

% BWF, HPF, LLF

if ~exist('dates','var')
    dates = Inf;
end

if isempty(dates)
    [varargout{1:nargout}] = deal(empty(x));
    return
end

% (lambda)/(lambda + (1/(1-L)^n)*(1/(1-(1/L))^n)).

%**************************************************************************

order = round(order);
q = exp(-1i*2*pi/cutoff);
lambda = real((1-q).*(1-1./q))^(-order);

[varargout{1:nargout}] = ...
    myfilter('bwf',x,dates,order,varargin{:},'lambda',lambda);

end