function varargout = arwm(this,ndraw,varargin)
% arwm  Adaptive random-walk Metropolis posterior simulator.
%
% Syntax
% =======
%
%     [THETA,LOGPOST,AR,SCALE,FINALCOV] = arwm(POS,NDRAW,...)
%
% Input arguments
% ================
%
% * `POS` [ poster ] - Initialised posterior simulator object.
%
% * `NDRAW` [ numeric ] - Length of the chain not including burn-in.
%
% Output arguments
% =================
%
% * `THETA` [ numeric ] - MCMC chain with individual parameters in rows.
%
% * `LOGPOST` [ numeric ] - Vector of log posterior density (up to a
% constant) in each draw.
%
% * `AR` [ numeric ] - Vector of cumulative acceptance ratios in each draw.
%
% * `SCALE` [ numeric ] - Vector of proposal scale factors in each draw.
%
% * `FINALCOV` [ numeric ] - Final proposal covariance matrix; the final
% covariance matrix of the random walk step is SCALE(end)^2*FINALCOV.
%
% Options
% ========
%
% * `'adaptProposalCov='` [ numeric | *`0`* ] - Speed of adaptation of the
% Cholesky factor of the proposal covariance matrix towards the target
% acceptanace ratio, `targetAR`; zero means no adaptation.
%
% * `'adaptScale='` [ numeric | *`1`* ] - Speed of adaptation of the scale
% factor to deviations of acceptance ratios from the target ratio,
% `targetAR`.
%
% * `'burnin='` [ numeric | *`0.10`* ] - Number of burn-in draws entered
% either as a percentage of total draws (between 0 and 1) or directly as a
% number (integer greater that one). Burn-in draws will be added to the
% requested number of draws `ndraw` and discarded after the posterior
% simulation.
%
% * `'estTime='` [ `true` | *`false`* ] - Display and update the estimated time
% to go in the command window.
%
% * `'gamma='` [ numeric | *`0.8`* ] - The rate of decay at which the scale
% and/or the proposal covariance will be adapted with each new draw.
%
% * `'initScale='` [ numeric | `1/3` ] - Initial scale factor by which the
% initial proposal covariance will be multiplied; the initial value will be
% adapted to achieve the target acceptance ratio.
%
% * `'progress='` [ `true` | *`false`* ] - Display progress bar in the command
% window.
%
% *`'targetAR='` [ numeric | *`0.234`* ] - Target acceptance ratio.
%
% Description
% ============
%
% Use the [`poster/stats`](poster/stats) function to process the raw chain
% produced by `arwm`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes & Troy Matheson.

% Validate required inputs.
P = inputParser();
P.addRequired('POS',@(x) isa(x,'poster'));
P.addRequired('NDRAW',@isnumericscalar);
P.parse(this,ndraw);

% Parse options.
options = passvalopt('poster.arwm',varargin{:});

%**********************************************************************

[varargout{1:nargout}] = simulate(this,'arwm',ndraw,options);

end