function [PHI,LIST] = vma(THIS,NPER,varargin)
% vma  Vector moving average representation of the model.
%
% Syntax
% =======
%
%     [PHI,LIST] = vma(M,P,...)
%
% Input arguments
% ================
%
% * `M` [ model ] - Model object for which the VMA representation will be
% computed.
%
% * `P` [ numeric ] - Order up to which the VMA will be evaluated.
%
% Output arguments
% =================
%
% * `PHI` [ namedmat | numeric ] - VMA matrices.
%
% * `LIST` [ cell ] - List of measurement and transition variables in
% the rows of the `Phi` matrix, and list of shocks in the columns of the
% `Phi` matrix.
%
% Option
% =======
%
% * `'output='` [ *'namedmat'* | numeric ] - Output matrix `PHI` will be
% either a namedmat object or a plain numeric array; if the option
% `'select='` is used, `'output='` is always `'namedmat'`.
%
% * `'select='` [ cellstr | *`Inf`* ] - Return the VMA matrices for
% selected variabes and/or shocks only; `Inf` means all variables.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

opt = passvalopt('model.vma',varargin{:});

if ischar(opt.select)
    opt.select = regexp(opt.select,'\w+','match');
end

isselect = iscellstr(opt.select);
isnamedmat = strcmpi(opt.output,'namedmat') || isselect;

%**************************************************************************

ny = length(THIS.solutionid{1});
nx = length(THIS.solutionid{2});
ne = length(THIS.solutionid{3});
nalt = size(THIS.Assign,3);

PHI = zeros(ny+nx,ne,NPER+1,nalt);
[flag,index] = isnan(THIS,'solution');
for ialt = find(~index)
   [T,R,K,Z,H,D,U,Omega] = mysspace(THIS,ialt,false);
   PHI(:,:,:,ialt) = timedom.srf(T,R,K,Z,H,D,U,Omega,NPER,1);
end

% Remove pre-sample period.
PHI(:,:,1,:) = [];

% Report solutions not available.
if flag
    utils.warning('model', ...
        '#Solution_not_available', ...
        sprintf(' #%g',find(index)));
end

% List of variables in rows (measurement and transion) and columns (shocks)
% of output matrices.
LIST = { ...
    [THIS.solutionvector{1:2}], ...
    THIS.solutionvector{3}, ...
    };

% Convert output matrix to namedmat object.
if isnamedmat
    PHI = namedmat(PHI,LIST{1},LIST{2});
end

% Select variables.
if isselect
    PHI = select(PHI,opt.select);
end

end