function [A,B,C,D,F,G,H,J,list,nf,deriv] = system(m,alt,varargin)
% system  System matrices before model is solved.
%
% Syntax
% =======
%
%     [A,B,C,D,F,G,H,J,list,nf] = system(M)
%
% Input arguments
% ================
%
% * `M` [ model ] - Model object whose system matrices will be returned.
%
% Output arguments
% =================
%
% * `A` [ numeric ] - Matrix at the vector of expectations in the
% transition equation.
%
% * `B` [ numeric ] - Matrix at current vector in the transition equations.
%
% * `C` [ numeric ] - Constant vector in the transition equations.
%
% * `D` [ numeric ] - Matrix at transition shocks in the transition
% equations.
%
% * `F` [ numeric ] - Matrix at measurement variables in the measurement
% equations.
%
% * `G` [ numeric ] - Matrix at predetermined transition variables in the
% measurement variables.
%
% * `H` [ numeric ] - Constant vector in the measurement equations.
%
% * `J` [ numeric ] - Matrix at measurement shocks in the measurement
% equations.
%
% * `list` [ cell ] - Lists of measurement variables, transition variables
% includint their auxiliary lags and leads, and shocks as they appear in
% the rows and columns of the system matrices.
%
% * `nf` [ numeric ] - Number of non-predetermined (forward-looking)
% transition variables.
%
% Description
% ============
%
% The system before the model is solved has the following form:
%
%     A E[xf;xb] + B [xf(-1);xb(-1)] + C + D e = 0
%
%     F y + G xb + H + J e = 0
%
% where `E` is a conditional expectations operator, `xf` is a vector of
% non-predetermined (forward-looking) transition variables, `xb` is a
% vector of predetermined (backward-looking) transition variables, `y` is a
% vector of measurement variables, and `e` is a vector of transition and
% measurement shocks.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

opt = passvalopt('model.system',varargin{:});

%**************************************************************************

nalt = size(m.Assign,3);
if nargin < 2
    alt = 1 : nalt;
elseif islogical(alt)
    alt = find(alt);
end

for ialt = transpose(alt(:))
    eqselect = affectedeqtn(m,ialt);
    eqselect(m.eqtntype >= 3) = false;
    [m,deriv] = myderiv(m,eqselect,ialt,opt.symbolic);
    [m,sys] = mysystem(m,deriv,eqselect,ialt);
    F(:,:,ialt) = full(sys.A{1});
    G(:,:,ialt) = full(sys.B{1});
    H(:,1,ialt) = full(sys.K{1});
    J(:,:,ialt) = full(sys.E{1});
    A(:,:,ialt) = full(sys.A{2});
    B(:,:,ialt) = full(sys.B{2});
    C(:,1,ialt) = full(sys.K{2});
    D(:,:,ialt) = full(sys.E{2});
end

list = { ...
    m.solutionvector{1}, ...
    myvector(m,m.systemid{2} + 1i), ...
    m.solutionvector{3}, ...
    };
nf = sum(imag(m.systemid{2}) >= 0);

end