function [S,FF,AA] = shockplot(THIS,SHOCKNAME,RANGE,LIST,varargin)
% shockplot  Short-cut for running and plotting plain shock simulation.
%
% Syntax
% =======
%
%     [S,FF,AA] = shockplot(M,SHOCKNAME,RANGE,LIST,...)
%
% Input arguments
% ================
%
% * `M` [ model ] - Model object that will be simulated.
%
% * `SHOCKNAME` [ char ] - Name of the shock that will be simulated.
%
% * `RANGE` [ numeric ] - Date range on which the shock will be simulated;
% the graphs will also include one pre-shock period.
%
% * `LIST` [ cellstr ] - List of variables that will be reported; you can
% use the syntax of [`dbase/dbplot`](dbase/dbplot).
%
% Output arguments
% =================
%
% * `S` [ struct ] - Simulated database.
%
% * `FF` [ numeric ] - Handles to figure windows created.
%
% * `AA` [ numeric ] - Handles to axes objects created.
%
% Options affecting the simulation
% =================================
%
% * `'deviation='` [ *`true`* | `false` ] - See the option `'deviation='` in
% [`model/simulate`](model/simulate).
%
% * `'dtrends='` [ *`true`* | `false` ] - See the option `'dtrends='` option in
% [`model/simulate`](model/simulate).
%
% * `'shockSize='` [ *`'std'`* | numeric ] - Size of the shock that will
% be simulated; `'std'` means that one std dev of the shock will be
% simulated.
%
% Options affecting the graphs
% =============================
%
% See help on [`dbase/dbplot`](dbase/dbplot) for other options available.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

ip = inputParser();
ip.addRequired('M',@(x) isa(x,'model'));
ip.addRequired('SHOCKNAME',@ischar);
ip.addRequired('RANGE',@isnumeric);
ip.addRequired('LIST',@(x) ischar(x) || iscellstr(LIST));

[opt,varargin] = passvalopt('model.shockplot',varargin{:});

if ischar(LIST)
    LIST = {LIST};
end

%**************************************************************************

elist = THIS.name(THIS.nametype == 3);
eindex = strcmp(elist,SHOCKNAME);
if ~any(eindex)
    utils.error('model', ...
        'This is not a valid name of a shock: ''%s''.', ...
        SHOCKNAME);
end

if strcmpi(opt.shocksize,'std')    
    opt.shocksize = permute(THIS.stdcorr(1,eindex,:),[1,3,2]);
end

if opt.deviation
    d = zerodb(THIS,RANGE);
else
    d = sstatedb(THIS,RANGE);
end

d.(SHOCKNAME)(1,:) = opt.shocksize;
S = simulate(THIS,d,RANGE, ...
    'deviation=',opt.deviation,'dtrends=',opt.dtrends);
S = dbextend(d,S);

[FF,AA] = dbplot(S,RANGE(1)-1:RANGE(end),LIST,varargin{:});

end