classdef SVAR < VAR
    % SVAR  Structural vector autoregressions: SVAR objects and functions.
    %
    % SVAR methods:
    %
    % Constructor
    % ============
    %
    % * [`SVAR`](SVAR/SVAR) - Convert reduced-form VAR to structural VAR.
    %
    % SVAR objects can call any of the [VAR](VAR/Contents) functions. In
    % addition, the following functions are available for SVAR objects.
    %
    % Getting information about SVAR objects
    % =======================================
    %
    % * [`fprintf`](SVAR/fprintf) - Format SVAR as a model code and write to text file.
    % * [`get`](SVAR/get) - Query SVAR object properties.
    % * [`sprintf`](SVAR/sprintf) - Format SVAR as a model code and write to text string.
    %
    % Simulation
    % ===========
    %
    % * [`srf`](SVAR/srf) - Shock (impulse) response function.
    %
    % Stochastic properties
    % ======================
    %
    % * [`fevd`](SVAR/fevd) - Forecast error variance decomposition for SVAR variables.
    %
    % Manipulating SVAR objects
    % ==========================
    %
    % * [`sort`](SVAR/sort) - Sort SVAR parameterisations by squared distance of shock reponses to median.
    %
    % See help on [VAR](VAR/Contents) objects for other functions available.
    %
    % Getting on-line help on SVAR functions
    % =======================================
    %
    %     help SVAR
    %     help SVAR/function_name
    %
    % Getting on-line help on SVAR functions that are inherited from VARs
    % ====================================================================
    %
    %     help VAR
    %     help VAR/function_name
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        B = []; % Coefficient matrix at structural residuals.
        std = []; % Std dev of structural residuals.
    end
    
    methods
        function this = SVAR(varargin)
            % SVAR  Convert reduced-form VAR to structural VAR.
            %
            % Syntax
            % =======
            %
            %     [S,DATA,B,COUNT] = SVAR(V,DATA,...)
            %
            % Input arguments
            % ================
            %
            % * `V` [ VAR ] - Reduced-form VAR object.
            %
            % * `DATA` [ struct | tseries ] - Data associated with the input VAR
            % object.
            %
            % Output arguments
            % =================
            %
            % * `S` [ VAR ] - Structural VAR object.
            %
            % * `DATA` [ struct | tseries ] - Data with transformed structural
            % residuals.
            %
            % * `B` [ numeric ] - Impact matrix of structural residuals.
            %
            % * `COUNT` [ numeric ] - Number of draws actually performed (both
            % successful and unsuccessful) when `'method'='draw'`; otherwise `COUNT=1`.
            %
            % Options
            % ========
            %
            % * `'maxIter='` [ numeric | *`0`* ] - Maximum number of attempts when
            % `'method'='draw'`.
            %
            % * `'method='` [ *`'chol'`* | `'draw'` | `'qr'` | `'svd'` ] - Method that
            % will be used to identify structural residuals.
            %
            % * `'nDraw='` [ numeric | *`0`* ] - Target number of successful draws when
            % `'method'='draw'`.
            %
            % * `'output='` [ *`'auto'`* | `'dbase'` | `'tseries'` ] - Format of output
            % data.
            %
            % * `'progress='` [ `true` | *`false`* ] - Display progress bar in the command
            % window.
            %
            % * `'rank='` [ numeric | *`Inf`* ] - Reduced rank of the covariance matrix of
            % structural residuals when `'method=' 'svd'`; `Inf` means full rank is
            % preserved.
            %
            % * `'std='` [ numeric | *`1`* ] - Std deviation of structural residuals.
            %
            % * `'test='` [ char ] - String that will be evaluated for each random draw of the
            % impact matrix `B`. The evaluation must result in true or false; only
            % those matrices `B` that evaluate to true will be kept. See Description
            % for more on how to write `test`.
            %
            % * `'ordering='` [ numeric | *empty* ] - Re-order VAR variables before
            % identifying structural residuals.
            %
            % * `'reorderResiduals='` [ *`true`* | `false` ] - Re-order identified
            % structural residuals back.
            %
            % Description
            % ============
            %
            % --Identification random Householder transformations
            %
            % The structural impact matrices `B` are randomly generated using a
            % Householder transformation algorithm. Each matrix is tested by evaluating
            % the `test` string supplied by the user. If it evaluates to true the
            % matrix is kept and one more SVAR parameterisation is created, if it is
            % false the matrix is discarded.
            %
            % The `test` string can refer to the following characteristics:
            %
            % * `S` -- the impulse (or shock) response function; the `S(i,j,k)` element
            % is the response of the `i`-th variable to the `j`-th shock in
            % period `k`.
            %
            % * `Y` -- the asymptotic cumulative response function; the `Y(i,j)`
            % element is the asumptotic (long-run) cumulative response of the `i`-th
            % variable to the `j`-th shock.
            %
            % Example
            % ========
            
            % -IRIS Toolbox.
            % -Copyright (c) 2007-2012 Jaromir Benes.
            this = this@VAR();
            if nargin == 0
                return
            elseif nargin == 1 && isa(varargin{1},'SVAR')
                this = varargin{1};
                return
            end
            % The case with a VAR object as the first input argument is
            % handled as a VAR method because otherwise we couldn't return
            % SVAR data as a second output argument (not allowed in
            % constructors).
        end
    end
    
    methods
        varargout = companion(varargin)
        varargout = fevd(varargin)
        varargout = fprintf(varargin)
        varargout = get(varargin)
        varargout = sort(varargin)
        varargout = sprintf(varargin)
        varargout = srf(varargin)
    end
    
    methods (Hidden)
        varargout = myidentify(varargin)
        varargout = mysystem(varargin)
        varargout = specget(varargin)
        
        function varargout = irf(varargin)
            % irf  Alias for SRF.
            %
            % See help on [`SVAR/srf`](SVAR/srf).
            
            % -IRIS Toolbox.
            % -Copyright (c) 2007-2012 Jaromir Benes.
            [varargout{1:nargout}] = srf(varargin{:});
        end
        
    end
    
    methods (Access=protected,Hidden)
        varargout = mysubsalt(varargin)
        varargout = mywoonvav(varargin)
    end
    
    
end