classdef FAVAR < userdataobj
% FAVAR  Factor-augmented vector autoregressions: FAVAR objects and functions.
%
% Constructor
% ============
%
% * [`FAVAR`](FAVAR/FAVAR) - Create new FAVAR object.
%
% Getting information about FAVAR objects
% ========================================
%
% * [`comment`](FAVAR/comment) - Get or set user comments in an IRIS object.
% * [`get`](FAVAR/get) - Query model object properties.
% * [`userdata`](FAVAR/userdata) - Get or set user data in an IRIS object.
% * [`VAR`](FAVAR/VAR) - Return a VAR object describing the factor dynamics.
%
% Estimation
% ===========
%
% * [`estimate`](FAVAR/estimate) - Estimate FAVAR using static principal components.
%
% Filtering and forecasting
% ==========================
%
% * [`filter`](FAVAR/filter) - Re-estimate the factors by Kalman filtering the data taking FAVAR coefficients as given.
% * [`forecast`](FAVAR/forecast) - Forecast FAVAR factors and observables.
%
% Getting on-line help on FAVAR functions
% ========================================
%
%     help FAVAR
%     help FAVAR/function_name
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

   properties
      range = []; % Estimation range specified by the user.
      fitted = []; % True for the periods within the user range that have been fitted.
      Mean = []; % Vector of estimated means used to standardise the input data.
      Std = []; % Vector of estimated stdevs used to standardise the input data.
      ynames = {}; % Names of observables.
      singval = []; % Singular values of the principal components.
      A = []; % Transition matrices A(L) for the factor VAR.
      B = []; % Coefficient matrix at orthonormalised shocks in factor VAR.
      C = []; % Measurement matrix.

      T = []; % Schur decomposition of the transition matrix.
      U = []; % Schur transformation of the factors.
      eigval = []; % Eigenvalues of the factor VAR.
      Omega = []; % Cov of reduced-form residuals in factor VAR.
      Sigma = []; % Cov of idiosyncratic residuals. 
      cross = NaN;
   end
   
   methods
      function this = FAVAR(varargin)
% FAVAR  Create new FAVAR object.
%
% Syntax
% =======
%
%     f = FAVAR()
%
% Output arguments
% =================
%
% * `f` [ FAVAR ] - New empty VAR object.
%
% Description
% ============
% 
% This function creates a new empty FAVAR object. It is usually followed by
% the [estimate](FAVAR/estimate) function to estimate the FAVAR parameters
% on data.
%
% Example
% ========
%
% To estimate a FAVAR, you first need to create an empty VAR object, and
% then run the [FAVAR](FAVAR/estimate) function on it, e.g.
%
%     f = FAVAR();
%     f = estimate(f,d,list,range);
%
% or simply
%
%     f = estimate(VAR(),d,list,range);

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

         this = this@userdataobj();
         if ~isempty(varargin) && isfavar(varargin{1})
            this = varargin{1};
         end
      end
   end
   
   methods
      varargout = eig(vararing)
      varargout = estimate(varargin)
      varargout = filter(varargin)
      varargout = forecast(varargin)
      varargout = get(varargin)
      varargout = mean(varargin)
      varargout = VAR(varargin)
   end
   
   methods (Hidden)
      varargout = standardise(varargin)
      disp(varargin)
   end
   
   methods (Static,Hidden)
      varargout = pc(varargin)
      varargout = estimatevar(varargin)
      varargout = error(varargin)
      varargout = cc(varargin);
      varargout = destandardise(varargin)
   end
   
end