classdef subheading < report.genericobj
    % subheading  Enter subheading in table.
    %
    % Syntax
    % =======
    %
    %     P.subheading(CAP,...)
    %
    % Input arguments
    % ================
    %
    % * `P` [ struct ] - Report object created by the
    % [`report.new`](report/new) function.
    %
    % * `CAP` [ char ] - Text displayed as a subheading on a separate line in
    % the table.
    %
    % Options
    % ========
    %
    % * `'justify='` [ `'c'` | *`'l'`* | `'r'` ] - (Inheritable from parent objects)
    % Horizontal alignment of the subheading (centre, left, right).
    %    
    % * `'separator='` [ char | *empty* ] - (Not inheritable from parent
    % objects) LaTeX commands that will be inserted immediately after the end
    % of the table row, i.e. appended to \\, within a tabular mode.
    %
    % * `'stretch='` [ *`true`* | `false` ] - (Inheritable from parent objects)
    % Stretch the subheading text also across the data part of the table; if
    % not the text will be contained within the initial descriptive columns.
    %
    % * `'typeface='` [ char | *`'\itshape\bfseries'`* ] - (Not inheritable from
    % parent objects) LaTeX code specifying the typeface for the subheading; it
    % must use the declarative forms (such as `\itshape`) and not the command
    % forms (such as `\textit{...}`).
    %
    % Generic options
    % ================
    %
    % See help on [generic options](report/Contents) in report objects.
    %
    % Description
    % ============
    %
    % Example
    % ========
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    methods
        
        function this = subheading(varargin)
            this = this@report.genericobj(varargin{:});
            this.childof = {'table'};
            this.default = [this.default,{ ...
                'justify','l', ...
                @(x) ischar(x) && any(strncmpi(x,{'l','c','r'},1)),true, ...
                'stretch',true,@islogicalscalar,true, ...
                'separator','',@ischar,false, ...
                'typeface','\itshape\bfseries',@ischar,false, ...
                }];
        end
        
        function [THIS,varargin] = specargin(THIS,varargin)
        end
        
        function THIS = setoptions(THIS,varargin)
            THIS = setoptions@report.genericobj(THIS,varargin{:});
            THIS.options.justify = lower(THIS.options.justify(1));
        end
        
        function [C,TEMPFILES] = speclatexcode(THIS)
            TEMPFILES = {};
            % TODO: Check if `'justify='` `'centre'` works.
            par = THIS.parent;
            range = par.options.range;
            totalncol = par.nlead + length(par.options.range);
            vline = par.options.vline;
            vline(vline < range(1)-1 | vline > range(end)) = [];
            vline = min(vline);
            isvline = ~isempty(vline);
            if THIS.options.stretch
                if strncmpi(THIS.options.justify,'l',1) && isvline
                    ncol = par.nlead + round(vline - range(1));
                else
                    ncol = totalncol;
                end
            else
                ncol = par.nlead;
            end
            C = ['\multicolumn{$ncol$}', ...
                '{$just$}{$typeface$ $title$$footnotemark$}', ...
                ' $empty$ \\'];
            C = strrep(C,'$just$',THIS.options.justify);
            C = strrep(C,'$ncol$',sprintf('%g',ncol));
            C = strrep(C,'$empty$',repmat('& ',1,totalncol-ncol));
            C = strrep(C,'$typeface$',THIS.options.typeface);
            C = strrep(C,'$title$',latex.stringsubs(THIS.title));
            C = strrep(C,'$footnotemark$',footnotemark(THIS));
        end
        
    end
    
end