classdef array < report.tabularobj
    % array  Insert an array with user data in report.
    %
    % Syntax
    % =======
    %
    %     P.array(CAP,DATA)
    %
    % Input arguments
    % ================
    %
    % * `P` [ struct ] - Report object created by the
    % [`report.new`](report/new) function.
    %
    % * `CAP` [ char | cellstr ] - Caption (title, subtitle) displayed at
    % the top of the matrix.
    %
    % * `DATA` [ cell ] - Cell array with input data; numeric and text entries
    % are allowed.
    %
    % Options
    % ========
    %
    % * `'arrayStretch='` [ numeric | *`1.15`* ] - (Inheritable from parent
    % objects) Stretch between lines in the array (in pts).
    %
    % * `'captionTypeface='` [ cellstr | char | *`'\large\bfseries'`* ] - \LaTeX\
    % format commands for typesetting the array caption; the subcaption format
    % can be entered as the second cell in a cell array.
    %
    % * `'colWidth' [ numeric | `NaN` ] - (Inheritable from parent objects) Width
    % of the array columns; if NaN the column width will be automatically
    % adjusted.
    %
    % * `'format='` [ char | *`'%.2f'`* ] - (Inheritable from parent objects)
    % Numeric format string; see help on the built-in `sprintf` function.
    %
    % * `'footnote='` [ char | *empty* ] - Footnote at the array title; only
    % shows if the title is non-empty.
    %
    % * `'heading='` [ char | *empty* ] - (Inheritable from parent objects)
    % User-supplied heading, i.e. an extra row or rows at the top of the array.
    %
    % * `'inf='` [ char | *`'$\infty$'`* ] - (Inheritable from parent objects)
    % \LaTeX\ string that will be used to typeset `Inf`s.
    %
    % * `'long='` [ `true` | *`false`* ] - (Inheritable from parent objects) If
    % `true`, the array may stretch over more than one page.
    %
    % * `'longFoot='` [ char | *empty* ] - (Inheritable from parent objects)
    % Footnote that appears at the bottom of the array (if it is longer than
    % one page) on each page except the last one; works only with `'long='`
    % `true`.
    %
    % * `'longFootPosition='` [ `'centre'` | *`'left'`* | `'right'` ] -
    % (Inheritable from parent objects) Horizontal alignment of the footnote in
    % long arrays; works only with `'long='` `true`.
    %
    % * `'nan='` [ char | *`'$\cdots$'`* ] - (Inheritable from parent objects)
    % \LaTeX\ string that will be used to typeset `NaN`s.
    %
    % * `'pureZero='` [ char | *empty* ] - (Inheritable from parent objects)
    % \LaTeX\ string that will be used to typeset pure zero entries; if empty the
    % zeros will be printed using the current numeric format.
    %
    % * `'printedZero='` [ char | *empty* ] - (Inheritable from parent objects)
    % \LaTeX\ string that will be used to typeset the entries that would appear
    % as zero under the current numeric format used; if empty these numbers
    % will be printed using the current numeric format.
    %
    % * `'separator='` [ char | *`'\medskip\par'`* ] - (Inheritable from parent
    % objects) \LaTeX\ commands that will be inserted after the array.
    %
    % * `'sideways='` [ `true` | *`false`* ] - (Inheritable from parent objects)
    % Print the array rotated by 90 degrees.
    %
    % * `'tabcolsep='` [ `NaN` | numeric ] - (Inheritable from parent objects)
    % Space between columns in the array, measured in em units; `NaN` means the
    % \LaTeX\ default.
    %
    % * `'typeface='` [ char | *empty* ] - (Not inheritable from parent objects)
    % \LaTeX\ code specifying the typeface for the array as a whole; it must use
    % the declarative forms (such as `\itshape`) and not the command forms
    % (such as `\textit{...}`).
    %
    % Generic options
    % ================
    %
    % See help on [generic options](report/Contents) in report objects.
    %
    % Description
    % ============
    %
    % Example
    % ========
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        data = {};
    end
    
    methods
        
        function THIS = array(varargin)
            THIS = THIS@report.tabularobj(varargin{:});
            THIS.childof = {'report','align'};
            THIS.default = [THIS.default,{...
                'format','%.2f',@ischar,true,...
                'inf','$\infty$',@ischar,true,...
                'heading','',@ischar,true,...
                'nan','$\cdots$',@ischar,true,...
                'purezero','',@ischar,true, ...
                'printedzero','',@ischar,true, ...
                'separator','\medskip\par',@ischar,true, ...
                'typeface','',@ischar,false, ...                
                }];
            THIS.vline = [];
            THIS.nlead = 0;
        end
        
        function [THIS,varargin] = specargin(THIS,varargin)
            if ~isempty(varargin)
                if iscell(varargin{1})
                    THIS.data = varargin{1};
                    varargin(1) = [];
                else
                    utils.error('report', ...
                        'Input data for array objects must be cell arrays.');
                end
            end
        end
        
        function [c,helperfiles] = speclatexcode(THIS)
            helperfiles = {};
            nl = sprintf('\n');
            nrow = size(THIS.data,1);
            ncol = size(THIS.data,2);
            % Start of tabular and tabular spec.
            if isempty(THIS.options.colspec)
                colspec = '';
                for i = 1 : ncol
                    if any(cellfun(@isnumeric,THIS.data(:,i)))
                        colspec(end+1) = 'r'; %#ok<AGROW>
                    else
                        colspec(end+1) = 'l'; %#ok<AGROW>
                    end
                end
            else
                colspec = THIS.options.colspec;
            end
            if colspec(1) == '{'
                colspec(1) = '';
            end
            if colspec(end) == '}'
                colspec(end) = '';
            end
            THIS.options.colspec = colspec;
            THIS.ncol = ncol;
            THIS.nlead = 0;
            c = begin(THIS);
            % `colspec` will be used to determine the position of the content
            % within a makebox.
            colspec(colspec == '|') = '';
            c = [c,nl,'\hline',nl];
            % User-supplied heading.
            if ~isempty(THIS.options.heading)
                c = [c,nl,THIS.options.heading];
            end
            % Cycle over the matrix rows.
            for i = 1 : nrow
                % Cycle over the matrix columns.
                c1 = cell([1,ncol]);
                for j = 1 : ncol
                    if ~isempty(THIS.data{i,j}) && isnumeric(THIS.data{i,j})
                        c1{j} = report.array.sprintf(THIS.data{i,j},THIS.options);
                    elseif ischar(THIS.data{i,j})
                        c1{j} = latex.stringsubs(THIS.data{i,j});
                    else
                        c1{j} = '';
                    end
                    c1{j} = report.matrix.makebox(...
                        c1{j},'',THIS.options.colwidth,colspec(j),'');
                    if j > 1
                        c = [c,' & ']; %#ok<AGROW>
                    end
                    c = [c,c1{j}]; %#ok<AGROW>
                end
                c = [c,' \\']; %#ok<AGROW>
            end
            c = [c,nl,finish(THIS)];
        end
        
    end
    
end