classdef series < report.genericobj & report.condformatobj
    % series  Add new data to graph or table.
    %
    % Syntax
    % =======
    %
    %     P.series(CAP,X,...)
    %
    % Input arguments
    % ================
    %
    % * `P` [ struct ] - Report object created by the
    % [`report.new`](report/new) function.
    %
    % * `CAP` [ char ] - Caption used as a default legend entry in a graph,
    % or in the leading column in a table.
    %
    % * `X` [ tseries ] - Input data that will be added to the current table or
    % graph.
    %
    % Options for both table series and graph series
    % ===============================================
    %
    % * `'marks='` [ cellstr | *empty* ] - (Inheritable from parent objects)
    % Marks that will be added to the legend entries in graphs, or printed in a
    % third column in tables, to distinguish the individual columns of possibly
    % multivariate input tseries objects.
    %
    % * `'showMarks='` [ *`true`* | `false` ] - (Inheritable from parent objects)
    % Use the marks defined in the `'marks='` option to label the individual
    % rows when input data is a multivariate tseries object.
    %
    % Options for table series
    % =========================
    %
    % * `'autoData='` [ function_handle | cell | *empty* ] - Function, or a cell
    % array of functions, that will be used to produce new columns in the input
    % tseries object (i.e. new rows of ouput in the report).
    %
    % * `'condFormat='` [ struct | *empty* ] - (Inheritable from parent
    % objects) Structure with .test and .format fields describing
    % conditional formatting of individual table entries.
    %
    % * `'decimal='` [ numeric | *`NaN`* ] - (Inheritable from parent objects)
    % Number of decimals that will be displayed; if NaN the `'format='` option
    % is used instead.
    %
    % * `'format='` [ char | *`'%.2f'`* ] - (Inheritable from parent objects)
    % Numeric format string; see help on the built-in `sprintf` function.
    %
    % * `'footnote='` [ char | *empty* ] - Footnote at the series text.
    %
    % * `'highlight='` [ numeric | *empty* ] - (Inheritable from parent objects)
    % Periods that will get highlighted in tables; to highlight date ranges in
    % graphs, use the `'highlight='` option in the parent graph object.
    %
    % * `'inf='` [ char | *`'\ensuremath{\infty}'`* ] - (Inheritable from parent objects)
    % LaTeX string that will be used to typeset Inf entries.
    %
    % * `'nan='` [ char | *`'\ensuremath{\cdot}'`* ] - (Inheritable from parent objects)
    % LaTeX string that will be used to typeset NaN entries.
    %
    % * `'pureZero='` [ char | *empty* ] - (Inheritable from parent objects)
    % LaTeX string that will be used to typeset pure zero entries; if empty the
    % zeros will be printed using the current numeric format.
    %
    % * `'printedZero='` [ char | *empty* ] - (Inheritable from parent objects)
    % LaTeX string that will be used to typeset the entries that would appear
    % as zero under the current numeric format used; if empty these numbers
    % will be printed using the current numeric format.
    %
    % * `'separator='` [ char | *empty* ] - (Not inheritable from parent
    % objects) LaTeX commands that will be inserted immediately after the end
    % of the table row, i.e. appended to \\, within a tabular mode.
    %
    % * `'units='` [ char ] - (Inheritable from parent objects) Description of
    % input data units that will be displayed in the second column of tables.
    %
    % Options for graph series
    % =========================
    %
    % * `'legend='` [ char | cellstr | `NaN` | *`Inf`* ] - (Not inheritable from
    % parent objects) Legend entries used instead of the series caption and
    % marks; Inf means the caption and marks will be used to construct legend
    % entries; NaN means the series will be exluded from legend.
    %
    % * `'plotFunc='` [ `@area` | `@bar` | `@barcon` | *`@plot`* | `@plotcmp` |
    % `@plotpred` | `@stem` ] - (Inheritable from parent objects) Plot function
    % that will be used to create graphs.
    %
    % * `'plotOptions='` [ cell | *empty* ] - Options passed as the last input
    % arguments to the plot function.
    %
    % Generic options
    % ================
    %
    % See help on [generic options](report/Contents) in report objects.
    %
    % Description
    % ============
    %
    % Using the `'nan='`, `'inf='`, `'pureZero='` and `'printedZero='` options
    % -------------------------------------------------------------------------
    %
    % When specifying the LaTeX string for these options, bear in mind that the
    % table entries are printed in the math model. This means that whenever you
    % wish to print a normal text, you need to use an appropriate text
    % formatting command allowed within a math mode. Most frequently, it would
    % be `'\textnormal{...}'`.
    %
    % Using the `'plotFunc='` option
    % -------------------------------
    %
    % When you set the option to `'plotpred'`, the input data `X` (second input
    % argument) must be a multicolumn tseries object where the first column is
    % the time series observations, and the second and further columns are its
    % Kalman filter predictions as returned by the `filter` function.
    %
    % Conditional formatting
    % -----------------------
    %
    % The conditional format struct (or an array of structs) specified through
    % the `'condFormat='` option must have two fields, `.test` and `.format`.
    %
    % The `.test` field is a text string with a Matlab expression. The
    % expression must evaluate to a scalar true or false, and can refer to the
    % following attributes associated with each entry in the data part of the
    % table:
    %
    % * `value` - the numerical value of the entry,
    % * `date` - the date under which the entry appears,
    % * `year` - the year under which the entry appears,
    % * `period` - the period within the year (e.g. month or quarter) under
    % which the entry appears,
    % * `freq` - the frequency of the date under which the entry appears,
    % * `text` - the text label on the left,
    % * `mark` - the text mark on the left used to describe the individual rows
    % reported for multivariate series,
    % * `row` - the row number within a multivariate series.
    % * `rowvalues` - a row vector of all values on the current row.
    %
    % If the table is based on user-defined structure of columns (option
    % `'colstruct='` in [`table`](report/table)), the following additional
    % attributes are available
    %
    % * `colname` - descriptor of the column (text in the headline).
    %
    % You can combine a number of attribues within one test, using the logical
    % operators, e.g.
    %
    %     'value > 0 && year > 2010'
    %
    % The `.format` fields of the conditional format structure consist of LaTeX
    % commands that will be use to typeset the corresponding entry. The
    % reference to the entry itself is through a question mark. The entries are
    % typeset in math mode; this for instance meanse that for bold or italic
    % typface, you must use the `\mathbf{...}` and `\mathit{...}` commands.
    %
    % Furthermore, you can combine multiple tests and their correponding
    % formats in one structure; they will be all applied to each entry in the
    % specified order.
    %
    % Example of a conditional format structure
    % ==========================================
    %
    %     cf = struct();
    %     cf(1).test = 'value < 0';
    %     cf(1).format = '\mathit{?}';
    %     cf(2).test = 'date < qq(2010,1)';
    %     cf(2).format = '\color{blue}';
    %
    
    % -IRIS Toolbox.
    % -Copyright (c) 2007-2012 Jaromir Benes.
    
    properties
        data = {tseries()};
    end
    
    methods
        
        function THIS = series(varargin)
            THIS = THIS@report.genericobj(varargin{:});
            THIS = THIS@report.condformatobj();
            THIS.childof = {'graph','table'};
            THIS.default = [THIS.default,{ ...
                'autodata',{},@(x) isempty(x) ...
                || isa(x,'function_handle') ...
                || iscell(x),true, ...
                'colstruct',[],@(x) isempty(x) ...
                || report.genericobj.validatecolstruct(x),true, ...
                'conditional',[], ...
                @(x) isempty(x) || ( ...
                isstruct(x) ...
                && isfield(x,'test') && isfield(x,'format') ...
                && ( all(cellfun(@isfunc,{x.test})) || all(cellfun(@ischar,{x.test})) ) ...
                && ( all(cellfun(@isfunc,{x.test})) || all(cellfun(@ischar,{x.format})) ) ...
                ), ...
                true, ...
                'condformat',[], ...
                @(x) isempty(x) || ( ...
                isstruct(x) ...
                && isfield(x,'test') && isfield(x,'format') ...
                && iscellstr({x.test}) && iscellstr({x.format}) ), ...
                true, ...
                'decimal',NaN,@isnumericscalar,true, ...
                'format','%.2f',@ischar,true, ...
                'highlight',[],@(x) isnumeric(x) || isfunc(x),true, ...
                'legend',Inf,@(x) isempty(x) ...
                || (isnumericscalar(x) && (isnan(x) || isinf(x))) ...
                || iscellstr(x) || ischar(x),false, ...
                'lhs',Inf,@(x) isnumericscalar(x) && ...
                (isequal(x,Inf) || (x == round(x) && x >= 1)), ...
                true, ...
                'marks',{},@(x) isempty(x) || iscellstr(x),true, ...
                'inf','\ensuremath{\infty}',@ischar,true, ...
                'nan','\ensuremath{\cdots}',@ischar,true, ...
                'plotfunc',@plot,@(x) isa(x,'function_handle') ...
                && any(strcmpi(char(x),{'plot','area','bar','conbar','barcon','stem','plotcmp','plotpred'})), ...
                true, ...
                'plotoptions',{},@(x) iscell(x) && iscellstr(x(1:2:end)), ...
                true, ...
                'purezero','',@ischar,true, ...
                'printedzero','',@ischar,true, ...
                'separator','',@ischar,false, ...
                'showmarks',true,@islogical,true, ...
                'units','',@ischar,true, ...
                }];
        end
        
        function [THIS,varargin] = specargin(THIS,varargin)
            if ~isempty(varargin)
                if istseries(varargin{1}) || iscell(varargin{1})
                    THIS.data = varargin{1};
                    if istseries(THIS.data)
                        THIS.data = {THIS.data};
                    end
                end
                varargin(1) = [];
            end
        end
        
        function THIS = setoptions(THIS,varargin)
            THIS = setoptions@report.genericobj(THIS,varargin{:});
            THIS.options.marks = THIS.options.marks(:).';
            if ~isempty(THIS.options.autodata)
                if isa(THIS.options.autodata,'function_handle')
                    THIS.options.autodata = {THIS.options.autodata};
                end
                THIS = autodata(THIS);
            end
            if ~isnan(THIS.options.decimal)
                THIS.options.format = sprintf('%%.%gf', ...
                    round(abs(THIS.options.decimal)));
            end
            THIS = assign(THIS,THIS.options.condformat);
            if ~isempty(THIS.options.conditional)
                utils.warning('report', ...
                    'The ''conditional'' option is obsolete and will be removed ', ...
                    'in future versions of IRIS. Use ''condformat'' instead.');
            end
            if ~THIS.options.showmarks
                THIS.options.marks = {};
            end
            if strcmp(char(THIS.options.plotfunc),'conbar')
                THIS.options.plotfunc = @barcon;
            end
        end
        
        varargout = plot(varargin)
        varargout = speclatexcode(varargin)
        varargout = latexonerow(varargin)
        varargout = latexdata(varargin)
        varargout = autodata(varargin)
        varargout = getdata(varargin)
        
    end
    
    methods (Access=protected,Hidden)
        varargout = mylegend(varargin)
    end

end