function f = uniform(low,upp)
% uniform  Create function proportional to log of uniform distribution.
%
% Syntax
% =======
%
%     F = logdist.uniform(LOW,UPP)
%
% Input arguments
% ================
%
% * `LOW` [ numeric ] - Lower bound of the uniform distribution.
%
% * `UPP` [ numeric ] - Upper bound of the uniform distribution.
%
% Output arguments
% =================
%
% * `F` [ function_handle ] - Handle to a function returning a value that
% is proportional to the log of the uniform density.
%
% Description
% ============
%
% See [help on the logdisk package](logdist/Contents) for details on
% using the function handle `F`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

if low > upp
    [low,upp] = deal(upp,low);
end
mu = 1/2*(low + upp);
sgm = sqrt(1/12*(upp - low)^2);
mode = mu;
f = @(x,varargin) xxuniform(x,low,upp,mu,sgm,mode,varargin{:});

end

% Subfunctions.

%**************************************************************************
function y = xxuniform(x,a,b,mu,sgm,mode,varargin)

y = zeros(size(x));
index = x >= a & x <= b;
y(~index) = -Inf;
if isempty(varargin)
    return
end

switch lower(varargin{1})
    case {'proper','pdf'}
        y(index) = 1/(b - a);
        y(~index) = 0;
    case 'info'
        y = 0;
    case {'a','location'}
        y = a;
    case {'b','scale'}
        y = b;
    case 'mean'
        y = mu;
    case {'sigma','sgm','std'}
        y = sgm;
    case 'mode'
        y = mode;
    case 'name'
        y = 'uniform';
    case 'draw'
        y = a + (b-a)*rand(varargin{2:end});
end

end
% xxuniform().