function f = lognormal(mu,sgm)
% lognormal  Create function proportional to log of log-normal distribution.
%
% Syntax
% =======
%
%     F = logdist.lognormal(MEAN,STD)
%
% Input arguments
% ================
%
% * `MEAN` [ numeric ] - Mean of the log-normal distribution.
%
% * `STD` [ numeric ] - Std dev of the log-normal distribution.
%
% Output arguments
% =================
%
% * `F` [ function_handle ] - Handle to a function returning a value that
% is proportional to the log of the log-normal density.
%
% Description
% ============
%
% See [help on the logdisk package](logdist/Contents) for details on
% using the function handle `F`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

a = log(mu^2/sqrt(sgm^2 + mu^2));
b = sqrt(log(sgm^2/mu^2 + 1));
mode = exp(a - b^2);
f = ...
    @(x,varargin) xxlognormal(x,a,b,mu,sgm,mode,varargin{:});

end

% Subfunctions.

%**************************************************************************
function y = xxlognormal(x,a,b,mu,sgm,mode,varargin)

y = zeros(size(x));
index = x > 0;
x = x(index);
if isempty(varargin)
    logx = log(x);
    y(index) = -0.5 * ((logx - a)./b).^2  - logx;
    y(~index) = -Inf;
    return
end

switch lower(varargin{1})
    case {'proper','pdf'}
        y(index) = ...
            1/(b*sqrt(2*pi)) .* exp(-(log(x)-a).^2/(2*b^2)) ./ x;
    case 'info'
        y(index) = (b^2 + a - log(x) + 1)./(b^2*x.^2);
    case {'a','location'}
        y = a;
    case {'b','scale'}
        y = b;
    case 'mean'
        y = mu;
    case {'sigma','sgm','std'}
        y = sgm;
    case 'mode'
        y = mode;
    case 'name'
        y = 'lognormal';
    case 'draw'
        y = exp(a + b*randn(varargin{2:end}));
end

end
% xxlognormal().
