function f = invgamma(mu,sgm)
% invgamma  Create function proportional to log of inv-gamma distribution.
%
% Syntax
% =======
%
%     F = logdist.invgamma(MEAN,STD)
%
% Input arguments
% ================
%
% * `MEAN` [ numeric ] - Mean of the inv-gamma distribution.
%
% * `STD` [ numeric ] - Std dev of the inv-gamma distribution.
%
% Output arguments
% =================
%
% * `F` [ function_handle ] - Handle to a function returning a value that
% is proportional to the log of the inv-gamma density.
%
% Description
% ============
%
% See [help on the logdisk package](logdist/Contents) for details on
% using the function handle `F`.
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

a = 2 + (mu/sgm)^2;
b = mu*(1 + (mu/sgm)^2);
mode = b/(a + 1);
f = @(x,varargin) xxinvgamma(x,a,b,mu,sgm,mode,varargin{:});

end

% Subfunctions.

%**************************************************************************
function y = xxinvgamma(x,a,b,mu,sgm,mode,varargin)

y = zeros(size(x));
index = x > 0;
x = x(index);
if isempty(varargin)
    y(index) = (-a-1)*log(x) - b./x;
    y(~index) = -Inf;
    return
end

switch lower(varargin{1})
    case {'proper','pdf'}
        y(index) = b^a/gamma(a)*(1./x).^(a+1).*exp(-b./x);
    case 'info'
        y(index) = -(2*b - x*(a + 1))./x.^3;
    case {'a','location'}
        y = a;
    case {'b','scale'}
        y = b;
    case 'mean'
        y = mu;
    case {'sigma','sgm','std'}
        y = sgm;
    case 'mode'
        y = mode;
    case 'name'
        y = 'invgamma';
    case 'draw'
        y = 1./gamrnd(a,1/b,varargin{2:end});
end

end
% xxinvgamma().