function [X,y0,k,y1,g1] = uncmean(ybar,mu,varargin)
% uncmean  Unconditional-mean dummy (or Sims' initial dummy) observations for BVARs.
%
% Syntax
% =======
%
%     [X,Y0,K,Y1,G1] = BVAR.uncmean(YBAR,MU,[NY,P])
%     [X,Y0,K,Y1,G1] = BVAR.uncmean(ybar,mu,[ny,p,ng])
%
% Input arguments
% ================
%
% * `YBAR` [ numeric ] - Vector of unconditional means imposed as priors.
%
% * `MU` [ numeric ] - Weight on the dummy observations.
%
% * `[NY,P,NG]` [ numeric ] - Number of variables, order of the VAR, and
% number of co-integrating vectors in the VAR for which the prior dummies
% will be created.
%
% Output arguments
% =================
%
% * `X` [ numeric ] - Array with prior dummy observations that can be used
% in the `'BVAR='` option of the [`VAR/estimate`](VAR/estimate) function.
%
% * `Y0`, `K`, `Y1`, `G1` [ numeric ] - These output arguments provide more
% details on the structure of the dummy observations, and have no use;
% `X=[Y0;K;Y1;G1]`.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

ng = 0;
if length(varargin) == 1
   % BVAR.uncmean(ybar,mu,[ny,p]);
   % BVAR.uncmean(ybar,mu,[ny,p,ng]);
   ny = varargin{1}(1);
   p = varargin{1}(2);
   if length(varargin{1}) > 2
      ng = varargin{1}(3);
   end
else
   % Bkw compatibility.
   % BVAR.uncmean(ybar,mu,ny,p).
   % BVAR.uncmean(ybar,mu,ny,p,ng).
   ny = varargin{1};
   p = varargin{2};
   if length(varargin) > 2
      ng = varargin{3};
   end
end

P = inputParser();
P.addRequired('ny',@isnumericscalar);
P.addRequired('p',@isnumericscalar);
P.addRequired('ng',@isnumericscalar);
P.addRequired('ybar',@isnumeric);
P.addRequired('mu',@isnumericscalar);
P.parse(ny,p,ng,ybar,mu);

if length(ybar) == 1
   ybar = ybar(ones([ny,1]));
else
   ybar = ybar(:);
end

if ny ~= numel(ybar)
   error('iris:BVAR', ...
      ['Number of variables in the VAR must match ', ...
      'the length of the vector of prior means.']);
end

%**************************************************************************

k = mu;
g1 = zeros([ng,1]);
y0 = ybar*mu;
y1 = repmat(ybar*mu,[p,1]);
X = [y0;k;y1;g1];

end
