function X = covmat(c,repeat,n)
% covmat  Covariance matrix prior dummy observations for BVARs.
%
% Syntax
% =======
%
%     X = BVAR.covmat(C,REP,[NY,P])
%     X = BVAR.covmat(C,REP,[NY,P,NG])
%
% Input arguments
% ================
%
% * `C` [ numeric ] - Prior covariance matrix of residuals; if `C` is a
% vector it will be converted to a diagonal matrix.
%
% * `REP` [ numeric ] - The number of times the dummy observations will
% be repeated.
%
% * `[NY,P,NG]` [ numeric ] - Number of variables, `NY`, order of the VAR,
% `P`, and number of co-integrating vectors in the VAR, `NG`, for which the
% prior dummies will be created.
%
% Output arguments
% =================
%
% * `X` [ numeric ] - Array with prior dummy observations that can be used
% in the `'BVAR='` option of the [`VAR/estimate`](VAR/estimate) function.
%
% Description
% ============
%
% Example
% ========
%

% -IRIS Toolbox.
% -Copyright (c) 2007-2012 Jaromir Benes.

ny = n(1);
p = n(2);
ng = 0;
if length(n) > 2
    ng = n(3);
end

% Parse input arguments.
P = inputParser();
P.addRequired('ny',@isnumericscalar);
P.addRequired('p',@isnumericscalar);
P.addRequired('ng',@isnumericscalar);
P.addRequired('cov',@(x) isnumeric(x) ...
    && ndims(x) == 2 && (all(size(x) == ny) || numel(x) == ny) );
P.addRequired('repeat',@(x) isnumericscalar(x) && x > 0 && x == round(x));
P.parse(ny,p,ng,c,repeat);

%**************************************************************************

if isvector(c)
    c = diag(sqrt(c));
else
    c = chol(c).';
end

nd = ny;
y0 = c;
k = zeros(1,nd);
g1 = zeros(ng,nd);
y1 = zeros(ny*p,nd);
X = [y0;k;y1;g1];
if repeat > 1
    X = repmat(X,1,repeat);
end

end