/*OPTIONS NOCENTER LS=155 PS=60 COMPRESS=YES NOFMTERR;*/
/***********************************************************************/
/*  Program:  INSOLE97_09full3.SAS                                   */
/*     Date:  2010 August 30                                            */
/*  Purpose:  Subset data from the 1997-09 INSOLES                     */
/*    Notes:  This program reads in data merged data from 1997-2009 insoles    */
/*            created in INSOLE97_09full2, and adds tax calculator            */
/*			  variables											   */
/*  PrevPgm:  INSOLE97_09full2.SAS                                  */
/***********************************************************************/

LIBNAME     SUBSUB  "G:\REV_EST\DeBacker\Panel8706\LaborInc";
LIBNAME	    w2		"G:\REV_EST\DeBacker\Panel8706\w2";



/* Merge CWHS returns from insoles, and create variables not in this dataset 
	using the variables above */

DATA SUBSUB.INSOLE97_09full (COMPRESS=YES);
	SET SUBSUB.INSOLE97_09;

	/* Create variables not in this dataset using the variables above */
	
	/* ITEMIZATION STATUS */
	ITEM = (TIDX > 0);

	/* Ages of taxpayers calculated in previous program */

	/************************************ 
	SELF EMPLOYMENT AND BUSINESS INCOME 
	*************************************/

	/* 
	SEERN: Total self-employment income 
	*/
	SEERN	=	SEFARM + SENONF;

	/* 
	SEERNP: Self employment income, primary earner 
	*/
	SEERNP 	= 	SEFARMP + SENONFP;

	/* 
	SEERNP: Self employment income, secondary earner 
	*/
	SEERNS	= 	SEFARMS + SENONFS;

	/*
	APART: Partnership non-passive income minus partnership non-passive loss
	*/
	APART	= 	APARTG - APARTL;

	/*
	PPART: Partnership passive income minus allowable partnership passive loss
	*/
	PPART	= 	PPARTG - PPARTL;

	/*
	PART: Partnership income minus allowable partnership loss.
	*/
	PART	= 	APART + PPART;

	/*
	PSBC: S-Corporation passive income minus allowable S-corporation passive loss.
	*/
	PSBC	= 	PSBCG - PSBCL;


	/*
	ASBC: S-Corporation non-passive income minus S-corporation non-passive loss.
	*/
	ASBC	= 	ASBCG - ASBCL;

	/*
	SBC: S-Corporation income minus S-corporation allowable loss.
	*/
	SBC		= 	ASBC + PSBC;

	/* Number of qulifying children for childcare deduction and eitc */
	CCQI 	=	(CDOB1 ~= 0) + (CDOB2 ~= 0); 
	IEIC_KID= 	(EICYB1 ~= 0) + (EICYB2 ~= 0); 
	CHCRED_KID = N24;

	/* Number of over65 exemptions */
	AGEDE 		= I65INDP + I65INDS;


	/* Filing Status */
	/* Assume single and other filers are filing singly */
	filertype = 's';
		if mars = 2 or mars = 5 then filertype = 'm';
		if mars = 4 then filertype = 'h';
		/* For MFS filers, multiply all variables by 2, then using MFJ tax table */
		if mars = 3 then filertype = 'm';

	if filertype="m" then married=1 ;
		else married=0 ;
	label married = "Marital status (1=married, 0=not)" ;


	/* NOTE - CALCULATOR WILL BE MOST ACCURATE FOR SINGLE, MFJ, AND HOH OBSERVATIONS THAT ARE 
		NOT DEPENDENTS */

	/* Age Variables*/	
	age = agep;
	agesp = ages;
	
	/* 
	VARIABLES THAT ARE NOT DIRECT INPUTS INTO THE TAX CALCULATOR
	Next are a set of variables that are not needed directly for the tax calcultor, but that are useful as inputs for other 
	variables, for performing checks on the internal consistency of the data, or for imputing values of other variables.  
	They are given descriptive names and relabeled. 
	*/

	/* AGI is in EP */
	
	label agix = "Adjusted gross income, VAR118" ;


	/* VAR119	X119	DX     Itemized deds less Std ded 79-86	DEDUCTIONS */

	/* NOT IN EDITED PANEL */
	dx = .;
	label dx = "Itemized deductions in excess of zba, VAR119" ;

	/* VAR173	X173	TID    Totl itemized deductions  E04470	TOTAL ITEMIZED DEDUCTIONS */
	tid = tidx ;
	label tid = "Total itemized deductions, VAR173" ;

	/* itemded_original 	Gross itemized deductions before limitation from file */
	itemded_original = GR_TID ;

	/* VAR264	X264	EXEM   Value of personal exempti E04600	EXEMPTION AMOUNT */
	exem = exemx ;
	label exem = "Value of exemptions, VAR264" ;

	/* VAR332		SDX Standard deduction amount    E04100	*/
	/* Only valid starting in 1989 */
	sdx = bsd ;
	label sdx = "Standard deduction amount, VAR332" ;

	/* VAR333		SDADD Additional standard dedctn E04200	*/
	/* Renamed above */
	label sdadd = "Additional std deductn amt, VAR333" ;

	/* 
	VAR120 X120 TINCX  Taxable income (before ex E04500	TAX TABLE INCOME
	*/
	tincx = E04500 ;
	label tincx = "Taxable income (before exemptions?), VAR120" ;

	/* VAR265 X265 TINC Taxable income E04800 TAXABLE INCOME AMOUNT, COMPUTER */
	/* Only valid starting in 1984 */
	tinc = tincx ;
	label tinc = "Taxable income, computer, VAR120";

	/* VAR121	X121	TAXBS  Tax on taxable income     E05100	TAX GENERATED */
	/*
	E05100 is called "tax on taxable income" in the SOI public-use documentation, and "tentative tax" in the edited panel
	documentation.  In the SOI public use documentation, it is defined as "the amount shown on line 43 less special taxes from 
	Form 4972 (Tax on Lump Sum Distributions).  This means that it is tax after applying the ordinary tax brackets and rates to
	taxable income, after applying special rates for capital gains and dividends, and after applying tax from Form 8814 
	(parent's election to include child's interest and dividends on parents tax return).  During this period there is an SOI
	variable called "tax generated," but it appears to be the ordinary tax rates and brackets applied to taxable income excluding
	capital gains and dividends, so "tax on tax table income" is closer to the "tax generated" variable from earlier years of data.
	*/
	taxgenerated = E05100 ;
	label taxgenerated = "Tax generated, VAR121" ;

	/* 
	Computed Regular Tax - E05200
	This is the amount arrived at by applying the Tax Rate Schedules to Taxable
	Income, without regard to the type of computation used by the taxpayer. 
	*/

	computedregulartax = E05200;

	/* 
	Income tax before credits (itbc)
	 
	Starting in year 2000 (and perhaps before), the SOI variable labeled "income tax before credits" (E05800) 
	already includes the AMT, and so differs fundamentally from income tax before credits from prior years.  
	In 1999, it is clear from the SOI documenation that "tentative income tax" (E05100), which comes directly from line 40 of
	Form 1040, corresponds to the old income tax before credits. For 2000 and later years, the variable "tentative
	income tax before credits" (E05750) corresponds to the old income tax before credits.
	*/
	if year=1999 then itbc=E05100 ;
		else itbc=E05750 ;
	label itbc = "Income tax before credits, VAR122" ;

	/* 
	E05700	FORM 4970 TAX
	Note: E05700 is "taxes from special computations", which may include more than just Form 4790 taxes.
	T05705 is "Form 4970 Tax (PY) (taxpayer)"
	*/

	f4970tax = E05700 ;
	label f4970tax = "Form 4970 tax" ;

	/* Income tax before credits less form 4970 tax */
	itbcx = itbc - f4970tax ;
	label itbcx = "Income tax before credits - F4970 tax" ;

	/* 
	ITAC:  E08795	INCOME TAX AFTER CREDITS
	Note: starting in 2000, the income tax after credits variable (E08795) already includes the AMT.
	Before that, ITAC (E08795) does not include the effects of the AMT.  Rather than trying to construct
	a measure of "ITAC" that is consistent over time, we will just adapt code later in the sequence
	of programs to account for this.
	*/
    itac = E08795 ;
	label itac = "Income tax after credits, VAR133" ;

	/* VAR135	X135	MTX    Minimum tax (79-81)	MINIMUM TAX */
	/* Minimum tax does not exist 1983-present.  VAR135 appears to be valid 1979-82 */
    minimumtax = . ;
	label minimumtax = "Minimum tax, VAR135" ;

	/* VAR136	X136	AMTX   Alternative minimum tax   E09600	ALTERNATIVE MINIMUM TAX */
	amtx = E09600 ;
	label amtx = "Alternative minimum tax, VAR136" ;

	/* VAR219	X219	MAXTAX Maximum tax	MAX TAX, MAXIMUM TAX */
	/* Maximum tax does not exist 1982-present.  VAR219 looks ok in 1979-81 */
    maxtax = 0 ;
	label maxtax = "Maximum tax, VAR219" ;

	/* incavg = indicator for whether it appears that income averaging was used (only relevant 1979-86) */
	/* VAR042 TXST	TXST V42 TAX STATUS CODE.  Valid 1979-83 for RTF, all years for SOI */
	/* VAR179	X179	GALTX  Schedule G, alternative tax	SCHED G ALTERNATIVE TAX (only matters in 1981) */
	/* VAR180	X180	GTAX   Schedule G, tax, income averaging	SCHED G TAX */
	/* VAR180 appears valid in RTF 1981 and 1984-86, and valid in SOI in 1979-86 */
    incavg=0 ;
	label incavg = "Income averaging indicator" ;

	/*
	VAR141	X141	TLIAB  Total tax liability       E10300	TOTAL TAX LIABILITY
	*/
	totaltaxliab = tot_liab;
	label totaltaxliab = "Total tax liability VAR141 (missing RTF 79-80, 82-83)" ;

	/* totaladj = total adjustments
	VAR117	X117	ADJT   Total adjustments E02900	TOTAL STATUTORY ADJUSTMENTS
	*/
	totaladj = ttadj ;
	label totaladj = "Total adjustments" ;

	/*
	VAR134	X134	SECATX Self employment tax       E09400	SELF-EMPLOYMENT TAX
	*/
	selfemptax = secatx ;
	label selfemptax = "Self-employment tax" ;

	/*
	othfedtax
	VAR134	X134	SECATX Self employment tax       E09400	SELF-EMPLOYMENT TAX
	VAR137	X137	RPYITC Tax from prior year inves E09700	TAX FROM PRIOR YEAR INVEST TAX CRED
	VAR138	X138	SSTIP                            E09800	SOCIAL SECURITY TAX ON TIP INCOME
	VAR139	X139	TXIRA                            E09900	TAX ON IRA
	VAR140	X140	AEIC   Advanced earned income ta E10000	ADVANCE EARNED INC CRED. PAYMENTS (only on 1040 1979-82, 1990-95)
	Back out using net of two total liability and tax after credits
	*/
	othfedtax = E09200 - E08795;
	label othfedtax = "Non-income fed. taxes on form 1040" ;

	/* 
	othinc_original = "other income" reported in Treasury file
	VAR108	X108	OTHINC Other income	OTHER NET INCOME OR LOSS
	*/
	label othinc = "Other income from file, VAR108" ;


	/*
	staxrefund = State income tax refund
	VAR096	X96	SITR   State income tax refund   E00700	STATE TAX REFUNDS
	*/
	staxrefund = sitr ;
	label staxrefund = "State income tax refund" ;

	/*
	alimonyrec = Alimony received
	VAR097	X97	ALI    Alimony                   E00800	ALIMONY RECEIVED
	*/
	alimonyrec = alimony ;
	label alimonyrec = "Alimony received" ;

	/* Variables for imputing spousal age, blindness, and exemptions */

	/* Primary taxpayer exemption code - xfpt is in panel*/
	/* Secondary taxpayer exemption code - xfst is in panel */
	/* Age exemptions - agede calculated above*/
	/* Blind exemptions */
	blnde = pbi + sbi ;
	/* Additional standard deduction amount - calculated above*/

	/* Pension variables */
	ftpen = 0 ;
	pengr = grpen ;
	ptpen = ptpen ;
	iradis = grirad ;
	tirad = tirad ;
	label ftpen = "Fully taxable pensions 79-86" ;
	label pengr = "Gross partly taxable pensions" ;
	label ptpen = "Partly taxable pensions, taxable amount" ;
	label iradis = "Gross IRA distributions" ;
	label tirad = "IRA distributions, taxable amount" ;


	/* SEP and Simple contributions */
	sepsimple = SEcont;

	/*
	WGT
	VAR261	X261	WGT2	WEIGHT VARIABLE (available 1979-91)
	*/
	wgt = wgt ;
	label wgt = "Weight variable, 99-05" ;

	/* Income Variables */
	income = totincx;


	/********************************************************************** 
	TAX CALCULATOR VARIABLES (ASIDE FROM THOSE ALREADY GENERATED)
	***********************************************************************/

	/*
	MOVING EXPENSES
	movex		Deductible moving expenses. Calculator treats this as either an itemized deduction or as an adjustment, 
				depending on the federal law for that year.
	*/

	movex = exmove_aj;

	/* 
	STATE

	When state is missing, we will attempt to impute it based on information from the nearest years with valid state data
	and adjustments or deductions for moving expenses.

	state			State of residence, SOI code, after imputations

	statedoubt		Degree of doubt in state imputation.
					Statedoubt = 0 if state is not imputed.  
					Statedoubt = 1 if state is imputed, but we have non-imputed state data from years on either side of the missing year, 
					and state is the same in both those years.  
					Statedoubt = 2 if we have non-imputed state data for years on either side of the missing year, and the state 
					is different in each of those years, but a moving expense deduction helps us pinpoint the date of change.
					Statedoubt = 3 if we have non-imputed state data from a nearby past or future year, but not both.   
					Statedoubt = 4 if we have non-imputed state data for years on either side of the missing year, the state is different 
					in each of those years, and we do not have information from an above-the-line moving expense adjustment to help us 
					pinpoint the date of change.  In those cases, we assume the move occured midway between the two non-imputed years, 
					with ties broken by going with the state from the earlier year.
	
	VAR027	STATE	STATE   State Code	V27 STATE CODE

	ZIPSTATE	State inferred from zip code (VAR029)
	*/

	/* recode all states higher than 51 (includes all territories) to 0 */
	state = istate;
	if state > 51 then state = 0;

	statedoubt = 0;
	zipstate = state;

	label state = "State SOI code after imputations" ;
	label statedoubt = "Degree of doubt in state variable" ; 
	label zipstate = "State from zip code (>=85), or state (<85)" ;

	/* YEAR		Four-digit calendar year  */
	year = yr ;

	/* STATEYEAR		Four-digit calendar year for applicable federal law. */
	stateyear = yr ;
		
	/* FEDYEAR		Four-digit calendar year for applicable state law. */
	fedyear = yr ;

	/*
	AGENUM
	agenum		Number of extra exemptions or additional standard deductions for age>=65.
	*/

	agenum = agede;

	label agenum = "Taxpayer and spouse aged>=65" ;

	/*
	BLIND		Number of blind adults (0, 1, or 2).  
	Note that this is only used to calculate federal exemptions and the federal standard deduction.	
	This is calculated above
	*/
	blind = pbi + sbi;
	label blind = "Blind taxpayer and/or spouse" ;


	/*
	DEPS		Number of dependent exemptions (not including taxpayer or spouse).	

	*/

	deps = depne;
	label deps = "Dependents" ;

	/* 
	kids2		Number of dependents eligible for federal earned income credit.
	kids3		Number of dependents eligible for federal child credit.

	*/

	kids2 = ieic_kid;
	kids3 = CHCRED_KID; /*Only applicable 1998 forward */


	/*
	CHILDCARE	Expenses for child care.  This is used to compute the federal credit for child care expenses,
				as well as relevant state provisions.

	*/

	childcare = ccecyr;

	/*	
	kids1		Number of dependents eligible for federal child care credit or deduction.

	kids1 is only used for determining child credits.  It will be inferred from "childcare"
	and "kids2"	
	*/
	kids1 = ccqi;


	label childcare = "Child care expenses" ;
	label kids1 = "Children for child care credit" ;
	label kids2 = "Children at home / for EITC" ;
	label kids3 = "Children for child credit" ;


	/* LABOR INCOME */

	/* Set wasp to was if taxpayer is filing single or hoh */
	if mars = 1 or mars = 4 then wasp = was ;

	/* If w2 forms are different from was on tax form by more than $5000, set wasp and wass to missing */
	/*
	if abs(wasp + wass - was) > 5000 then do;
		wasp = .;
		wass = .;
	end;
	*/
	wagsal1 = wasp;
	wagsal2 = wass;


	/* Business income */
	/* VAR098	X98	BIL    Sched C net income or los E01000	BUS. OR PROF NET PROFIT OR LOSS */
	/* Note: farm income is being combined into business income */
	businc1 = 0;
	businc2 = 0;
    if bil > 0 and senonfp > 0 and senonfs > 0 then do;
 		senonf = senonfp + senonfs;
		businc1 = bil*(senonfp)/(senonf);
		businc2 = bil*(senonfs)/(senonf);
		end;
    else do;
		businc1 = bil;
		businc2 = 0;
	end;


	/* Farm income */
	/* VAR105	X105	FIL    Farm net income or loss   E02100	NET FARM INCOME OR LOSS */
	/* Note: farm income is being combined into business income */

	/* Allocate Farm income according to split on schedule SE */
	/* If nothing reported on SE, assume primary was recipient */
	farminc1 = 0;
	farminc2 = 0;
	if fil > 0 and sefarmp > 0 and sefarms > 0 then do;
		sefarm = sefarmp + sefarms;
		farminc1 = fil*(sefarmp)/(sefarm);
		farminc2 = fil*(sefarms)/(sefarm);
		end;
	else do;
		farminc1 = fil;
		farminc2 = 0;
	end;

	/* Combine business and farm income into business income */
	businc1 = businc1 + farminc1;
	businc2 = businc2 + farminc2;

	farminc1 = 0 ;
	farminc2 = 0 ;


	/* Auxiliary variables not used in edited panel */
	ded2earn = .;
	EIspouseFromDed = .;
	WSspouseshare = .;
	eikidcare = .;
	spouseWSimpute = 0; 

	label wagsal1 = "Wage and salary income primary" ;
	label wagsal1 = "Wage and salary income spouse" ;
	label businc1 = "Sch. C & Sch. F income, primary" ;
	label businc2 = "Sch. C & Sch. F income, spouse" ;
	label farminc1 = "Zero (farm income combined into businc)" ;
	label farminc2 = "Zero (farm income combined into businc)" ;
	label ded2earn = "Federal two-earner couple deduction" ;
	label EIspouseFromDed = "Spouse earned income from ded2earn" ;
	label WSspouseshare = "Spouse's share of earnings imputed from other years" ;
	label eikidcare = "Spouse earnings from child care credit form" ;
	label spouseWSimpute = "Indicator imputed spousal share of wages and salaries" ;


	/******************************************** 
	INCOME ASIDE FROM LABOR
	*********************************************/
	
	/*
	DIVIDENDS
	div1		Dividend income, primary earner
	div2		Dividend income, spouse

	Note: starting in 2003, "div" only includes "qualified dividends" that are eligible for the
	special lower tax rate.  Non-qualified dividends are combined with interest income in these years.
	We will save the variable "divnonqual" to preserve information on non-qualified dividends.
	*/
	divqual = e00650 ;
	if year >= 2003 then divnonqual = dbe - divqual ;
		else divnonqual=0 ;
	if year >= 2003 then div = divqual ;
		else div = dbe ;
	if married=1 then do ;
		div1=0.5*div ;
		div2=0.5*div ;
	end ;
	else do ;
		div1=div ;
		div2=0 ;
	end ;
	label div1 = "Gross dividend income primary" ;
	label div1 = "Gross dividend income spouse" ;
	label divnonqual = "Non-qualified dividend income (>=2003)" ;

	/*
	INTEREST INCOME
	int1		Interest income, primary earner.	
	int2		Interest income, spouse.

	Note that starting in 2003, "non-qualified dividends" are combined with interest income.	
	*/
	int = intst+divnonqual;
	if married=1 then do ;
		int1 = int*0.5 ;
		int2 = int*0.5 ;
	end ;
	else do ;
		int1 = int ;
		int2 = 0 ;
	end ;
	label int1 = "Interest income primary" ;
	label int2 = "Interest income spouse" ;

	/* excldivint = Excluded dividends and/or interest */
	/* This is needed in order to impute "othinc" (or "othcg" in 1985) via the residual method */
	excldivint = 0 ;					
	label excldivint = "Excluded dividends and/or interest" ;

	/*
	TAX-EXEMPT INTEREST
	teint1		Federally tax-exempt interest income (on state and local bonds), primary earner. 
	teint2		Federally tax-exempt interest income, spouse.
	VAR268	X268	TEXINT Tax exempt interest       E00400	TAX EXEMPT INTEREST
	Tax-exempt interest is only available starting in 1987.  Include it in the data set in order to get EIC and taxable social 
	security benefits right, but don't use to construct overall measure of income.			
	*/
	
	teint1 = texint;
	teint2 = 0;
	label teint1 = "Tax-exempt interest primary" ;
	label teint2 = "Tax-exempt interest spouse" ;

	/*
	PENSIONS

	pension1		Federally taxable pension income, primary earner.	
	pension2		Federally taxable pension income, spouse.	
	*/

	/* Only include part of pension included in AGI */
	pension = ptpen + tirad;

	if married=1 then do ;
		pension1 = 0.5*pension ;
		pension2 = 0.5*pension ;
	end ;
	else do ;
		pension1 = pension ;
		pension2 = 0 ;
	end ;

	/* Auxiliary variables not needed for edited panel */
	ptpenimpute = 0;
	ftpenimpute = 0;

	label pension1 = "Taxable pension & IRA primary" ;
	label pension2 = "Taxable pension & IRA spouse" ;
	label ptpenimpute = "ptpen imputed from nearby years" ;
	label ftpenimpute = "ftpen imputed from nearby years" ;

	/* Amounts excluded from total income */
	ptpexc = max(grpen - ptpen, 0);
	iraexc = max(grirad - tirad,0);
	ssexc = max(ssinc - ssagix,0);

	/*
	SCHEDULE E INCOME

	rentinc1		Net rental income or loss, primary earner (may be negative).	
	rentinc2		Net rental income or loss, spouse.	
	partscorp1		Partnership and S-corporation income, primary earner.	
	partscorp2		Partnership and S-corporation income, spouse.	

	In the 1979-95 data, we defined "rentinc" as combined rent and royalty income, and defined "partscorp" as 
	total Sch. E income less rentinc and any pension income reported on Sch. E.  I've changed the coding below
	to make the variables for 99-05 consistent with earlier practice.

	*/
	/* Partnership and SCorp and Other income*/
	rentinc = E25870 ;
	partscorp = sche - rentinc ;

	if married=1 then do ;
		rentinc1 = 0.5*rentinc ;
		rentinc2 = 0.5*rentinc ;
		partscorp1 = 0.5*partscorp ;
		partscorp2 = 0.5*partscorp ;
	end ;
	else do ;
		rentinc1 = rentinc ;
		rentinc2 = 0 ;
		partscorp1 = partscorp ;
		partscorp2 = 0 ;
	end ;

	label rentinc1 = "Rent & royalty income primary" ;
	label rentinc2 = "Rent & royalty income spouse" ;
	label partscorp1 = "Oth. non-pension Sch. E income primary" ;
	label partscorp2 = "Oth. non-pension Sch. E income spouse" ;

	/*
	UNEMPLOYMENT COMPENSATION
	ui1		Unemployment compensation, primary earner.  Include gross amount; calculator determines the portion that is includable in 
			AGI under federal law.
	ui2		Unemployment compensation, spouse.
	*/
	/* Unemployment and SS benefits */
	ui = ucagix;

	/* Divide unemployment insurance between spouses */
	if married=1 then do ;
		ui1=0.5*ui ;
		ui2=0.5*ui ;
	end ;
	else do ;
		ui1 = ui ;
		ui2 = 0 ;
	end ;

	/* Unemployment insurance in AGI */
	/* This is needed for computing othinc (or othcg in 1985) via the residual method */
	uiinagi = ui ;

	label ui1 = "Gross unemployment insurance primary" ;
	label ui2 = "Gross unemployment insurance spouse" ;
	label uiinagi = "Unemployment insurance in AGI" ;

	/*
	SOCIAL SECURITY BENEFITS
	ssben1		Gross social security benefits, primary earner.	
	ssben2		Gross social security benefits, spouse.	
	ssinagi		Social security benefits in AGI
	VAR262	X262	SSAGIX Social Security benefits  E02500	SOCIAL SECURITY BENEFITS IN AGI
	VAR263	X263	SSINC  Gross Social Security ben E02400	GROSS SOCIAL SECURITY
	*/

	ssben = ssinc;

	/* Divide social security benefits between spouses */
	if married=1 then do ;
		ssben1 = 0.5*ssben ;
		ssben2 = 0.5*ssben ;
	end ;
	else do ;
		ssben1 = ssben ;
		ssben2 = 0 ;
	end ;
	label ssben1 = "Gross social security benefits primary" ;
	label ssben2 = "Gross social security benefits spouse" ;


	/* Social security in AGI*/
	/* This is needed for computing othinc (or othcg in 1985) via the residual method */
	ssinagi = ssagix ;
	label ssinagi = "Social security benefits in AGI" ;

	/************************************** 
	CAPITAL GAINS
	***************************************/

	/* Net capital gain less loss from Schedule D included in AGI (after exclusion) */
	/* NGL Net capital gain or loss     E01000	SOCA NET CAPITAL GAIN OR LOSS */
	cgagi_SchD = cgagix ;

	label cgagi_SchD = "Net Schedule D capital gain or loss included in AGI" ;

	/* Capital gain distributions reported on Form 1040 (after exclusion) */
	/* VAR099 CGD40  Capital gains distributio E01100	CAPITAL GAIN DISTRIBUTIONS */
	cgdist = cgd40 ;
	label cgdist = "Capital gains distributions Form 1040" ;

	/* Supplemental schedule gain/loss */
	/* VAR100 SGAGI  Supplemental gains or los E01200 SAL OF NON-CAPITAL ASSET PROPERTY	 */		
	cgsup = SUPPGL ;
	label cgsup = "Supplemental gain or loss non capital asset" ;

	 /* Long term gains - long term losses */

   	netltcg = nltgl ;
   	label netltcg = "Net long-term capital gain" ;

    /* Short term gains - short term losses */

   	netstcg = nstgl ;
   	label netstcg = "Net short-term capital gain" ;


	/* Net gain */
	netgain = netstcg + netltcg ;
	if (netltcg <= 0 or netgain <= 0) then ltcg_SchD = 0 ;
		else ltcg_SchD = min(netltcg, netgain) ;
	label netgain = "Net capital gain" ;
	label ltcg_SchD = "Capital gains subj. to pref. treatment on Sch. D";

	/* Capital gains subject to preferential treatment */
	/*
	In 1991-93 capital gains distributions not reported on Schedule D are no longer eligible for the low rate.  Taxpayers always have the option
	of reporting those distributions on schedule D, in which case they will	be taxed at the lower rate.  So the code for ltcg should not add back 
	in cgdist in 1991-93.  Starting in 1994, there is no line item on the 1040 form for capital gain distributions not reported on schedule D.  
	Instead, for people who do not file Schedule D, capital gains distributions could be included directly on the same line item as Schedule D 
	gains would have been reported, and they benefit from the lower tax rate.  So starting in 1994, capital gains distributions not reported on 
	Schedule D should be part of ltcg again. 
	*/
	ltcg = ltcg_SchD + cgdist ;

	/* Other capital gains or losses */
	othcg = cgagi_SchD + cgdist + cgsup - ltcg ; 

	/* 
	The following variables are only used with the 79-95 panel.
	*/
	ltcglate = 0;
    oamtadj=0 ;
	label ltcglate = "1981 cap gains after June 9, 1981" ;
	label oamtadj = "Other AMT adjustments or late 1981 gains" ;

	/* Divide capital gains between spouses */
	if married = 1 then do ;
		ltcg1 = 0.5*ltcg ;
		ltcg2 = 0.5*ltcg ;
		othcg1 = 0.5*othcg ;
		othcg2 = 0.5*othcg ;
	end ;
	else do ;
		ltcg1 = ltcg ;
		ltcg2 = 0 ;
		othcg1 = othcg ;
		othcg2 = 0 ;
	end ;

	label ltcg1 = "Capital gains pref. treatment primary" ;
	label ltcg2 = "Capital gains pref. treatment spouse" ;
	label othcg1 = "Capital gains or losses not subj. pref. primary" ;
	label othcg2 = "Capital gains or losses not subj. pref. spouse" ;



	if married=1 then do ;
		othinc1 = 0.5*othinc ;
		othinc2 = othinc-othinc1 ;
	end ;
	else do ;
		othinc1 = othinc ;
		othinc2 = 0 ;
	end ;
	label othinc1 = "Other income, computed, primary" ;
	label othinc2 = "Other income, computed, spouse" ;

	/*
	busex		Unreimbursed employee business expenses.  Calculator treats this as either an itemized deduction or as an adjustment, 
				depending on the federal law for that year. Include full amount; calculator will incorporate 2% of AGI floor for 1987 
				and later.
	*/
	
	busex = ebe;
	label busex = "Employee business expenses" ;


	/* 
	othadj1		Other federal adjustments to income, primary earner.  This should include adjustments to income that are 
				subtracted from gross income to get AGI on the federal return, except for the following: the two-earner couple
				deduction applying in 1982-86; moving expenses; unreimbursed employee business expenses; and one-half of social 
				security taxes on self-employment income.  	

	othadj2		Other federal adjustments to income, spouse.	

	VAR109	X109	EXMOVE Moving expenses, above th E20300	MOVING EXPENSE ADJUSTMENT
	VAR110	X110	EBE    Employee business expense E20550+	EMPLOYEE BUSINESS EXPENSE ADJ.
	VAR111	X111	IRA    IRA deductions            E03150	PAYMENTS TO IRA ADJUSTMENT
	VAR112	X112	KEOGH  Keogh deductions          E03300	PAYMENTS TO KEOGH ADJUSTMENT
	VAR113	X113	FIP                              E03400	FORFEITED INTEREST PENALTY ADJ.
	VAR114	X114	ALIP   Alimony paid              E03500	ALIMONY PAID ADJUSTMENT
	VAR115	X115	DIE    Disability income exclusion	DISABILITY INCOME EXLUSION ADJ.
	VAR116	X116	ADJO   Other adjustments         E03900	OTHER ADJUSTMENTS
	VAR339		SEHEAL Self employd health insur E03270	
	VAR134	X134	SECATX Self employment tax       E09400	SELF-EMPLOYMENT TAX
			Sum of adjustment components	
	VAR117	X117	ADJT   Total adjustments         E02900	TOTAL STATUTORY ADJUSTMENTS

	From VAR117, we need to subtract: the two-earner couple deduction applying in 1982-86 (ded2earn), moving expenses 1979-86 and 1994-present, 
	unreimbursed employee business expenses 1979-86, and one-half of social security taxes on self-employment income 1990-present.  
	*/
	/* Adjustments to income other than moving expenses, unreimbursed business expenses, and seca deduction */
	/* Do not remove REIMBURSED business expenses */
	othadj = ttadj - secaded - exmove_aj;

	/* If no business or farm income, don't subtract off secaded, since this is not imputed
	by tax calculator for those with partnership income */
	if businc1 = 0 and businc2 = 0 and farminc1 = 0 and farminc2 = 0 then othadj = ttadj - exmove_aj;

	if married=1 then do ;
		othadj1 = 0.5*othadj ;
		othadj2 = othadj - othadj1 ;
	end ;
	else do ;
		othadj1 = othadj ;
		othadj2 = 0 ;
	end ;
	label othadj1 = "Other adjustments primary" ;
	label othadj2 = "Other adjustments spouse" ;

	/*	
	income		Broad measure of income.	

	A reasonable approach would be income = AGI + adjustments + excluded capital gains (1979-86) + excluded dividends (1979-86) + excluded taxable
	interest (1981) - soc. sec. in AGI + excluded unemployment benefits (where available)- (1/2) self-employment tax - state tax refunds.  We don't	
	have excluded soc. sec. 1979-84, or for low-income returns in any year.  Ideally, we'd like to add in non-taxable pension, but we're missing 
	that for RTF in some years.  It would also be nice to allow the adjustment for alimony paid, but this is missing for RTF for 1979-80, 1982-83.  
	Also check definitions used by Randolph and by Auten, Sieg, Clotfelter.	
	*/

    	income = agix + totaladj - ssinagi - 0.5*selfemptax - staxrefund ;

		/* Re-define income to exclude partscorp losses.  This will increase the consistency of
	    the income measure over time, because partscorp losses before TRA86 are largely passive losses,
	    often associated with tax shelters, that are disallowed after TRA86.  See, for example,
	    how the mean of Schedule E income evolves over time in the 1979-95 panel, shown in TaxCalcTranslate.xls.
	    */

		income = income - min(partscorp1,0) - min(partscorp2,0) ;

	    label income = "Constant-law gross income" ;

	/* Total income implied by above*/
	totalinc = wagsal1 + wagsal2 + businc1 + businc2 + farminc1 + farminc2 + ltcg1 + ltcg2 
		+ othcg1 + othcg2 + div1 + div2 + int1 + int2 + pension1 + pension2 + rentinc1 + rentinc2
		+ ui1 + ui2 + partscorp1 + partscorp2 + othinc1 + othinc2;

	/*
	charity		Charitable donations.

	VAR167	X167	CCASH  Charitable contributions  E19800	CASH CONTRIBUTIONS UNDER 3000 EACH
	VAR168	X168	CONOTC Noncash charitable contri E20100	NONCASH CONTRIBUTIONS
	VAR169	X169	CONCO  Carryover of charitable c E20200	CONTRIBUTION CARRYOVERS
	VAR170	X170	TTCON,DN1 Tot charitable dedctn  E19700	TOTAL CONTRIBUTIONS
	VAR178	X178	CCASH, over$3,000	CASH CONTRIBUTIONS OVER 3000 EACH
	VAR267	X267	CCNONI Nonitemizer charity deductions	CHARITABLE CONT., NONITEMIZERS
	VAR351		Charity, noncash, RTF	
	VAR352		Charity, carrover,RTF	
	VAR353		Charitable dedn, RTF

	Non-cash contributions are missing for RTF 1979-80.
	Also note that in this data set, in some years, VAR170 was capped at 50% of AGI, while in other years, VAR170 was total charitable 
	contributions (current-year deductions are limited to 50% of AGI throughout, but the excess can be carried over to future years).	
	*/
	/* Non-itemizer deductions (inapplicable 99-05) */
    charnonitem = 0 ;

	/* Total charitable contributions, including carryovers. Limited to 50% of AGI 1987-91. */
	charity = ttcon ;

	/* Non-cash charitable contributions */
	charcash = tcash ;

	/* Non-cash charitable contributions */
	noncash = tncash ;

	/* Charity carryovers from prior years */
	charcarry = E20200;

	label charity = "Total charitable contributions (limited 50% AGI 87-91)" ;
	label noncash = "Non-cash charity, missing RTF 79-80" ;
	label charcash = "Cash charity" ;
	label charnonitem = "Non-itemizer charity, 1984-86" ;
	label charcarry = "Charity carryovers from prior years" ;

	/*
	charcg		Unrealized capital gains on donations of appreciated property.  This is included because the AMT at one time taxed
				these gains (from 1987-90 for tangible property, and from 1987-92 for intangible property), and because it may be 
				useful for calculating marginal tax rates in studies of charitable giving. If the marginal tax rate is calculated 
				with respect to charcg, it is done so holding overall charity constant (so it essentially just returns the AMT 
				marginal rate, or zero if AMT is not applicable).

				The Treasury data has charcg for people subject to the AMT from 1989-1992.  We will need to make some adjustments to
				this before running the tax calculator.

	VAR168	X168	CONOTC Noncash charitable contri E20100	NONCASH CONTRIBUTIONS
	VAR351		Charity, noncash, RTF	
	VAR336		Charity Pref,AMT                 E60820	
	*/			
	charcg = 0 ;
	label charcg = "Capital gains in non-cash charity" ;


	/*
	intpaid		Interest paid, other than investment interest.  Only include interest that is deductible at the federal level  for instance, 
				do not include consumer interest when it is not deductible, and include only the deductible portion in the years when its 
				deductibility is being phased-out.

	invint		Investment interest paid. This is separated from other interest for purposes of calculating federal limitation on itemized
				deductions since 1991.

	*/
	intpaid = tti_ded - iie;
	invint = iie;
	label intpaid = "Mortgage interest paid" ;
	label invint = "Other interest paid" ;

	/*
	stinctax		State income taxes paid. Calculator will ignore this if state is non-zero.  We still need it
					for checking how closely we can match federal tax liability, though.

	proptax			State and local property taxes paid.

	salestax		State and local sales taxes paid.	

	taxnodetail		Total deductible state and local taxes, when detail by tax is unavailable.  Set this variable equal to zero if
					information on the particular taxes listed above is available. This variable is used in situations where the data
					set only provides information on the total state and local taxes paid deduction, with no detail by type of tax.
					In those cases, the deduction for taxes other than income taxes will be set equal to taxnodetail, minus endogenously
					computed state income taxes where applicable. This value for other taxes is then held constant in the second iteration used to calculate marginal rates, so that correct marginal rates are still obtained.	
	
	JMB note May 21, 2008.  Starting in 2004, taxpayers could take a deduction for the larger of state and local sales taxes and state 
	and local income taxes.  SOI public-use documentation for 2004 suggests that E18425 is state and local income taxes, and E18450 is
	state and local sales taxes.
	*/
	if year<2004 then stinctax = slitx;
		else stinctax = E18425 ;
	if year<2004 then salestax = 0 ;
		else salestax = E18450 ;
	proptax = ttsltx - stinctax - salestax ; 
	taxnodetail = 0;

	label taxnodetail = "Total state & local tax, if no detail" ;
	label stinctax = "State & local income tax" ;
	label proptax = "State & local property tax" ;
	label salestax = "State & local sales tax" ;


	/*
	medexp		Medical and dental expenses.  Include the full amount of such expenses. The calculator endogenously applies the percentage
				of AGI floor in the federal personal income tax. States that allow medical expense deductions are assumed to follow federal
				provisions, which is almost always the case now but was not always so several decades ago.
	

	*/

	medexp = ttmed;
	meddeduct = dedmed ;

	label medexp = "Gross medical expenditures" ;
	label meddeduct = "Deductible medical expenditures" ;


	/*
	casualty		Casualty and theft losses.  Include the full amount of losses here. The calculator applies any percentage of AGI floors 
					endogenously.

	JMB note May 21, 2008: casualty is supposed to be the full amount of casualty loss, not just the amount above the 10% of AGI floor.
	I have corrected that here.
	*/

	if ttcasu > 0 then casualty = ttcasu + 100 + 0.1*agix ;
		else casualty = 0 ;

	casdeduct = ttcasu;

	label casualty = "Gross casualty expenditures" ;
	label casdeduct = "Deductible casualty expenditures" ;
	


	/*	
	miscdedlim		Miscellaneous itemized deductions subject to 2% of AGI floor.  Include full amount (other than unreimbursed employee
					business expenses). Calculator endogenously applies the 2% of AGI floor when applicable.
	
	VAR287	X287	TTMISL Gross limited miscellaneo E20400	GROSS LIMITED MISC. DEDUCTIONS

	miscdedlim = and VAR287 in 1987 and later years.
	*/
	miscdedlim = t2misc - ebe;
	miscdedlimnet = max(0, miscdedlim - 0.02*agix) ;
	label miscdedlim = "Gross limited miscellaneous deductions" ;
	label miscdedlimnet = "Miscellaneous deductions after limitation" ;


	/*
	omiscded		Other miscellaneous itemized deductions.	

	JMB note May 21, 2008: redefined omiscded_res so we can check for internal consistency.
	*/
	omiscded = fdmisc;

	omiscded_res = GR_TID -
		(charity + intpaid + invint + taxnodetail + stinctax + salestax + proptax + meddeduct + casdeduct + miscdedlimnet) ;

	label omiscded = "Other miscellaneous deductions" ;
	label omiscded_res = "Other miscellaneous deductions, residual method" ;

	/*			
	fdcred		Federal credits other than EITC, child credit, child care credit, and elderly credit. EITC, child credit, child care 
				credit and credit for elderly (i.e., retirement income credit) are computed endogenously by the tax calculator; 
				other federal credits are included in this fdcred variable and are taken as given.  Note that the calculator does not
				accurately capture the interaction between the foreign tax credit and the alternative minimum tax. This is unavoidable,
				because a correct specification of the interaction would require more information than is generally available even in
				income tax return data. For some applications, removing the value of the foreign tax credit from fdcred may be appropriate.
				If the goal of calculating tax liability is to compute the effect of income taxes on the individuals disposable income, 
				then ignoring the foreign tax credit yields results that are closer to the conceptually preferred combined domestic and 
				foreign tax liability anyway.

	*/

	/*
	VAR123	X123	PCCX   Political contribution credit	POLITICAL CONTRIB TAX CREDITS
	*/
	polcred = 0 ;
	label polcred = "Political contributions credit, VAR123" ;

	/*
	VAR124	X124	ELDCR  Elderly credits           E07200	ELDERLY TAX CREDITS
	*/
	eldcr_original = eldcrx ;
	label eldcr_original = "Elderly credit from file VAR124" ;

	/*
	VAR125	X125	CCCX   Child care credits        E07180	CHILD CARE TAX CREDITS
	*/
	kidcarecred_original = cccx ;
	label kidcarecred_original = "Child care credit from file VAR125" ;

	/*
	VAR126	X126	ITC    Investment tax credit     E53220	INVESTMENT TAX CREDITS
	*/
	invcred = E53220 ;
	label invcred = "Investment tax credit, VAR126" ;

	/*
	VAR127	X127	FTC    Foreign tax credit        E07300	FOREIGN TAX CREDITS
	*/
	foreigncred = E07300 ;
	label foreigncred = "Foreign tax credit, VAR127" ;

	/*
	VAR128	X128	WIN    Win credits               E53240	WIN TAX CREDITS
	*/
	wincred = 0 ;
	label wincred = "WIN credit VAR128" ;

	/*
	VAR129	X129	JOBS   Jobs credits	JOB TAX CREDITS
	*/
	jobscred = E53240 ;
	label jobscred = "Jobs credit VAR129" ;

	/*
	VAR130	X130	ECX    Residential energy credits	RESIDENTIAL ENERGY TAX CREDITS
	*/
	energycred = 0 ;
	label energycred = "Energy credit VAR130" ;

	/*
	VAR131	X131	OTXCR  Other tax credits         E08000	OTHER TAX CREDITS
	*/
	othcred = E08000 ;
	label othcred = "Other tax credits VAR131" ;

	/* 
	VAR132	X132	TTXCR  Total tax credits         E07100	TOTAL TAX CREDITS
	*/
	totalcred = ttcrx ;
	label totalcred = "Total tax credits, VAR132" ;

	eic_original = eicx ;
	label eic_original = "EIC (non-refundable portion?) from file VAR144" ;

	/*
	VAR269	X269	GBC    General Business Credit   E07400	GENERAL BUSINESS CRED.
	*/
	genbuscred = E07400 ;
	label genbuscred = "General businsess credit VAR269" ;


	fdcred = ttcrx - cccx - eldcrx - foreigncred; 
	label fdcred = "Other federal credits" ;



	/*
	NOTE: SEE AMTPREFS.DOC FOR MORE INFORMATION ON WHAT BELONGS IN OTAXPREF AND OAMTADJ

	otaxpref		Other tax preferences in base of federal minimum tax or AMT. Include here the value of any tax preference items that
					are added back into the base of minimum tax (1970-82) or the alternative minimum tax (1983-present), aside from
					capital gains exclusions, dividends exclusions, itemized deductions, standard deductions, and personal exemptions,
					which are computed endogenously based on other information already provided above.

	JMB edit May 22, 2008: We are unable to determine exactly what "ttpref" means from the documentation.  However, we do have every line
	item from Form 6251 in the 99-05 panel.  So I have modified the code for constructing "otaxpref" to build it up from the individual
	line items, so we can be sure we are getting it right.
	*/

	amtadjpref = ttpref ;
	label amtadjpref = "AMT adjustments and preferences? 89-95 VAR335" ;

	/* Legacy code */
	/*
	amtnol = E60100;
	amtnol_alt = 62000;
	if year <= 2001 then otaxpref = 
		 (amtnol - amtnol_alt) + max(0, amtadjpref - (tid>0)*(proptax + stinctax + min(meddeduct,0.025*agix) + miscdedlimnet) 
									- (tid=0)*sdx + staxrefund) ;

	else if year >= 2002 then otaxpref = 
		 (amtnol - amtnol_alt) + max(0, amtadjpref - (tid>0)*(proptax + stinctax + min(meddeduct,0.025*agix) + miscdedlimnet) 
									- (tid=0)*sdx + (gr_tid - tidx) + staxrefund)  ;	
	*/

	if year<=1999 then otaxpref=
		E60290
		+ E60300
		+ E60340
		+ E60500
		+ E60550
		+ E60680
		+ E60720
		+ E60840
		+ E61450
		+ E60100
		- E62000 ;

	else if year=2000 then otaxpref=
		E60290
		+ E60300
		+ E60340
		+ E60500
		+ E60550
		+ E60680
		+ E60720
		+ E60840
		+ E61450
		+ E60100
		- E62000 ;

	else if year=2001 then otaxpref=
		E60290
		+ E60300
		+ E60340
		+ E60500
		+ E60550
		+ E60680
		+ E60720
		+ E60840
		+ E61450
		+ E60100
		- E62000 ;

	else if year=2002 then otaxpref=
		E60290
		+ E60300
		+ E60860
		+ E60100
		+ E60840
		+ E60630
		+ E60550
		+ E60720
		+ E60430
		+ E60500
		+ E60340
		+ E60680
		+ E60600
		+ E60405
		+ E60440
		+ E60420
		+ E60410
		- E60480
		+ E61400
		+ E60660
		- E62000 ;

	else if year>=2003 then otaxpref=
		E60290
		+ E60300
		+ E60860
		+ E60100
		+ E60840
		+ E60630
		+ E60550
		+ E60720
		+ E60430
		+ E60500
		+ E60340
		+ E60680
		+ E60600
		+ E60405
		+ E60440
		+ E60420
		+ E60410
		- E60480
		+ E61400
		+ E60660
		- E62000 ;

	/*
	oamtadj		Ideally, we would use this variable to distinguish AMT "adjustments" from AMT "preferences," but this is not
				absolutely necessary.
	*/	
	oamtadj = 0;

	/*
	avglaginc		This is only relevant when income averaging was in effect (before 1987).
	*/
	avglaginc = 0 ;
	label avglaginc = "Avg. of lagged taxable inc. for Sch. G" ;
	


	/*
	psincome		Personal service income, used to compute the federal maximum tax on personal service income 
					(1971-1981, also known as the maximum tax on earned income).
	
	VAR216	X216	PSI    Max tax, personal service income	MAX TAX, PERSONAL SERVICE INCOME
	*/
	psincome = 0 ;
	label psincome = "Personal service income 1979-81" ;

	/*
	psded		Deductions allowable against personal service income, used to compute the federal maximum tax on personal service income (1971-1981, also known as the maximum tax on earned income).	
	VAR217	X217	DAPSI  Max tax personal service dedctns	MAX TAX, PERSONAL SERVICE DEDUCTIONS
	*/
	psded = 0 ;
	label psded = "Personal service deductions 1979-81" ;

	/* 
	rentpay		Annual rent payment (minus any government rent subsidies).	Missing, set to zero.  This is only relevant for computing
				circuit breaker property tax credits, which mainly affect low-income households.  We will supress calculation of these
				credits.

	homeval		Value of home.	Missing, set to zero.  This is only relevant for computing circuit breaker property tax credits, 
				which mainly affect low-income households.  We will supress calculation of these credits.

	*/
	rentpay = 0 ;
	homeval = 0 ;
	label rentpay = "Rent for circuit-breaker, suppressed" ;
	label homeval = "Home value for circuit-breaker, suppressed" ;


	/* create uncomp variable consistent with other years (as possible) */
	uncomp = ucagix ;

	/* create keogh name as in past (now keogh and sepsimple */
	keogh = sepsimple ;

	/* Multiply all variables by 2 for married filing singly returns, since their
	bracket ranges are twice those of MFJ */
	if mars = 3 then do;
		wagsal1 = wagsal1 * 2;
		wagsal2 = wagsal2 * 2;
		businc1 = businc1 * 2;
		businc2 = businc2 * 2;
		farminc1 = farminc1 * 2;
		farminc2 = farminc2 * 2;
		ltcg1 = ltcg1 * 2;
		ltcg2 = ltcg2 * 2;
		othcg1 = othcg1 * 2;
		othcg2 = othcg2 * 2;
		div1 = div1 * 2;
		div2 = div2 * 2;
		int1 = int1 * 2;
		int2 = int2 * 2;
		teint1 = teint1 * 2;
		teint2 = teint2 * 2;
		pension1 = pension1 * 2;
		pension2 = pension2 * 2;
		rentinc1 = rentinc1 * 2;
		rentinc2 = rentinc2 * 2;
		ui1 = ui1 * 2;
		ui2 = ui2 * 2;
		ssben1 = ssben1 * 2;
		ssben2 = ssben2 * 2;
		partscorp1 = partscorp1 * 2;
		partscorp2 = partscorp2 * 2;
		othinc1 = othinc1 * 2;
		othinc2 = othinc2 * 2;
		othadj1 = othadj1 * 2;
		othadj2 = othadj2 * 2;
		charity = charity * 2;
		charcg = charcg * 2;
		intpaid = intpaid * 2;
		invint = invint * 2;
		stinctax = stinctax * 2;
		proptax = proptax * 2;
		salestax = salestax * 2;
		taxnodetail = taxnodetail * 2;
		medexp = medexp * 2;
		casualty = casualty * 2;
		movex = movex * 2;
		busex = busex * 2;
		miscdedlim = miscdedlim * 2;
		omiscded = omiscded * 2;
		childcare = childcare * 2;
		fdcred = fdcred * 2;
		otaxpref = otaxpref * 2;
		oamtadj = oamtadj * 2;
		avglaginc = avglaginc * 2; 
		rentpay = rentpay * 2;
		homeval = homeval * 2;
	end;


	/* Create actual values for tax calculator variables that equal the values in their 
	nominal amounts in the year reported */
	wagsal1act = wagsal1;
	wagsal2act = wagsal2;
	businc1act = businc1;
	businc2act = businc2;
	farminc1act = farminc1;
	farminc2act = farminc2;
	ltcg1act = ltcg1;
	ltcg2act = ltcg2;
	othcg1act = othcg1;
	othcg2act = othcg2;
	div1act = div1;
	div2act = div2;
	int1act = int1;
	int2act = int2;
	teint1act = teint1;
	teint2act = teint2;
	pension1act = pension1;
	pension2act = pension2;
	rentinc1act = rentinc1;
	rentinc2act = rentinc2;
	ui1act = ui1;
	ui2act = ui2;
	ssben1act = ssben1;
	ssben2act = ssben2;
	partscorp1act = partscorp1;
	partscorp2act = partscorp2;
	othinc1act = othinc1;
	othinc2act = othinc2;
	othadj1act = othadj1;
	othadj2act = othadj2;
	charityact = charity;
	charcgact = charcg;
	intpaidact = intpaid;
	invintact = invint;
	stinctaxact = stinctax;
	proptaxact = proptax;
	salestaxact = salestax;
	taxnodetailact = taxnodetail;
	medexpact = medexp;
	casualtyact = casualty;
	movexact = movex;
	busexact = busex;
	miscdedlimact = miscdedlim;
	omiscdedact = omiscded;
	childcareact = childcare;
	fdcredact = fdcred;
	otaxprefact = otaxpref;
	oamtadjact = oamtadj;
	avglagincact = avglaginc; 
	rentpayact = rentpay;
	homevalact = homeval;


	keep id issns issnp fileyr year fedyear stateyear state filertype WGT recidyr 

		/* Demographic characteristics */
		mars married deps depne cahe cafhe kids1 kids2 kids3 agenum age agesp 
		pgender sgender agep ages dobyr sdobyr

		/* Tax parameters */
		tincx income
		exemx bsd sdadd sdx tidx
		totalinc iexemh blnde
		  
		/* Income variables */ 
		AGIX     

		WAS     wass 	wasp
		seern 	seernp 	seerns
		INTST   TEXINT  DBE DIV       
		BIL 	FIL   
		SITR    ALIMONY    texint   grpen
		CGAGIX       CGD40   CGDIST NETLTCG NETSTCG     SUPPGL   
 		TIRAD         
        PTPEN         
        SCHE  ESTU   uncomp adjtotinc ptpen grpen      
        UCAGIX        
        SSINC	SSAGIX	OTHINC        
		TOTINCX SOLENETINCC part sbc scorp PENSION SOLEGRRECEIPTC PARTPAS PARTACT SCORPPAS SCORPACT

		/* sole prop vars */
		SOLEWAGESC SOLEEXPC
		SOLELABORCOGSC
		SOLEEMPBENC
		SOLEPENPROFITC
		SOLEDEPRC
		SOLEDEPR8829C
		SOLEDEPRCOGSC
		SOLEPAYROLLCOGSC
		SOLECONTRLABORC

 		/* Amounts excluded from total income */
		ptpexc iraexc ssexc
        
		/* Adjustment varaibles */
		IRACONT	SECADED		HEALSE		EXMOVE_AJ		SEPSIMPLE		MSA
		EDUCEXP TUITFEES
		TTADJ

		/* Deduction variables */
		TIDX 	DEDMED        TTMED         TTSLTX        SLITX         RETX          PPTX 
		OTSLTX  TTI_DED       HMIE          POINTS        IIE           TTCON         TCASH
        TNCASH  T2MISC        TTCASU        EBE           DEDMIS        FDGAMB        FDOTHD        
		FDMISC  DISIDX        TOTMIS        GR_TID       SECATX ALIP CHARITY INTPAID CHILDCARE TAXNODETAIL SALESTAX
		 RENTPAY PROPTAX keogh

		/* Tax and credit variables */
		TOT_LIAB EICX ACTCX

  		/* Retirement contribution variables */
		iracont iracontp iraconts 
		empcont empcontp empconts 
		SEcont SEcontp SEconts
		allcont allcontp allconts 

		rothcont rothcontp rothconts

		/* depreciation variables */
		E54010 E54015 E54100 E57800 E57900 E58020 E58040 E58100 E58120 E58140 E58060 E58200 E58250 E58300

		limitp      limits		iralimitp	iralimits

		earlywd


		/* Tax Calculator Variables */

		/*wagsal1 wagsal2 businc1 businc2 farminc1 farminc2 ltcg1 ltcg2 othcg1 othcg2
		div1 div2 int1 int2 teint1 teint2 pension1 pension2 rentinc1 rentinc2 ui1 ui2
		ssben1 ssben2 partscorp1 partscorp2 othinc1 othinc2 othadj1 othadj2 charity
		charcg intpaid invint stinctax proptax salestax taxnodetail medexp casualty
		movex busex miscdedlim omiscded childcare fdcred otaxpref oamtadj avglaginc 
		rentpay homeval psincome psded blind

		wagsal1act wagsal2act businc1act businc2act farminc1act farminc2act ltcg1act ltcg2act othcg1act othcg2act
		div1act div2act int1act int2act teint1act teint2act pension1act pension2act rentinc1act rentinc2act ui1act ui2act
		ssben1act ssben2act partscorp1act partscorp2act othinc1act othinc2act othadj1act othadj2act charityact
		charcgact intpaidact invintact stinctaxact proptaxact salestaxact taxnodetailact medexpact casualtyact
		movexact busexact miscdedlimact omiscdedact childcareact fdcredact otaxprefact oamtadjact avglagincact 
		rentpayact homevalact*/
		;

		/* Keep first 1K observations for testing */
		/* if _N_ <= 1000; */


   RUN;

PROC MEANS data=SUBSUB.INSOLE97_09full;
	var year othinc intst texint dbe cgagix suppgl bil fil;
RUN; 


PROC SORT;
	BY fileyr;
RUN;


PROC MEANS;
	VAR empcontp issnp was wasp ;
	BY fileyr;
RUN;

PROC MEANS data=SUBSUB.INSOLE97_09full;
	VAR empcontp issnp was wasp age wass ;
	BY fileyr;
RUN;
