/* This file estimates Phillips curves for Japan and runs dynamic forecasts based on adaptive expectations */

#delimit;
clear all;
set more off;
version 13;

/* Note: change Stata directory to the ../Stata/Code directory to run the code */
/*qui: cd "DIRECTORY IN HERE/Stata/Code"; */

/* setting paths */
qui: do paths;


/*******************************************************************************
*
* 	Gathering data: inflation, output gap, inflation expectations
*
*******************************************************************************/

/* Annual inflation */
/* Taking monthly file and averaging over months */
run japan_cpi_update;
use "$StDataPath/jpcpi", clear;
tempfile cpiann;
qui keep date cpi cpisa;
qui su date;
local maxdate = r(max);
qui gen dateq = qofd(dofm(date));
qui drop date;
qui collapse (mean) cpi cpisa, by(dateq);
duplicates drop;
tsset dateq, quarterly;
qui gen inflsa_q4 = (cpisa/L4.cpisa-1)*100;
qui gen infl_q4 = (cpi/L4.cpi-1)*100;

qui gen datey = yofd(dofq(dateq));
qui drop dateq;

qui collapse (mean) cpi cpisa (last) infl_q4 inflsa_q4, by(datey);
tsset datey;

qui gen inflsa = (cpisa/L.cpisa-1)*100;
qui gen infl = (cpi/L.cpi-1)*100;
foreach var in infl inflsa infl_q4 inflsa_q4 {;
	qui replace `var' = `var'*12/(`maxdate'-ym(2014,12)) if datey==2015;
};
qui replace infl = inflsa if datey==2015;
drop cpi cpisa;
save `cpiann';


/* use our quarterly data for Q4-Q4 forecasts */
use "$StDataPath/jpcequart", clear;

/* this part uses latest forecast for 2015 */
local old = _N;	
local new = _N + 116;
set obs `new';
qui replace survey=ym(2015,12) if survey==.;
qui replace date=ym(1988,2)+(_n-`old'-1)*3 if survey==ym(2015,12);
tsset date survey;
qui replace cpi = L6.cpi if survey==ym(2015,12);
sort survey date;

qui egen firstdate = min(-1/date*(cpi<.)), by(survey);
qui replace firstdate = -round(1/firstdate);
qui gen currcpi = 1 if date==firstdate;
qui gen dateq = qofd(dofm(date));
tsset survey dateq;
qui replace currcpi = L.currcpi*(1+cpi/400) if currcpi==.;
qui gen cpi_y = (L3.currcpi+L2.currcpi+L.currcpi+currcpi)/4;

/* Q4-Q4 forecast is forecast for inflation 4 quarters from now */
qui gen inflfc_q4 = F4.cpi;
qui gen inflfc_q4f = F7.cpi;
qui keep if date==survey | date==survey-1;
qui keep if month(dofm(survey))>=10 | month(dofm(survey))<=3;

qui replace dateq = qofd(dofm(date));
tsset dateq;

/* IV is lagged inflation forecast */
qui gen inflfc_q4_iv = L3.inflfc_q4f;

qui keep if month(dofm(survey))>=10;
qui keep date inflfc_q*;
qui gen datey = yofd(dofm(date));
qui drop date;
tempfile cefc;
save `cefc', replace;

/* Output gap */
/* Get data from IMF WEO -- uncomment next line to download new data */
*copy "https://www.imf.org/external/pubs/ft/weo/2015/01/weodata/weoreptc.aspx?sy=1980&ey=2020&sort=country&ds=.&br=1&pr1.x=47&pr1.y=5&c=158&s=NGAP_NPGDP&grp=0&a=" "$StDataPath/weogap.txt", replace;
insheet using "$StDataPath/weogap.txt", clear;
drop if _n==_N;
drop scale units country est*;
foreach v of var v* {;
	local curryear: var label `v' ;
	disp "`curryear'";
	rename `v' gap`curryear';
};
gen type = "gap";
reshape long gap, i(type) j(datey);
drop type;
tempfile imfgap;
save `imfgap';


/* merging data */
merge m:1 datey using `cpiann', nogenerate;
merge m:1 datey using `cefc', nogenerate;

/* saving data */
save "$StDataPath/phillips", replace;

/*******************************************************************************
*
* 	Running PC regressions
*
*******************************************************************************/

/* adding observations to gap */
local cutoff = 2015;
local addvar = 10;
local maxlen = `addvar'+4;
local new = _N + `addvar';
set obs `new';
qui su date;
local maxdate = r(max);
replace datey = `maxdate'+(_n-(`new'-`addvar')) if datey==.;


/* guess for gap: linearly goes to zero */
local newmax = `maxdate'+`addvar';
qui su gap if datey==`maxdate';
qui replace gap = `r(mean)'*(1-(datey-`maxdate')/`addvar') if gap==.;


tsset datey, yearly;
local lhs infl;


label variable infl "Inflation";
label variable gap "Output Gap";
label variable inflfc_q4 "Infl. FC";



foreach set in   `"_q4"' {;

	/* variables in setup */
	qui gen inflmfc`set' = `lhs'`set' - inflfc`set';
	qui gen Linflmfc`set' = L.`lhs'`set' - inflfc`set';
	qui gen Linflmfc`set'_iv = L.`lhs'`set' - inflfc`set'_iv;
	
	label variable gap "Output Gap";
	label variable inflmfc`set' " $\pi_{t}-E_{t}\pi_{t+1}$";
	label variable Linflmfc`set' " $\pi_{t-1}-E_{t}\pi_{t+1}$";
	
	
	local cc = 0;
	
	/* iterate over OLS and IV regression specifications */
	foreach rhs in `"Linflmfc`set' gap"' `"(Linflmfc`set' gap=Linflmfc`set'_iv L.gap)"'  {;
	
		/* regression imposes vertical PC */
		ivreg2 inflmfc`set' `rhs' if date<=`cutoff', kernel(bar) bw(auto) nocons;
		eststo pc`set'`cc';
		
		/* test if vertical PC is rejected by the data */
		if `"`cc'"'==`"0"' {;	
			ivreg2 inflmfc`set' `rhs' inflfc`set' if date<=`cutoff', kernel(bar) bw(auto)  nocons;	
			test inflfc`set'=0;
			estadd scalar rpval = r(p) : pc`set'`cc';
		};
		else if `"`cc'"'==`"1"' {;
			ivreg2 inflmfc`set' (Linflmfc`set' inflfc`set' gap=Linflmfc`set'_iv L.gap inflfc`set'_iv)  if date<=`cutoff', kernel(bar) bw(auto)  nocons;
			test inflfc`set'=0;
			estadd scalar rpval = r(p) : pc`set'`cc';
		};
		
		local cc = `cc'+1;

		
		/* forecasting using PC and adaptive expectations: we pick lambda and limit (mu in text) and iterate until convergence */
		foreach lambda in 0 0.1 0.2 0.3 0.4 0.45 0.5 0.55 0.6 0.7 0.8 0.9 {; 

			local ll = round(`lambda'*100);
			
			foreach limit in 2 1.75 1.5 {;
				local lim = round(`limit'*100);
		
				qui gen s`set'_`cc'_lambda`ll'_limit`lim' = `lhs' if datey<=`cutoff';
				qui gen s_fc`set'_`cc'_lambda`ll'_limit`lim' = inflfc`set' if datey<=`cutoff';
				qui replace s`set'_`cc'_lambda`ll'_limit`lim' = `limit' if datey>`cutoff';
				qui replace s_fc`set'_`cc'_lambda`ll'_limit`lim' = `limit' if datey>`cutoff';
				
				forvalues iter=1(1)100 {;
					forvalues kk=1(1)`maxlen' {;
						qui replace s`set'_`cc'_lambda`ll'_limit`lim' = _b[Linflmfc`set']*L.s`set'_`cc'_lambda`ll'_limit`lim' + _b[gap]*gap + (1-_b[Linflmfc`set'])*s_fc`set'_`cc'_lambda`ll'_limit`lim' if datey == `cutoff'+`kk';
						qui replace s_fc`set'_`cc'_lambda`ll'_limit`lim' = `lambda'*L.s`set'_`cc'_lambda`ll'_limit`lim' + (1-`lambda')*(F.s`set'_`cc'_lambda`ll'_limit`lim') if datey == `cutoff'+`kk';
					};
				};
				
				qui egen mean_s_fc`set'cc`cc'lambda`ll'limit`lim' = sum(ln(1+s`set'_`cc'_lambda`ll'_limit`lim'/100)*(datey>=`cutoff')*(datey<=`cutoff'+9)/(10)) ;
				qui egen inf20_s_fc`set'cc`cc'lambda`ll'limit`lim' = max(ln(1+s`set'_`cc'_lambda`ll'_limit`lim'/100)*(datey==2020)) ;
			
			};
		
			
			local limitlist `"`limitlist' inf20_s_fc`set'cc`cc'lambda`ll'limit mean_s_fc`set'cc`cc'lambda`ll'limit "';
			
		};
				
		local lambdalist `"`lambdalist'  inf20_s_fc`set'cc`cc'lambda mean_s_fc`set'cc`cc'lambda "';
		
	};
	
	
	/* table with regression output */
	local si 1.2;
	local stats_lab `" `"\\ \hline p-value: vertical PC"'   `" F-statistic"' `"\$ R^2$"' `"Observations"'  "';
	local stats " rpval widstat r2 N ";
	local stats_fmt " %12.2f %12.2f %12.2f %12.0fc ";
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local nregs	=	`cc';
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable: Inflation minus expected inflation $\pi_{t}-E_{t}\pi_{t+1}$}"'; 
	local groups2   `"\multicolumn{1}{c}{OLS} & \multicolumn{1}{c}{IV}"' ;	
	local midrules1 `"\cmidrule(l{.75em}){2-`=`nregs'+1'}"';
	local groups `" "`groups1'\\ `midrules1'" "& `groups2'\\" "';
	
	
	local title `"Phillips Curve estimates"';
	local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';
	local filename `"phillips-curve-estimates`=subinstr("`set'","_","-",.)'"';
	local notes `"Notes: Estimation of $ \pi_{t}-E_{t}\pi_{t+1} = \beta_{1}(\pi_{t-1}-E_{t}\pi_{t+1}) +\beta_{2}x_{t}+\epsilon_{t}$ using OLS and IV. Instruments are a lag of the output gap $ x_{t-1}$ and $ \pi_{t-1}-E_{t-1}\pi_{t+1}$. Testing for a vertical Phillips curve is the t-test on $\beta_{3}$ in the equation $ \pi_{t}-E_{t}\pi_{t+1} = \beta_{1}(\pi_{t-1}-E_{t}\pi_{t+1}) +\beta_{2}x_{t}+\beta_{3}E_{t}\pi_{t+1}+\epsilon_{t}$. Newey-West standard errors in parenthesis."';
	esttab * using "$StTabPath/`filename'.tex",  replace cells(b(star fmt(a2)) se(par fmt(a2) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) keep(`keepvars') drop(_cons, relax) `prehead' `posthead' order(`order') label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_lab')) collabels(,none) numbers nomtitles substitute(LatexMath \$ sub _ # " X " tabular* tabularx `"{1}{c}{("' `"{1}{L}{("' "\centering" "\centering \sisetup{table-format=`si'}"  < \textless [1em] "" 00000000000001 "" 0000000000001 "" 000000000001 "" 00000000001 "" 0000000001 "" 999999999999 "") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	

	qui estimates drop _all;
	
	local cclist `"`cclist'  inf20_s_fc`set'cc mean_s_fc`set'cc "';

};



keep if datey==`maxdate'+`addvar';
keep *_s_fc*;
gen test = 1;

/* shaping limit */
reshape long `limitlist', i(test) j(limit);
foreach var of varlist *_s_fc* {;
	local varname  =  subinstr("`var'","limit","",.) ;
	rename `var' `varname';
};

/* shaping lambda */
reshape long `lambdalist', i(test limit) j(lambda);
foreach var of varlist *_s_fc* {;
	local varname  =  subinstr("`var'","lambda","",.) ;
	rename `var' `varname';
};

/* shaping regression */
reshape long `cclist', i(test limit lambda) j(cc);
foreach var of varlist *_s_fc* {;
	local varname  =  subinstr("`var'","cc","",.) ;
	rename `var' `varname';
};

/* shaping fc type */
reshape long inf20_s_fc mean_s_fc, i(test limit lambda cc) j(type) string;

/* simplifying */
replace limit = limit/100;
replace lambda = lambda/100;
qui save "$DataPath/phillips_simul.dta", replace;
#delimit;
clear all;
use "$DataPath/phillips_simul.dta", clear;
keep if type=="_q4";
keep if cc==2;
drop test;


/* plotting outcomes */
replace limit = round(limit*100);
replace mean_s_fc = mean_s_fc*100;
replace inf20_s_fc = inf20_s_fc*100;
reshape wide mean_s_fc inf20_s_fc , i(cc type lambda) j(limit); 

/*xline(0.44, lcolor(black) lpattern(dash)) xline(0.52, lcolor(black) lpattern(dash))*/

graph twoway (line  mean_s_fc200 lambda, lcolor(black) lwidth(medthick)) (line  mean_s_fc175 lambda, lcolor(blue) lwidth(medthick) clpattern(dash)) (line  mean_s_fc150 lambda, lcolor(red) lwidth(medthick) clpattern(dash_dot)) , ///
   xtitle("Adaptiveness in Expectations {&lambda}") ytitle("10-year Inflation Forecast (%)") yline(1.45, lcolor(black) lpattern(shortdash) lwidth(medium))    ///
	 name(mean_s_fc200)  graphregion(color(white)) legend(cols(1) ring(0) position(2) label(1 "{&mu}=2% Long-run inflation") label(2 "{&mu}=1.75% Long-run inflation") label(3 "{&mu}=1.5% Long-run inflation"));
	graph export "$StGrphPath/phillips_fcmean.pdf", replace;
	
graph twoway (line  inf20_s_fc200 lambda, lcolor(black) lwidth(medthick)) (line  inf20_s_fc175 lambda, lcolor(blue) lwidth(medthick) clpattern(dash)) (line  inf20_s_fc150 lambda, lcolor(red) lwidth(medthick) clpattern(dash_dot)) , ///
   xtitle("Adaptiveness in Expectations {&lambda}") ytitle("Inflation in 2020 (%)")   ///
	name(inf20_s_fc200)  graphregion(color(white)) legend(cols(1) ring(0) position(2) label(1 "{&mu}=2% Long-run inflation") label(2 "{&mu}=1.75% Long-run inflation") label(3 "{&mu}=1.5% Long-run inflation"));
	graph export "$StGrphPath/phillips_fc20.pdf", replace;
